package com.tandbergtv.neptune.widgettoolkit.client.file;

import com.google.gwt.core.shared.GWT;
import com.google.gwt.dom.client.InputElement;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.DragEnterEvent;
import com.google.gwt.event.dom.client.DragEnterHandler;
import com.google.gwt.event.dom.client.DragLeaveEvent;
import com.google.gwt.event.dom.client.DragLeaveHandler;
import com.google.gwt.event.dom.client.DragOverEvent;
import com.google.gwt.event.dom.client.DragOverHandler;
import com.google.gwt.event.dom.client.DropEvent;
import com.google.gwt.event.dom.client.DropHandler;
import com.google.gwt.i18n.client.NumberFormat;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.DockPanel;
import com.google.gwt.user.client.ui.FileUpload;
import com.google.gwt.user.client.ui.FocusPanel;
import com.google.gwt.user.client.ui.SimplePanel;
import com.gwtext.client.widgets.ProgressBar;
import com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader.UploadStatusCallback;
import com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetConstants;
import com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetMessages;
import com.tandbergtv.neptune.widgettoolkit.client.util.FormatUtil;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class FileUploadPopup extends DialogBox {

    private static final String STYLE_FILE_UPLOAD_PROGRESSBAR = "file-upload-progressbar";

    private static final String STYLE_DROPZONE = "file-dropzone";

    private static final String STYLE_DROPZONE_HOVER = "file-dropzone-hover";

    private static final NeptuneWidgetConstants CONSTANTS = GWT.create(NeptuneWidgetConstants.class);

    private static final NeptuneWidgetMessages MESSAGES = GWT.create(NeptuneWidgetMessages.class);

    private FocusPanel dropZone;

    private ProgressBar progressBar;

    private FileUpload fileInput;

    private SimplePanel mainPanel;

    private ButtonWidget pauseButton;

    private ButtonWidget resumeButton;

    private ButtonWidget cancelButton;

    private ButtonWidget uploadButton;

    private HorizontalContainer buttonPanel;

    private FileUploader fileUploader;

    private File file;

    private String providerId;

    private String titleId;

    private String assetId;

    private String accessToken;

    private FileUploadCallback callback;

    public FileUploadPopup(String providerId, String accessToken, FileUploadCallback callback) {
        super(false);

        this.providerId = providerId;
        this.accessToken = accessToken;
        this.callback = callback;

        init();
    }

    public FileUploadPopup(String providerId, String accessToken) {
        this(providerId, accessToken, null);
    }

    public FileUploadPopup setTitleId(String titleId) {
        this.titleId = titleId;
        return this;
    }

    public FileUploadPopup setAssetId(String assetId) {
        this.assetId = assetId;
        return this;
    }

    private void init() {
        setModal(true);
        setGlassEnabled(true);
        setText(CONSTANTS.fileupload());

        fileInput = new FileUpload();
        fileInput.setVisible(false);
        fileInput.addChangeHandler(new ChangeHandler() {

            @Override
            public void onChange(ChangeEvent event) {
                FileInput files = fileInput.getElement().cast();
                if ((files != null) && (files.getFiles() != null) && (files.getFiles().length() > 0)) {
                    processFile((File) files.getFiles().get(0).cast());
                }
            }
        });

        dropZone = createDropZone();
        mainPanel = new SimplePanel();
        mainPanel.setWidget(dropZone);

        buttonPanel = createButtons();

        DockPanel dock = new DockPanel();
        dock.setSpacing(4);
        dock.add(mainPanel, DockPanel.NORTH);
        dock.add(buttonPanel, DockPanel.SOUTH);
        dock.setCellHorizontalAlignment(buttonPanel, DockPanel.ALIGN_CENTER);
        dock.add(fileInput, DockPanel.SOUTH);
        dock.addStyleName("file-upload-popup");

        setWidget(dock);
    }

    private HorizontalContainer createButtons() {
        HorizontalContainer hc = new HorizontalContainer();
        hc.setSpacing(5);

        ButtonWidget closeButton = new ButtonWidget(CONSTANTS.buttonClose());
        closeButton.setStyleName(StyleNames.STYLE_EB_BTN);
        closeButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                hide();
            }
        });

        uploadButton = new ButtonWidget(CONSTANTS.buttonUpload());
        uploadButton.setStyleName(StyleNames.STYLE_EB_BTN);
        uploadButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_GREEN);
        uploadButton.setVisible(false);
        uploadButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                startUpload();
            }
        });

        pauseButton = new ButtonWidget(CONSTANTS.buttonPause());
        pauseButton.setStyleName(StyleNames.STYLE_EB_BTN);
        pauseButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_BLUE);
        pauseButton.setVisible(false);
        pauseButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                pauseUpload();
            }
        });

        resumeButton = new ButtonWidget(CONSTANTS.buttonResume());
        resumeButton.setStyleName(StyleNames.STYLE_EB_BTN);
        resumeButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_BLUE);
        resumeButton.setVisible(false);
        resumeButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                resumeUpload();
            }
        });

        cancelButton = new ButtonWidget(CONSTANTS.buttonCancel());
        cancelButton.setStyleName(StyleNames.STYLE_EB_BTN);
        cancelButton.setVisible(false);
        cancelButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                cancelUpload();
            }
        });

        hc.add(uploadButton);
        hc.add(pauseButton);
        hc.add(resumeButton);
        hc.add(cancelButton);
        hc.add(closeButton);

        return hc;
    }

    private ProgressBar createProgressBar() {
        ProgressBar pb = new ProgressBar();
        pb.addStyleName(STYLE_FILE_UPLOAD_PROGRESSBAR);
        return pb;
    }

    private FocusPanel createDropZone() {
        final FocusPanel fp = new FocusPanel();
        fp.addStyleName(STYLE_DROPZONE);
        fp.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                fileInput.getElement().<InputElement> cast().click();
            }
        });

        fp.addDragLeaveHandler(new DragLeaveHandler() {

            @Override
            public void onDragLeave(DragLeaveEvent event) {
                Object target = event.getNativeEvent().getEventTarget();
                if ((target != null) && target.equals(fp.getElement())) {
                    fp.removeStyleName(STYLE_DROPZONE_HOVER);
                }
                event.preventDefault();
                event.stopPropagation();
            }
        });

        fp.addDragEnterHandler(new DragEnterHandler() {

            @Override
            public void onDragEnter(DragEnterEvent event) {
                fp.addStyleName(STYLE_DROPZONE_HOVER);
                event.preventDefault();
                event.stopPropagation();
            }
        });

        fp.addDragOverHandler(new DragOverHandler() {

            @Override
            public void onDragOver(DragOverEvent event) {
                event.preventDefault();
                event.stopPropagation();
            }
        });

        fp.addDropHandler(new DropHandler() {

            @Override
            public void onDrop(DropEvent event) {
                fp.removeStyleName(STYLE_DROPZONE_HOVER);
                event.preventDefault();
                event.stopPropagation();

                FileInput files = event.getDataTransfer().cast();
                if ((files != null) && (files.getFiles() != null) && (files.getFiles().length() > 0)) {
                    processFile((File) files.getFiles().get(0).cast());
                }
            }
        });

        return fp;
    }

    private void processFile(File file) {
        if (file == null) {
            return;
        }

        this.file = file;

        dropZone.setVisible(false);
        progressBar = createProgressBar();
        progressBar.setText(MESSAGES.fileuploadProgressSummary(file.getName(),
                FormatUtil.formatBytes(Long.parseLong(file.getSize()))));
        mainPanel.remove(dropZone);
        mainPanel.setWidget(progressBar);

        uploadButton.setVisible(true);
    }

    private void startUpload() {
        if (file == null) {
            return;
        }

        progressBar.setText(CONSTANTS.fileuploadProgressUploadStart());
        uploadButton.setVisible(false);
        pauseButton.setVisible(true);
        cancelButton.setVisible(true);

        fileUploader = new FileUploader(file, new UploadStatusCallback() {

            @Override
            public void onDone(String fileUrl) {
                pauseButton.setVisible(false);
                cancelButton.setVisible(false);
                updateProgress(1.0, CONSTANTS.fileuploadComplete());
                callback.onSuccess(fileUrl);
            }

            @Override
            public void onProgress(double progress) {
                updateProgress(progress);
            }

            @Override
            public void onPaused(double progress) {
                pauseButton.setVisible(false);
                updateProgress(progress);
                resumeButton.setVisible(true);
            }

            @Override
            public void onPause() {
                pauseButton.setVisible(false);
            }

            @Override
            public void onResume(double progress) {
                resumeButton.setVisible(false);
                updateProgress(progress);
                pauseButton.setVisible(true);
            }

            @Override
            public void onError() {
                pauseButton.setVisible(false);
                resumeButton.setVisible(false);
                cancelButton.setVisible(false);
                updateProgressError();
                callback.onFailure();
            }

            @Override
            public void onCancelled() {
                pauseButton.setVisible(false);
                resumeButton.setVisible(false);
                cancelButton.setVisible(false);
                updateProgress(1.0, CONSTANTS.fileuploadCancelled());
                callback.onCancelled();
            }
        }).setProviderId(providerId).setTitleId(titleId).setAssetId(assetId).setAccessToken(accessToken);
        fileUploader.start();
    }

    private void pauseUpload() {
        if (fileUploader == null) {
            return;
        }
        fileUploader.pause();
    }

    private void resumeUpload() {
        if (fileUploader == null) {
            return;
        }
        fileUploader.resume();
    }

    private void cancelUpload() {
        if (fileUploader == null) {
            return;
        }
        fileUploader.cancel();
    }

    private void updateProgress(double progress) {
        if (progressBar == null) {
            return;
        }
        progressBar.setValue((float) progress);
        progressBar.setText(NumberFormat.getFormat("0.#").format(progress * 100) + "%");
    }

    private void updateProgress(double progress, String text) {
        if (progressBar == null) {
            return;
        }
        progressBar.setValue((float) progress);
        progressBar.setText(text);
    }

    private void updateProgressError() {
        if (progressBar == null) {
            return;
        }
        progressBar.setValue(1.0F);
        progressBar.setText(CONSTANTS.fileuploadError());
        progressBar.addClass("progress-error");

    }

    /**
     * Interface to implement follow up action of file uploading.
     *
     * @author evan
     */
    public static interface FileUploadCallback {

        void onSuccess(String fileUrl);

        void onFailure();

        void onCancelled();
    }
}
