package com.tandbergtv.neptune.widgettoolkit.client.file;

import com.google.gwt.xhr.client.ReadyStateChangeHandler;
import com.google.gwt.xhr.client.XMLHttpRequest;


/**
 * File uploader that provides start, pause, resume and cancel feature. The uploading is based on AJAX calls.
 *
 * @author evan
 */
public class FileUploader {

    private String providerId;

    private String titleId;

    private String assetId;

    private String accessToken;

    private boolean isAlwaysSendToken;

    private File file;

    private UploadStatusCallback callback;

    private String uploadUrl = "/fileserver/upload";

    private String uploadLocation;

    private Long fileSize;

    private Long start;

    private Long end;

    /** 2M */
    private Long chunkSize = 2097152L;

    private boolean isCancel = false;

    private boolean isCancelled = false;

    private boolean isFinished = false;

    private boolean isPause = false;

    private boolean isPaused = false;

    public FileUploader(File file, UploadStatusCallback callback) {
        this.file = file;
        this.callback = callback;
    }

    public FileUploader(FileInput fileInput, UploadStatusCallback callback) {
        if (fileInput != null) {
            this.file = (File) fileInput.getFiles().get(0);
        }
        this.callback = callback;
    }

    public FileUploader setChunkSize(long chunkSize) {
        this.chunkSize = chunkSize;
        return this;
    }

    public FileUploader setUploadUrl(String uploadUrl) {
        this.uploadUrl = uploadUrl;
        return this;
    }

    public FileUploader setProviderId(String providerId) {
        this.providerId = providerId;
        return this;
    }

    public FileUploader setTitleId(String titleId) {
        this.titleId = titleId;
        return this;
    }

    public FileUploader setAssetId(String assetId) {
        this.assetId = assetId;
        return this;
    }

    public FileUploader setAccessToken(String accessToken) {
        this.accessToken = accessToken;
        return this;
    }

    /**
     * If set to true, the file uploader will always send the access token in the request even for each file trunk.
     * Otherwise only the start and cancel request will have the token.
     *
     * @param isAlwaysSendToken
     * @return
     */
    public FileUploader setAlwaysSendToken(boolean isAlwaysSendToken) {
        this.isAlwaysSendToken = isAlwaysSendToken;
        return this;
    }

    /**
     * Start the file uploader.
     */
    public void start() {
        isCancelled = false;
        isFinished = false;

        XMLHttpRequest xhr = XMLHttpRequest.create();
        String params = (uploadUrl.indexOf('?') > -1 ? "&" : "?") + "providerId=" + providerId + "&titleId=" + titleId
                + "&assetId=" + assetId;
        xhr.open("POST", uploadUrl + params);
        xhr.setRequestHeader("Authorization", "Bearer " + accessToken);
        xhr.setRequestHeader("X-File-Name", file.getName());
        xhr.setRequestHeader("X-File-Size", file.getSize());
        xhr.setRequestHeader("X-File-Type", file.getType());

        xhr.setOnReadyStateChange(new ReadyStateChangeHandler() {

            @Override
            public void onReadyStateChange(XMLHttpRequest xhr) {
                if (xhr.getReadyState() == 4) {
                    if (xhr.getStatus() == 200) {
                        uploadLocation = xhr.getResponseHeader("Location");
                        sendFirstChunk();
                    } else {
                        onError();
                    }
                }
            }
        });
        addErrorHandler(xhr);
        xhr.send();
    }

    /**
     * Cancel the file uploading.
     */
    public void cancel() {
        if (isFinished || isCancelled) {
            return;
        }
        isCancel = true;
        if (isPaused) {
            sendCancelRequest();
        }
    }

    /**
     * Pause the file uploading.
     */
    public void pause() {
        if (isFinished || isCancelled || isPause) {
            return;
        }
        isPause = true;
        if (callback != null) {
            callback.onPause();
        }
    }

    /**
     * Resume the paused file upload process.
     */
    public void resume() {
        if (isFinished || isCancelled) {
            return;
        }
        if (isPaused) {
            // Resume
            if (callback != null) {
                callback.onResume(((double) end) / fileSize);
            }
            isPause = false;
            isPaused = false;
            sendNextChunk();
        }
    }

    /**
     * Send a cancel HTTP request to the file server.
     */
    private void sendCancelRequest() {
        XMLHttpRequest xhr = XMLHttpRequest.create();
        xhr.open("DELETE", uploadLocation);
        xhr.setRequestHeader("Authorization", "Bearer " + accessToken);
        xhr.setOnReadyStateChange(new ReadyStateChangeHandler() {

            @Override
            public void onReadyStateChange(XMLHttpRequest xhr) {
                if (xhr.getStatus() == 200) {
                    onCancel();
                } else {
                    onError();
                }
            }
        });
        xhr.send();
    }

    private void sendFirstChunk() {
        fileSize = Long.parseLong(file.getSize());
        start = 0L;
        end = chunkSize;
        if (end >= fileSize) {
            end = fileSize;
        }
        sendFileChunk();
    }

    private void sendNextChunk() {
        start += chunkSize;
        end += chunkSize;
        if (end >= fileSize) {
            end = fileSize;
        }
        sendFileChunk();
    }

    private void onTransferComplete(String response) {
        if (isCancel) {
            sendCancelRequest();
            return;
        }

        if (isPause) {
            isPaused = true;
            if (callback != null) {
                callback.onPaused(((double) end) / fileSize);
            }
            return;
        }

        if (end >= fileSize) {
            // Finished
            isFinished = true;
            if (callback != null) {
                callback.onDone(response);
            }
        } else {
            if (callback != null) {
                callback.onProgress(((double) end) / fileSize);
            }
            sendNextChunk();
        }
    }

    private void onError() {
        callback.onError();
    }

    private void onCancel() {
        isCancelled = true;
        callback.onCancelled();
    }

    private native void sendFileChunk() /*-{
        var xhr = @com.google.gwt.xhr.client.XMLHttpRequest::create()();
        xhr.open('PUT', this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::uploadLocation, true);

        var _this = this;
        xhr.onload = function() {
            if (xhr.status == 200) {
                _this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onTransferComplete(Ljava/lang/String;)(xhr.responseText);
            } else {
                _this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onError()();
            }
        };
        this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::addErrorHandler(Lcom/google/gwt/xhr/client/XMLHttpRequest;)(xhr);

        if (this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::isAlwaysSendToken) {
            xhr.setRequestHeader("Authorization", "Bearer " + this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::accessToken);
        }
        xhr.setRequestHeader("Content-Type", "application/octet-stream");
        xhr.setRequestHeader("Content-Range", this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::start + "-" + (this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::end - 1));

        var chunk = this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::file.slice(this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::start, this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::end);
        xhr.send(chunk);
    }-*/;

    private native void addErrorHandler(XMLHttpRequest xhr) /*-{
        var _this = this;
        xhr.onerror = function() {
            _this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onError()();
        };
    }-*/;

    public static interface UploadStatusCallback {

        /**
         * File uploading completed.
         *
         * @param fileUrl
         */
        void onDone(String fileUrl);

        /**
         * File uploading progress updated.
         *
         * @param progress
         */
        void onProgress(double progress);

        /**
         * File uploading paused.
         *
         * @param progress
         */
        void onPaused(double progress);

        /**
         * File uploading pause request sent.
         */
        void onPause();

        /**
         * File uploading resumed.
         *
         * @param progress
         */
        void onResume(double progress);

        /**
         * File uploading error.
         */
        void onError();

        /**
         * File uploading is cancelled.
         */
        void onCancelled();
    }
}
