/**
 * This file is modified from GWT Portlets Demo.java.
 */
/*
 * GWT Portlets Framework (http://code.google.com/p/gwtportlets/)
 * Copyright 2009 Business Systems Group (Africa)
 *
 * GWT Portlets is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GWT Portlets is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with GWT Portlets.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.tandbergtv.neptune.ui.portalpage.client;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.portalpage.client.portaluserpref.UiPortalUserPreference;
import com.tandbergtv.neptune.ui.portalpage.client.ui.ColumnSizingLayoutPanel;
import com.tandbergtv.neptune.ui.portalpage.client.ui.Gadget;
import com.tandbergtv.neptune.ui.portalpage.client.ui.GadgetCloseEvent;
import com.tandbergtv.neptune.ui.portalpage.client.ui.GadgetCloseHandler;
import com.tandbergtv.neptune.ui.portalpage.client.ui.HtmlGadget;
import com.tandbergtv.neptune.ui.portalpage.client.ui.NeptunePortletPage;
import com.tandbergtv.neptune.ui.portalpage.client.ui.WidgetAreaPanel;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;

import org.gwtportlets.portlet.client.WidgetFactory;
import org.gwtportlets.portlet.client.WidgetRefreshHook;
import org.gwtportlets.portlet.client.event.BroadcastManager;
import org.gwtportlets.portlet.client.event.PageChangeEvent;
import org.gwtportlets.portlet.client.layout.Container;
import org.gwtportlets.portlet.client.layout.RowLayout;
import org.gwtportlets.portlet.client.layout.RowLayout.Constraints;

/**
 * Fetches the root page layout and uses this to create the GUI.
 */
public class PortalPage extends Composite implements GadgetCloseHandler {
	
	private final String UPDATE_BY_ID = "PORTLET_ID=";
	
	private final SimpleContainer mainContainer;
    // The ClientAreaPanel always fills the whole of the browsers client area.
    // It forms the topmost container for the application and adds itself to
    // the normal GWT RootPanel in its constructor. It uses a LayoutPanel
    // internally
    private WidgetAreaPanel widgetAreaPanel;
    // The pageEditor is responsible for editing and saving pages
    private ArrangementEditor pageEditor;

	private ArrayList<Gadget> gadgetList = new ArrayList<Gadget>();
	
	private List<PortletConfig> availablePortlets;
	
	public PortalPage(List<PortletConfig> availablePortlets) {
		this.availablePortlets = availablePortlets;
		mainContainer = new SimpleContainer();
		widgetAreaPanel = new WidgetAreaPanel();
		pageEditor = new ArrangementEditor();
		initWidget(mainContainer);

		// The framework calls the WidgetRefreshHandler instance when it
		// needs to refresh a Portlet or Widget with new data from the server.
		// This handler just calls our refresh service method.
		WidgetRefreshHook.App.set(new WidgetRefreshHook() {
			@SuppressWarnings("rawtypes")
			public void refresh(Widget w, WidgetFactory wf, AsyncCallback<WidgetFactory> cb) {
				PortalPageService.App.get().refresh(History.getToken(), wf, cb);
			}

			public void onRefreshCallFailure(Widget w, Throwable caught) {
				String msg = null;
				if (caught instanceof IllegalArgumentException) {
					msg = caught.getMessage();
				}
				if (msg == null || msg.length() == 0) {
					msg = "Refresh failed: " + caught;
				}
				Window.alert(msg);
			}
		});
		addEventListener(this);
		pageEditor.setPage(this);
		fetchRootPage();
    }

    private final native void addEventListener(PortalPage instance) /*-{
		function updatePortletParams(event) {
		    instance.@com.tandbergtv.neptune.ui.portalpage.client.PortalPage::changeParams(Ljava/lang/String;)(event.data);
		}
		if ($wnd.attachEvent) {
			$wnd.attachEvent("onmessage", updatePortletParams);
		} else {
			$wnd.addEventListener("message", updatePortletParams, false);
		}
	}-*/;

	/**
	 * Create the com.tandbergtv.neptune.ui.portalpage application GUI by fetching the 'root page' from the server. The
	 * root page contains a {@link org.gwtportlets.portlet.client.ui.PagePortlet} which forms a 'content area' to
	 * display the selected page.
	 */
	private void fetchRootPage() {
		PortalPageService.App.get().getRootPage(History.getToken(), new NeptuneAsyncCallback<PageConfig>() {
			public void onNeptuneFailure(Throwable caught) {
				Window.alert("Oops " + caught);
			}

			public void onNeptuneSuccess(PageConfig p) {
				widgetAreaPanel.clear();
				widgetAreaPanel.add((new NeptunePortletPage.Factory()).createWidget(),
						new RowLayout.Constraints(Constraints.CSS));
				widgetAreaPanel.layout();
				onPageChange(p);
			}
		});
	}

	/**
	 * Notify all portlets that the page has changed. The {@link org.gwtportlets.portlet.client.ui.PagePortlet} uses
	 * this notification to display the new page.
	 */
	private void onPageChange(final PageConfig p) {
		final boolean notFound = p.preferences == null || p.preferences.size() == 0;
		if (notFound) {
			addAllPortlets(p);
		} else {
			loadPreferences(p);
		}
	}

	private void addAllPortlets(PageConfig p) {
		ColumnSizingLayoutPanel columns = new ColumnSizingLayoutPanel();
		int numGadgets = 0;
		for (PortletConfig next : availablePortlets) {
			Gadget gadget = Gadget.createGadget(next);
			// add portlet to next available slot (fill by col1,row1; col2,row1; col1,row2; col2,row2; etc)
			int columnIndex = (numGadgets % ColumnSizingLayoutPanel.DEFAULT_NUM_COLUMNS);
			columns.addWidgetToColumn(columnIndex, gadget, gadget.getDefaultHeight());
			numGadgets++;
		}
		p.widgetFactory = new ColumnSizingLayoutPanel.Factory(columns);

		// The page change event knows how to edit the current page
		PageChangeEvent pce = new PageChangeEvent(this) {
			private static final long serialVersionUID = 1L;

			public void editPage(Container container) {
				pageEditor.beginEditing(getPageName(), container);
			}
		};
		pce.setPageName(p.pageName);
		pce.setEditable(p.canEditPage);
		pce.setWidgetFactory(p.widgetFactory);

		// Send the event to every AppEventListener in the container tree.
		// The PagePortlet uses this event to change the widget tree in the
		// 'content area' of the application and to display the gear icon
		// for editable pages
		BroadcastManager.get().broadcast(pce);

		saveConfig();
		updateGadgetList();
	}

	private void loadPreferences(PageConfig p) {
		ColumnSizingLayoutPanel columns = new ColumnSizingLayoutPanel();
		for (UiPortalUserPreference next : p.preferences) {
			if (next.isColumnPreference()) {
				columns.setColumnWidth(next.getColumnIndex(), next.getWidth());
			} else {
				for (PortletConfig config : availablePortlets) {
					if (config.getPortletId().equals(next.getKey())
							&& config.getComponent().equals(next.getComponent())) {
						Gadget gadget = Gadget.createGadget(config);
						if (next.getColumnIndex() >= 0) {
							columns.addWidgetToColumn(next.getColumnIndex(), gadget, next.getHeight());
						}
						break;
					}
				}
			}
		}
		p.widgetFactory = new ColumnSizingLayoutPanel.Factory(columns);

		// The page change event knows how to edit the current page
		PageChangeEvent pce = new PageChangeEvent(this) {
			private static final long serialVersionUID = 1L;

			public void editPage(Container container) {
				pageEditor.beginEditing(getPageName(), container);
			}
		};
		pce.setPageName(p.pageName);
		pce.setEditable(p.canEditPage);
		pce.setWidgetFactory(p.widgetFactory);

		// Send the event to every AppEventListener in the container tree.
		// The PagePortlet uses this event to change the widget tree in the
		// 'content area' of the application and to display the gear icon
		// for editable pages
		BroadcastManager.get().broadcast(pce);

		updateGadgetList();
	}

	public void updateGadgetList() {
		ArrayList<Gadget> updatedList = new ArrayList<Gadget>();
		PageUtils.getGadgets(RootPanel.get(), updatedList, null, null);
		for (int i = 0; i < updatedList.size(); i++) {
			Gadget next = updatedList.get(i);
			if (!gadgetList.contains(next)) {
				next.addGadgetCloseHandler(this);
			}
		}
		gadgetList = updatedList;
	}

	private HtmlGadget getGadget(String url) {
		HtmlGadget htmlGadget = null;
		int numGadgets = gadgetList.size();
		for (int index = 0; index < numGadgets; index++) {
			Gadget gadget = gadgetList.get(index);
			if (gadget.getParent() == null) {
				gadgetList.remove(index);
			} else if (gadget instanceof HtmlGadget) {
				htmlGadget = (HtmlGadget) gadget;
				if (htmlGadget.matchesUrl(url)) {
					return htmlGadget;
				}
			}
		}
		htmlGadget = (HtmlGadget) (PageUtils.getGadgets(RootPanel.get(), gadgetList, url, null));
		htmlGadget.addGadgetCloseHandler(this);
		return htmlGadget;
	}

	private Gadget getGadgetById(String id) {
		Gadget gadget;
		int numGadgets = gadgetList.size();
		for (int index = 0; index < numGadgets; index++) {
			gadget = gadgetList.get(index);
			if (gadget.getParent() == null) {
				gadgetList.remove(index);
			} else if (gadget.getPortletId().equals(id)) {
				return gadget;
			}
		}
		gadget = PageUtils.getGadgets(RootPanel.get(), gadgetList, null, id);
		gadget.addGadgetCloseHandler(this);
		return gadget;
	}

	private void changeParams(String params) {
		// format is either UPDATE_BY_ID[id][&amp;name=value]+ or [url][&amp;name=value]+
		final String heightToken = "h";
		final String titleToken = "title";
		final String DELIM = "&amp;";
		int height = -1;
		String title = null;

		String[] paramTokens = params.split(DELIM);
		if (paramTokens.length < 2) {
			return;
		}
		for (int i = 1; i < paramTokens.length; i++) {
			String[] nameValue = paramTokens[i].split("=", 2);
			if (nameValue.length == 2) {
				if (nameValue[0].equals(heightToken)) {
					height = Integer.parseInt(nameValue[1]);
				} else if (nameValue[0].equals(titleToken)) {
					title = nameValue[1];
				}
			}
		}

		Gadget gadget = null;
		if (paramTokens[0].startsWith(UPDATE_BY_ID)) {
			String id = paramTokens[0].substring(UPDATE_BY_ID.length());
			gadget = getGadgetById(id);
		} else {
			String url = paramTokens[0];
			gadget = getGadget(url);
		}
		if (null != gadget) {
			if (title != null) {
				gadget.setTitleText(title);
				gadget.update();
			}
			if (height >= 0) {
				gadget.setHeight(height);
			}
		}
	}

	@Override
	public void onGadgetClosed(GadgetCloseEvent event) {
		if (event.getSource() instanceof Gadget) {
			Gadget gadget = (Gadget) event.getSource();
			deleteGadgetFromPreferences(gadget);
		}
	}

	public void saveConfig() {
		List<UiPortalUserPreference> preferences = new ArrayList<UiPortalUserPreference>();
		PageUtils.getPortalUserPreferences(RootPanel.get(), preferences);
		PortalPageService.App.get().savePortletConfig(preferences, new NeptuneAsyncCallback<Object>() {
			public void onNeptuneFailure(Throwable caught) {
				Window.alert("Oops " + caught);
			}

			public void onNeptuneSuccess(Object result) {
			}
		});
	}
	
	public void deleteGadgetFromPreferences(Gadget gadget) {
		PortalPageService.App.get().deletePreference(gadget.getComponent(), gadget.getPortletId(),
				new NeptuneAsyncCallback<Object>() {
					public void onNeptuneFailure(Throwable caught) {
						Window.alert("Oops " + caught);
					}

					public void onNeptuneSuccess(Object result) {
					}
				});
	}

	public void setAvailablePortlets(List<PortletConfig> availablePortlets) {
		this.availablePortlets = availablePortlets;
	}

	public List<PortletConfig> getAvailablePortlets() {
		return availablePortlets;
	}

	public void showWidget() {
		if (!widgetAreaPanel.isAttached()) {
			RootPanel.get().add(widgetAreaPanel);
		}
		fetchRootPage();
	}
	
	public void clearWidget() {
		widgetAreaPanel.clear();
		if (widgetAreaPanel.isAttached()) {
			RootPanel.get().remove(widgetAreaPanel);
		}
		pageEditor.endEditing(false);
		NeptunePortletPage.removeEditButton();
	}
}
