package com.tandbergtv.cms.portal.util.transaction;

import javax.naming.InitialContext;
import javax.transaction.TransactionManager;

import org.apache.log4j.Logger;
import org.aspectj.lang.annotation.SuppressAjWarnings;

/**
 * This aspect enforces the Transactional annotation. Any java method can be
 * marked as Transactional and this aspect will ensure that the transaction is
 * started iff it's not already running.
 * 
 * @author trybak
 * 
 */
public aspect TransactionEnforcer {
	private final static Logger LOGGER = Logger.getLogger(TransactionEnforcer.class);
	pointcut arbitraryJavaMethod(Transactional transactional) : execution(@Transactional * *.*(..)) && @annotation(transactional);

	@SuppressAjWarnings("adviceDidNotMatch")
	Object around(Transactional transactional) : arbitraryJavaMethod(transactional) {
		TransactionManager transactionManager = null;
		boolean isTransactionOwner = false;
		try {
			transactionManager = (TransactionManager) new InitialContext()
					.lookup("java:/TransactionManager");
			isTransactionOwner = transactionManager.getTransaction() == null;
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e.getLocalizedMessage(), e);
		}
		
		if (!isTransactionOwner) {
			return proceed(transactional);
		} else {
			try {
				transactionManager.begin();
				Object object = proceed(transactional);
				transactionManager.commit();
				return object;
			} catch (Exception e) {
				LOGGER.error("Transaction Rollback", e);
				try {
					transactionManager.rollback();
				} catch(Exception e1) {
					LOGGER.error("Exception rolling back the transaction", e1);
				}
				
				/* Do not wrap the exception into another runtime exception unless required */
				if (e instanceof RuntimeException) {
					throw (RuntimeException) e;
				}

				throw new TransactionalException(e);
			}
		}
	}
}
