/*
 * Created on Sep 10, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_MIDDLE;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.resources.client.ImageResource;
import com.google.gwt.user.client.Window;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageNavigationEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;

/**
 * The widget that contains all paging information used by the page feature handler for the table
 * 
 * @author Vijay Silva
 */
public class PageFeatureWidget extends PageWidget {

	/* 'Page Navigation' widgets */
	
	private LabelWidget pageCountWidget;
	private ImageWidget leftLeftWidget;
	private ImageWidget leftWidget;
	private ImageWidget rightWidget;
	private ImageWidget rightRightWidget;

	/* Build the 'Page Navigation' panel */
	@Override
	protected HorizontalContainer buildPageNavigationPanel() {
		HorizontalContainer panel = new HorizontalContainer();
		panel.setVerticalAlignment(ALIGN_MIDDLE);
		panel.setSpacing(TableConstants.NAV_SPACING);

		leftLeftWidget = new ImageWidget();
		leftLeftWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		leftLeftWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToFirstPageClick(event);
			}
		});
		leftLeftWidget.setResource(imageBundle.firstPageDisabled());
		panel.add(leftLeftWidget);

		leftWidget = new ImageWidget();
		leftWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		leftWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToPreviousPageClick(event);
			}
		});
		leftWidget.setResource(imageBundle.previousPageDisabled());
		panel.add(leftWidget);

		pageCountWidget = new LabelWidget("");
		pageCountWidget.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
		panel.add(pageCountWidget);

		rightWidget = new ImageWidget();
		rightWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		rightWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToNextPageClick(event);
			}
		});
		rightWidget.setResource(imageBundle.nextPageDisabled());
		panel.add(rightWidget);

		rightRightWidget = new ImageWidget();
		rightRightWidget.setStyleName(TableConstants.STYLE_NAV_BUTTON);
		rightRightWidget.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGoToLastPageClick(event);
			}
		});
		rightRightWidget.setResource(imageBundle.lastPageDisabled());
		panel.add(rightRightWidget);

		return panel;
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/* The 'first page' button click */
	private void handleGoToFirstPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at first page, ignore */
		if (getInput().getPageNumber() == 1)
			return;

		int pageNumber = 1;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageNumber, getInput()));
	}

	/* The 'previous page' button click */
	private void handleGoToPreviousPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at first page, ignore */
		if (getInput().getPageNumber() == 1)
			return;

		int pageCount = getTotalPages();
		int pageNumber = getInput().getPageNumber() - 1;
		if (pageNumber <= 0)
			pageNumber = 1;
		else if (pageCount > 0 && pageNumber > pageCount)
			pageNumber = pageCount;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageNumber, getInput()));
	}

	/* The 'next page' button click */
	private void handleGoToNextPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at last page, ignore */
		int pageCount = this.getTotalPages();
		if (pageCount > 0 && getInput().getPageNumber() == pageCount)
			return;

		int pageNumber = getInput().getPageNumber() + 1;
		if (pageCount > 0 && pageNumber > pageCount)
			pageNumber = pageCount;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageNumber, getInput()));
	}

	/* The 'last page' button click */
	private void handleGoToLastPageClick(ClickEvent event) {
		/* If there is no input, no event is fired */
		if (this.getInput() == null)
			return;

		/* If at last page, ignore */
		int pageCount = this.getTotalPages();
		if (pageCount > 0 && getInput().getPageNumber() == pageCount)
			return;

		if (pageCount <= 0)
			pageCount = 1;

		/* Fire event indicating that the page navigation required */
		fireEvent(new PageNavigationEvent(pageCount, getInput()));
	}

	/* The key press event for the page number widget */
	@Override
	protected void handlePageNumberKeyPress(KeyPressEvent event) {
		/* If there is no input, no event is fired */
		if (getInput() == null)
			return;

		/* Determine if a complete page number has been entered */
		if (event.getNativeEvent().getKeyCode() == KeyCodes.KEY_ENTER) {
			int pageNumber = 0;
			try {
				pageNumber = Integer.parseInt(pageNumberWidget.getText());
			} catch (NumberFormatException e) {
				Window.alert(constants.validPageNumberMessage());
				return;
			}

			if (pageNumber <= 0) {
				Window.alert(constants.validPageNumberMessage());
				return;
			}

			// find total pages
			int totalPages = getTotalPages();

			// bug fix #3836: if the entered page number is later than last page, goto last page
			if (totalPages > 0 && pageNumber > totalPages) {
				pageNumber = totalPages;
				setPageNumberText(pageNumber);
			}

			/* Fire event indicating that the page navigation required */
			fireEvent(new PageNavigationEvent(pageNumber, getInput()));
		}
	}

	/*
	 * Update the panel containing the page navigation options based on the current feature state
	 */
	@Override
	protected void updatePageNavigationPanel() {
		/* Update the page count */
		int totalPages = this.getTotalPages();
		int currentPage = (getInput() != null) ? getInput().getPageNumber() : DEFAULT_PAGE_NUMBER;
		String pageCount = (totalPages <= 0) ? "-" : Integer.toString(totalPages);
		pageCountWidget.setText(messages.pageOfTotal(currentPage, pageCount));

		/* Update the page navigation images */
		ImageResource image = null;
		boolean isLeftEnabled = (currentPage > 1);
		image = isLeftEnabled ? imageBundle.firstPageEnabled() : imageBundle.firstPageDisabled();
		leftLeftWidget.setResource(image);
		image = isLeftEnabled ? imageBundle.previousPageEnabled() : imageBundle
		        .previousPageDisabled();
		leftWidget.setResource(image);

		boolean isRightEnabled = (totalPages <= 0 || currentPage < totalPages);
		image = isRightEnabled ? imageBundle.lastPageEnabled() : imageBundle.lastPageDisabled();
		rightRightWidget.setResource(image);
		image = isRightEnabled ? imageBundle.nextPageEnabled() : imageBundle.nextPageDisabled();
		rightWidget.setResource(image);
	}

	/*
	 * Get the total number of pages or -1 if the record count is not initialized.
	 */
	private int getTotalPages() {
		/* No input present, return -1 */
		if (getInput() == null)
			return -1;

		/* If no records present, return a page count of -1 */
		int recordCount = getInput().getRecordCount();
		if (recordCount < 0)
			return -1;

		int recordsPerPage = getInput().getPageSize();
		int totalPages = recordCount / recordsPerPage;
		if (recordCount % recordsPerPage > 0)
			totalPages++;

		if (totalPages <= 0)
			totalPages = 1;

		return totalPages;
	}
}
