/**
 * 
 */
package com.ericsson.cms.contractmgmt.service;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.ejb.EJB;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.ejb3.annotation.Depends;
import org.jboss.ejb3.annotation.Service;

import com.ericsson.cms.scheduler.IScheduler;
import com.ericsson.cms.scheduler.entities.Job;
import com.ericsson.cms.scheduler.entities.ScheduleRule;
import com.ericsson.cms.scheduler.entities.ScheduleRuleParameter;
import com.ericsson.cms.scheduler.entities.ScheduleRuleType;
import com.ericsson.cms.scheduler.quartz.IQuartzScheduler;
import com.ericsson.mail.ICMEmailNotificationManager;
import com.ericsson.service.ContractManagerValidationException;
import com.ericsson.service.IContractManagerExpiryTimer;
import com.tandbergtv.neptune.licensing.ILicenseManager;
import com.tandbergtv.neptune.licensing.InvalidLicenseException;

/**
 * This service will ensure that during the upgrade if there any EJBTimerService
 * jobs active, then those jobs will be cancelled, and a default Scheduler
 * job will be created. After which the user will have the option to update /
 * delete the Scheduler job from the UI.
 * 
 * @author amit.r.seth
 *
 */
@Service(objectName = "ContractManager:service=com.ericsson.cms.contractmgmt.service.InitializeOldScheduleService")
@Depends(IQuartzScheduler.OBJECT_NAME)
public class InitializeOldScheduleService implements IInitializeOldScheduleService {
	
	private static final Log log = LogFactory.getLog(InitializeOldScheduleService.class);

	private static final String ASSOCIATED_ENTITY_TYPE_CMEMAILNOTIFICATION = "CMEmailNotification_SCHEDULER";

	@EJB
	private IContractManagerExpiryTimer timer;

	// FIXME: Make sure to change the jar file name when the neptune jar is updated...
	@Depends({ "jboss.j2ee:ear=cms.ear,jar=neptune-1.0.0.288.jar,name=AuthorizationService,service=EJB3" })
    @EJB
	private IScheduler scheduler;
	
	@EJB
	private ICMEmailNotificationManager oldNotificationManager;

	/* (non-Javadoc)
	 * @see com.ericsson.cms.contractmgmt.service.IInitializeOldScheduleService#start()
	 */
	@Override
	public void start() throws Exception {
		// Make sure that the previous schedule from the EJBTimerService
		// is loaded at startup and the EJB Timer jobs are cancelled...
		log.info("In initSchedule()");
		if (oldNotificationManager.deleteRegisteredEJBTimer()) {
			try {
				if (isLicensed()) {
					List<Job> jobs = scheduler
							.getJobsByAssociatedEntityType(ASSOCIATED_ENTITY_TYPE_CMEMAILNOTIFICATION);

					if (jobs.size() <= 0) {
						createDefaultJob();
					}
				}
			} catch (InvalidLicenseException e) {
				log.error("Error in obtaining the license for Contract Manager " + e.getLocalizedMessage());
			}
		}
	}

	/**
	 * 
	 */
	private void createDefaultJob() {
		Calendar currentCal = Calendar.getInstance();

		Calendar tomorrow = Calendar.getInstance();
        tomorrow.setLenient(true);
        tomorrow.clear();
        tomorrow.set(currentCal.get(Calendar.YEAR), currentCal.get(Calendar.MONTH), currentCal.get(Calendar.DATE)+1, 0, 0, 0);
        
        ScheduleRule scheduleRule = new ScheduleRule();
		scheduleRule.setStartDate(tomorrow.getTime());
		scheduleRule.setType(ScheduleRuleType.EVERY_N_DAYS);

		List<ScheduleRuleParameter> params = new ArrayList<ScheduleRuleParameter>();
		ScheduleRuleParameter everyDay = new ScheduleRuleParameter();
		everyDay.setOrder(1);
		everyDay.setValue("1");
		params.add(everyDay);

		ScheduleRuleParameter atTime = new ScheduleRuleParameter();
		atTime.setOrder(2);
		atTime.setValue("12:05 am");
		params.add(atTime);

		scheduleRule.setParams(params);

		try {
			log.info("Calling createUpdateTimer()");
			timer.createUpdateTimer(scheduleRule);
			log.info("Finished Calling createUpdateTimer()");
		} catch (Exception e) {
            ContractManagerValidationException cmve = findContractManagerValidationException(e);
            if ( cmve != null ) {
                log.error("Could not create the Scheduler job: " + cmve.getValidationMessages());
            } else {
                log.error("Could not create the Scheduler job: " + e.getLocalizedMessage());
            }
            e.printStackTrace();
		}
	}

    private ContractManagerValidationException findContractManagerValidationException( Exception e ) {
        ContractManagerValidationException result = null;
        if ( e instanceof ContractManagerValidationException ) {
            result = (ContractManagerValidationException) e;
        } else {
            Throwable current = e;
            while ( current.getCause() != null ) {
                current = current.getCause();
                if ( current instanceof ContractManagerValidationException ) {
                    result = (ContractManagerValidationException) current;
                    break;
                }
            }
        }
        
        return result;
    }

    private static final String CM_LICENSE = "ContractMgrLicense"; 

    @EJB
    private ILicenseManager neptuneLicenseManager;
    /**
     * @return true is the product is licensed.
     * @throws InvalidLicenseException
     */
    private boolean isLicensed() throws InvalidLicenseException {
        String value = neptuneLicenseManager.getProperty(CM_LICENSE);
        if (value == null || value.trim().isEmpty()) 
            return false;
        
        return Boolean.parseBoolean(value);
    }
}
