/*
 * Copyright www.gdevelop.com.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.gdevelop.gwt.syncrpc;


import com.google.gwt.user.client.rpc.RemoteServiceRelativePath;

import java.io.IOException;

import java.lang.reflect.Proxy;

import java.net.CookieManager;
import java.net.CookiePolicy;

import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;


/**
 * Sync Proxy for GWT RemoteService
 * Usage:
 * MyServiceInterface myService = newProxyInstance(MyServiceInterface.class,
 *    "http://localhost:8888/myapp/", "myServiceServlet", policyName);
 *  where policyName is the file name (with gwt.rpc extenstion) generated
 *  by GWT RPC backend
 *
 * Or
 * MyServiceInterface myService = newProxyInstance(MyServiceInterface.class,
 *    "http://localhost:8888/myapp/", "myServiceServlet");
 * In this case, the SyncProxy search for the appropriate policyName file in
 * the system classpath
 *
 * If not specified, SyncProxy uses a <em>default</em> {@link CookieManager}
 * to manage client-server communication session
 *
 * To perform multi-session:
 * CookieManager cookieManager = LoginUtils.loginAppEngine(...);
 * MyServiceInterface myService = newProxyInstance(MyServiceInterface.class,
 *    "http://localhost:8888/myapp/", "myServiceServlet", cookieManager);
 * @see com.gdevelop.gwt.syncrpc.test.ProfileServiceTest example
 */
public class SyncProxy {
  /**
   * Map from ServiceInterface class name to Serialization Policy name
   */
  private static final Map<String, String> POLICY_MAP = RpcPolicyFinder.searchPolicyFileInClassPath();
  
  private static final CookieManager DEFAULT_COOKIE_MANAGER 
    = new CookieManager(null, CookiePolicy.ACCEPT_ALL);
  
  /**
  * Create a new Proxy for the specified <code>serviceIntf</code>
  * @param serviceIntf The remote service interface
  * @param moduleBaseURL Base URL
  * @param remoteServiceRelativePath The remote service servlet relative path
  * @param policyName Policy name (*.gwt.rpc) generated by GWT RPC backend
  * @param cookieManager Used to perform session management such as login. 
  * @param waitForInvocation Used for Async RemoteService.
  * @return A new proxy object which implements the service interface serviceIntf
  */
  @SuppressWarnings("unchecked")
  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                        String remoteServiceRelativePath, 
                                        String policyName, 
                                        CookieManager cookieManager, 
                                        boolean waitForInvocation){
    if(cookieManager == null){
      cookieManager = DEFAULT_COOKIE_MANAGER;
    }
    
    if (policyName == null){
      try {
        POLICY_MAP.putAll(RpcPolicyFinder.fetchSerializationPolicyName(moduleBaseURL));
        policyName = POLICY_MAP.get(serviceIntf.getName());
      } catch (IOException e) {
        e.printStackTrace();
      }
    }

    return Proxy.newProxyInstance(SyncProxy.class.getClassLoader(), 
                new Class[]{serviceIntf}, 
                new RemoteServiceInvocationHandler(moduleBaseURL, 
                                                   remoteServiceRelativePath, 
                                                   policyName, 
                                                   cookieManager,
                                                   waitForInvocation));
  }

  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                        String remoteServiceRelativePath, 
                                        String policyName, 
                                        CookieManager cookieManager){
    return newProxyInstance(serviceIntf, moduleBaseURL, 
                            remoteServiceRelativePath, policyName, 
                            cookieManager, false);
  }
  
  /**
   * Create a new Proxy for the specified service interface <code>serviceIntf</code>
   * 
   * @param serviceIntf The remote service interface
   * @param moduleBaseURL Base URL
   * @param remoteServiceRelativePath The remote service servlet relative path
   * @param policyName Policy name (*.gwt.rpc) generated by GWT RPC backend
   * @return A new proxy object which implements the service interface serviceIntf
   */
  @SuppressWarnings("unchecked")
  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                       String remoteServiceRelativePath, 
                                       String policyName){
    return newProxyInstance(serviceIntf, moduleBaseURL, remoteServiceRelativePath, 
                            policyName, DEFAULT_COOKIE_MANAGER);
  }

  /**
  * Create a new Proxy for the specified service interface <code>serviceIntf</code>
  * 
  * @param serviceIntf The remote service interface
  * @param moduleBaseURL Base URL
  * @param remoteServiceRelativePath The remote service servlet relative path
  * @return A new proxy object which implements the service interface serviceIntf
  */
  @SuppressWarnings("unchecked")
  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                       String remoteServiceRelativePath){
    return newProxyInstance(serviceIntf, moduleBaseURL, remoteServiceRelativePath, 
                            POLICY_MAP.get(serviceIntf.getName()), DEFAULT_COOKIE_MANAGER);
  }
  /**
  * Create a new Proxy for the specified service interface <code>serviceIntf</code>
  * 
  * @param serviceIntf The remote service interface
  * @param moduleBaseURL Base URL
  * @param remoteServiceRelativePath The remote service servlet relative path
  * @param cookieManager Used to perform session management such as login. 
  * @return A new proxy object which implements the service interface serviceIntf
  */
  @SuppressWarnings("unchecked")
  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                       String remoteServiceRelativePath, 
                                        CookieManager cookieManager){
    return newProxyInstance(serviceIntf, moduleBaseURL, remoteServiceRelativePath, 
                            POLICY_MAP.get(serviceIntf.getName()), cookieManager);
  }

  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                        CookieManager cookieManager, 
                                        boolean waitForInvocation){
	  
	if (Boolean.valueOf(System.getProperty("IgnoreInvalidCertificates","true"))) {
	  acceptAllSslCertificates();  
	}
	
    RemoteServiceRelativePath relativePathAnn = (RemoteServiceRelativePath)
      serviceIntf.getAnnotation(RemoteServiceRelativePath.class);
    if (serviceIntf.getName().endsWith("Async")){
      // Try determine remoteServiceRelativePath from the 'sync' version of the Async one
      String className = serviceIntf.getName();
      try {
        Class clazz = Class.forName(className.substring(0, className.length()-5));
        relativePathAnn = (RemoteServiceRelativePath)
              clazz.getAnnotation(RemoteServiceRelativePath.class);
      } catch (ClassNotFoundException e) {
        e.printStackTrace();
      }
    }
    if (relativePathAnn == null){
      throw new RuntimeException(serviceIntf + " does not has a RemoteServiceRelativePath annotation");
    }
    String remoteServiceRelativePath = relativePathAnn.value();
    return newProxyInstance(serviceIntf, moduleBaseURL, remoteServiceRelativePath, 
                            POLICY_MAP.get(serviceIntf.getName()), cookieManager, waitForInvocation);
  }

  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                        CookieManager cookieManager){
    return newProxyInstance(serviceIntf, moduleBaseURL, cookieManager, false);
  }
  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL){
    return newProxyInstance(serviceIntf, moduleBaseURL, DEFAULT_COOKIE_MANAGER);
  }

  public static Object newProxyInstance(Class serviceIntf, String moduleBaseURL, 
                                        boolean waitForInvocation){
    return newProxyInstance(serviceIntf, moduleBaseURL, 
                            DEFAULT_COOKIE_MANAGER, waitForInvocation);
  }
  
  private static void acceptAllSslCertificates () {
	// Create a trust manager that does not validate certificate chains
	  TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager(){
	      public X509Certificate[] getAcceptedIssuers(){return null;}
	      public void checkClientTrusted(X509Certificate[] certs, String authType){}
	      public void checkServerTrusted(X509Certificate[] certs, String authType){}
	  }};
	  
	  // Ignore differences between given hostname and certificate hostname
	  HostnameVerifier hv = new HostnameVerifier() {
	    public boolean verify(String hostname, SSLSession session) { return true; }
	  };


	  // Install the all-trusting trust manager
	  try {
	      SSLContext sc = SSLContext.getInstance("TLS");
	      sc.init(null, trustAllCerts, new SecureRandom());
	      HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
	      HttpsURLConnection.setDefaultHostnameVerifier(hv);
	  } catch (Exception e) {
	      throw new RuntimeException(e);
	  }
  }
  
}
