import java.io.*;
import java.awt.*;
import java.util.*;


import jet.api.*;
import jet.jetc.*;
import jet.universe.exception.*;

/**
 * This class uses the Designer class API to programmatically create a page report (.cls) named by the -report parameter. 
 * The report that is created is similar to InvoiceReport.cls in SampleReports.
 * The report will use the catalog specified by -cat on a path that is specified by -path that must contain the catalog.
 * Optionally, a specific log file can be named to hold log messages.
 * 
 * Note the first 7 steps listed below.  All reports require a similar sequence to build the report structure
 *
 * Usage: TestDesignInvoice -path=<path> -cat=<catalog> -report=<name> [-log=<logfile>]
 */
public class TestDesignInvoice
{
    Designer designer = null;   // Instance of the Designer object used by this class.
    Boolean ready = false;      // True - designerReady is ready to use.
    String errorMsg = null;     // Holds error message related to a failure that is visible to main().
    HashMap<Integer, String> handleNames = new HashMap<Integer, String>();

    public static int index=0; 

    /**
     * Constructor. Create a new TestDesignInvoice object holding an instance of the Designer object, with a report catalog context, which will be used to create a report.
     *              On success, the state variable "designerReady" will be true.
     *              On failure, the variable "designerErrorMsg" will hold an explanation.
     */
    public TestDesignInvoice(String user, String password, String path, String catalog, String log) throws IOException, ReadFailureException, UniverseFileException
    {
        // Create an authorized user object based on a licensed user and password.
        DesignerUserInfo authUser = new DesignerUserInfo(user, password);

        // Create the report Designer object that will work within the context of a given catalog at the given full path.
        // The Designer object will be used to create and define a report programmatically.
        this.designer = new Designer(path, catalog, authUser);
        if (this.designer == null) {
            this.errorMsg = "Failed to create a Designer object for " + path + " " + catalog;
            this.ready = false;
            return;
        } 

        // turn on logging and have it go to the given file, when one is given.
        if ( log != null && log.length() > 0 ) {
            this.designer.setLog(new FileOutputStream(log), "8859_1");
        } 


        // Fill up hash map to support converting handle type to name, for logging.
        handleNames.put(new Integer(Designer.PAGE),        "Page");
        handleNames.put(new Integer(Designer.PAGEHEADER),  "Page Header");
        handleNames.put(new Integer(Designer.PAGEFOOTER),  "Page Footer");
        handleNames.put(new Integer(Designer.PAGEEND),     "Page End");
        handleNames.put(new Integer(Designer.PAGESTART),   "Page Start");
        handleNames.put(new Integer(Designer.DATASET),     "Dataset");
        handleNames.put(new Integer(Designer.BANDEDOBJECT_HEADER),     "Banded Object Header");
        handleNames.put(new Integer(Designer.BANDEDOBJECT_FOOTER),     "Banded Object Footer");
        handleNames.put(new Integer(Designer.BANDEDOBJECT),            "Banded Object");
        handleNames.put(new Integer(Designer.BANDEDOBJECT_PAGEHEADER), "Banded Object Page Header");
        handleNames.put(new Integer(Designer.BANDEDOBJECT_PAGEFOOTER), "Banded Object Page Footer");
        handleNames.put(new Integer(Designer.GROUPHEADER), "Group Header");
        handleNames.put(new Integer(Designer.GROUPFOOTER), "Group Footer");
        handleNames.put(new Integer(Designer.DETAIL),      "Detail");
        handleNames.put(new Integer(Designer.REPORTSET), "Report Set");
        handleNames.put(new Integer(Designer.REPORTBODY), "Report Body");
        handleNames.put(new Integer(Designer.DATAOBJECT),  "Data Container"); 
        handleNames.put(new Integer(Designer.REPORTSHEET), "Report Sheet");
        handleNames.put(new Integer(Designer.REPORTFOOTER),  "Report Footer"); 
        handleNames.put(new Integer(Designer.REPORTHEADER), "Report Header");

        // initialization stage was a success
        this.ready = true;
        return;
    }

    public boolean isReady()
    {
       return this.ready;
    }

    public String getErrorMsg()
    {
       return this.errorMsg;
    }
/**
 * Show the handles of the current report structure, to assist in debugging programs using the Designer class methods.
 */
    public void showHandles(String label)
    {
        Integer type;
        String sType;
        String handles[] = designer.getHandles();    
        System.out.println("---- " + label + " --- size: " + handles.length + " ----");
        for (int i = 0; i < handles.length; i++) {
            type=designer.getObjectType(handles[i]);

            // generate readable and output-formatted name of type
            // intent is to show type and name and ignore unknown types
            sType=handleNames.get(type);
            if(sType != null) 
            {
            	sType="("+sType+")";
            	System.out.println(handles[i] + "  Type: " + type + sType  );
            }
            else if (type != -1)
            {
            	System.out.println(handles[i] + "Type: " + type);
            }
        }
        System.out.println("---- " + label + " --- end");
   
        return;
    }

    /**
     *  Close the report designer object gracefully.
     *  Save edits when requested.
     */
    public void close(Boolean fSaveEdits)
    {
        if (fSaveEdits) {
             // Leave edit mode and save the changes.
             designer.exit();
        } else {
             // Leave edit mode and do not save the changes.
             designer.quit();
        }

        // Close logging last, in case something is logged during exit() or quit().
        try {
            designer.closeLog();
        }
        catch ( IOException e ) {
            e.printStackTrace();
        }
    }

    /**
     * Create the invoice report using the designated name.
     *
     *   The specifics of the report, including data source and all layout details are hard coded here.
     *   The intent is to show the use of the methods that set the report structure and attributes.
     *
     *   Error handling:
     *     Parts of the code here use rigorous error checking,  making sure that each method call was a success, to show how this style eould be done if needed.
     *     Other parts do not check that each method is a success.  If a failure happens within a method call , 
     *     there will an exception thrown or a malformed report layout will occur. The failure in the method call is caught that way.
     *     Use the level of error checking in production code that is appropriate for the target application.
     *     The error handling code here is just an example showing various styles of error handling that can be used.
     */
    public boolean create(String name)
    {
        String reportset;     // Holds internal name of the created report set. Used while defining the report set.
        String report;        // Holds internal name of the created report. Used while defining the report.
        String dataset;       // Holds internal name of the created dataset. Used while registering it with the report.
        String groupCust,groupOrd;         // Holds internal name of a group that is inserted into report. Used while defining the group.
        String header;        // Holds internal name of a header that is inserted into the report. Used while defining the header.
        String detail;        // Holds internal name of a page detail.  Used while defining the detail.
        String footer;        // Holds internal name of a footer.  Used while defining the footer.
        String handle;        // Holds internal name of a variety of inserted report elements.  Used while setting attributes for each.
        String handles[];     // Array of report element names returned from getHandles() call. 
        String banded;        // Holds the internal name of the banded object
        String reportbody;    // Holds the internal name of the report body that the banded object goes into 
        Boolean fSuccess = false;  //  true - the goal of creating a report is accomplished 
  
        System.out.println("---start create -----");
  
        /**
         *  Section 1 - 
         *    This section uses a style of rigorous error handling - check the result of each method call, and handle each specific error.
         *
         * Approach:
         *  When something fails during a method call, the code will know it immediately and will know exactly the case.
         *  
         *  Called methods report failure in two ways - throw an exception, or return a value that indicates error.
         *  This section watches for failures in two ways - by having calls within a try/catch loop for noticing thrown exceptions
         *  and by checking the returned values for the particular value that indicates error.
         *  
         *  Code is more complicated.
         *
         *  Appropriate when doing a sequence of operations that depend on the one before, with the full sequence needing to be done.
         */

        try {
            // Remove any existing page report (.cls) with the same name to avoid an error when creating one.
            // This generates an entry in the error log reporting that "name" does not exist, when it doesn't.
            designer.deleteReportFile(name);

            // 1. Create a page report.  use the name from the -name parameter. 
            reportset = designer.createReportSet(name);
            if (reportset == null) {
                 this.errorMsg = "Failure createReportSet(): " + designer.getError();
                 return false;
            }

            // 2. Register the existing query "WorldWideSales" to this report set.
            //    The existing query in the catalog is the third parameter.
            //    The corresponding new data source is the second parameter.
            //    The name for this registered dataset is fourth parameter.
            //    The type of dataset is in the fifth parameter.
            dataset = designer.addDataset(reportset, "Data Source 1", "WorldWideSales","dsWorldWideSales", "",Designer.DATASET_TYPE_QUERY);
            if (dataset == null) {
                this.errorMsg = "Failure addDataSet(): " + designer.getError();
                return false;
            }
                  
            // 3. Create the named report tab within the Page Report
            //    For this example program, the name of the added report tab is hard coded.
            report = designer.addReport(reportset, "Invoice");
            if (report == null) {
                this.errorMsg = "Failure addReport(): " + designer.getError();
                return false;
            }
 
            reportbody = designer.getHandles(report, Designer.REPORTBODY, -1)[0];
             
            // 4. Insert a PagePanel into the Invoice tab
            String page = designer.insertPage(report);
             
            designer.insertPageStartEnd(page, reportbody, 0, 0);
            designer.set(page, "Width", 8.0f);
            designer.set(page, "Height", 9.2f);
              
            //showHandles("after insert page");
              
            String[] pageFooter = designer.getHandles(report, Designer.PAGEFOOTER,    -1);
            designer.set(pageFooter[0], "Height", 0.2f);

            // 5. Insert the banded object for the invoice report             
            banded = designer.insertBandedObject(reportbody, true, true, true);
            designer.set(banded, "Width", 8.0f);
            // designer.set(banded, "Height", 9.2f);

            String[] sections = designer.getSections(banded);

            //to set width of all sections in banded object 
            for ( int i = 0; i< sections.length;i++)
            {
            	designer.set(sections[i], "Width", 8.0f);
            }

            handles = designer.getHandles(banded, Designer.BANDEDOBJECT_HEADER);
            System.out.println("banded object header handle: " + handles[0]);
            designer.set(handles[0], "Invisible", true);
             
            // You should see that we have a Report tab called Invoice, a ReportBody, a PagePanel and a BandedObject in the tree
            showHandles("after insert banded object");

            // 6. bind the dataset to the report body and set banded object to inherit the dataset
            // it is an container object that must have a data set bound to it.
            if( ! designer.bindDataSet(reportbody, "dsWorldWideSales") ) {
                this.errorMsg = "Failure bindDataSet(): " + designer.getError();
                return false;
            }
            designer.set(banded, "dataInherit", true);

            //showHandles("after bind dataset");

                
            // 7. ================draw the report
            // insert group panel 1 on Customer ID - we will not show anything in this group
            // use insertSectionGroup() since V8 to add group to banded object.
            handles = designer.getHandles(banded, Designer.PAGE);
            System.out.println("page within banded: " + banded + "  " + handles[0]);
 
            GroupInfo ginfo =new GroupInfo( "Customers_Customer ID", GroupInfo.DESCENDING );
            groupCust = designer.insertSectionGroup( handles[0], ginfo );
 
            if (groupCust == null) {
                this.errorMsg = "Failure insert GroupPanel1(): " + designer.getError();
                return false;
            }
            // You should now also see the group header and group footer panels
            showHandles("after insert group panel ");

            // After inserting a group, the details of the report are moved to this group, and a group header/footer is created.
            // Remove the group header and footer.
            handles = designer.getHandles(groupCust, Designer.GROUPHEADER);
            System.out.println("group header 1 handle: " + handles[0]);
            designer.set(handles[0], "Width", 7.5f);
            designer.set(handles[0], "Height", 0.0f);
            designer.set(handles[0], "Invisible", true);
            // Hide the special group field that is added automatically with the group
            handles = designer.getHandles(handles[0], Designer.GROUPNAME);
            designer.set(handles[0], "Invisible", true);
            // showHandles("after set Group Header invisible ");

            handles = designer.getHandles(groupCust, Designer.GROUPFOOTER);
            designer.set(handles[0], "Width", 7.5f);
            designer.set(handles[0], "Height", 0.0f);
            designer.set(handles[0], "Invisible", true);
             
            // showHandles("after set Group Footer invisible ");

            //insert group panel 2 based on Order ID within Customer ID

            ginfo = new GroupInfo( "Orders_Order ID", GroupInfo.DESCENDING );
            groupOrd = designer.insertSectionGroup( groupCust, ginfo );
            handles = designer.getHandles(groupOrd, Designer.GROUPHEADER);
            System.out.println("group header 2 handle: " + handles[0]);
 
            if (groupOrd == null) {
                this.errorMsg = "Failure insert GroupPanel2(): " + designer.getError();
                return false;
            }
            // After inserting a group into a panel the detail panel of the parent group are moved into new group
            //showHandles("after insert Group Panel2");
  
            // style the header
            header = designer.getHandles(groupOrd,Designer.GROUPHEADER)[0];
            if (! designer.set(header, "Label", "GroupHeaderPanel(OrderID)") ) {
                this.errorMsg = "Failure set Label GroupHeaderPanel: " + designer.getError();
                return false;
            }
            designer.set(header, "Width", 7.5f);
            designer.set(header, "Height", 1.75f);
   

        }
        catch ( Exception e ) {
            e.printStackTrace();
            this.errorMsg = "Exception while registering report set and report, and while defining top section of report.";
            return fSuccess;
        }

        /**
         *  Section 2 - 
         *    This section uses a style of casual error handling - don't check the result of each method call.
         *
         *  Error handling approach:
         *   Don't check the result of each method call.
         *   When the insert() call fails, the use of the null returned handle will cause the following set() call to throw an exception.
         *   This will percolate up the stack and be caught by main(), where main logic can back out the edits made to the report.
         *   If an insert() succeeds but a set() fails, the set() method returns a boolean which is ignored by this style error handling.  
         *   If the set() method fails for an attibute, that attribute will not be set.  This causes the report design to look wrong.
         *   It should be acceptable that the report may look bad after running this style error handling code.
         *
         *   This style makes the code simple to read.
         *
         *   This style is appropriate where exact details of the failure is not important, and where
         *   failures in a set command only cause acceptable cosmetic problems.
         */
        System.out.println("group header handle in second section: " + header);
        
        //1. Draw GroupHeader 1
        handle = designer.insert(header, Designer.LABEL, "Label1");
        designer.set(handle, "Width", 1.74f);
        designer.set(handle, "Height", 0.35f);
        designer.set(handle, "Text", "Invoice");
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Foreground", new Color(0x900000));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.31f);

        handle = designer.insert(header, Designer.LABEL, "Label2");
        designer.set(handle, "Width", 0.68f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "Text", "Date:");
        designer.set(handle, "Y", 0.36f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Foreground", new Color(0x00007f));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label3");
        designer.set(handle, "Width", 0.86f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "Text", "Invoice No:");
        designer.set(handle, "Y", 0.55f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Foreground", new Color(0x00007f));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.PARAMETER, "Parameter1", "PStartDate");
        designer.set(handle, "Width", 1.14f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "Format", "MM/dd/yyyy");
        designer.set(handle, "X", 0.68f);
        designer.set(handle, "Y", 0.36f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "FontSize", 0.10f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField1", "Orders_Order ID");
        designer.set(handle, "Width", 0.64f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 0.93f);
        designer.set(handle, "Y", 0.55f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "#####");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.IMAGE, "Image1");
        designer.set(handle, "Width", 0.93f);
        designer.set(handle, "Height", 0.81f);
        designer.set(handle, "X", 3.80f);
        designer.set(handle, "Y", 0.05f);
        designer.set(handle, "AutoSize", true);
        designer.set(handle, "PictureName", "Coffee.gif");

        handle = designer.insert(header, Designer.LABEL, "Label4");
        designer.set(handle, "Width", 0.576f);
        designer.set(handle, "Height", 0.21f);
        designer.set(handle, "Text", "Bill to:");
        designer.set(handle, "Y", 0.94f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField2", "Customer Name");
        designer.set(handle, "Width", 1.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 0.74f);
        designer.set(handle, "Y", 0.93f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.DBFIELD, "DBFieldAdd1", "Address1");
        designer.set(handle, "Width", 1.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 0.74f);
        designer.set(handle, "Y", 1.11f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.FORMULA, "formula2", "CustomerCityStateZip");
        designer.set(handle, "Width", 1.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 0.74f);
        designer.set(handle, "Y", 1.29f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField3", "Customers_Country");
        designer.set(handle, "Width", 1.30f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 0.74f);
        designer.set(handle, "Y", 1.47f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label5");
        designer.set(handle, "Width", 0.646f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "Text", "Ship to:");
        designer.set(handle, "X", 2.80);
        designer.set(handle, "Y", 0.93f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField4", "Customer Name");
        designer.set(handle, "Width", 1.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 3.49f);
        designer.set(handle, "Y", 0.93f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.DBFIELD, "DBFieldAdd2", "Address1");
        designer.set(handle, "Width", 1.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 3.49f);
        designer.set(handle, "Y", 1.11f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.FORMULA, "formula4", "CustomerCityStateZip");
        designer.set(handle, "Width", 1.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 3.49f);
        designer.set(handle, "Y", 1.29f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField5", "Customers_Country");
        designer.set(handle, "Width", 1.30f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 3.49f);
        designer.set(handle, "Y", 1.47f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        
        handle = designer.insert(report, Designer.LINE, "Line1");
        designer.setReference(handle, "TopAttach", header);
        designer.setReference(handle, "BottomAttach", header);
        designer.set(handle, "TopAttachPosX", 0f);
        designer.set(handle, "TopAttachPosY", 0.9f);
        designer.set(handle, "BottomAttachPosX", 2.2f);
        designer.set(handle, "BottomAttachPosY", 0.9f);
        designer.set(handle, "LineColor", Color.blue);
        designer.set(handle, "LineWidth", 0.01f);

        handle = designer.insert(report, Designer.LINE, "Line2");
        designer.setReference(handle, "TopAttach", header);
        designer.setReference(handle, "BottomAttach", header);
        designer.set(handle, "TopAttachPosX", 2.8f);
        designer.set(handle, "TopAttachPosY", 0.9f);
        designer.set(handle, "BottomAttachPosX", 4.8f);
        designer.set(handle, "BottomAttachPosY", 0.9f);
        designer.set(handle, "LineColor", new Color(0x00007f));
        designer.set(handle, "LineWidth", 0.01f);

try{
        header = designer.insertSectionPanel(groupOrd, header,Designer.GROUPHEADER, false);
        designer.set(header, "Label", "GroupHeaderPanel(OrderID)");
        designer.set(header, "Height", 0.86f);
        designer.set(header, "Width", 7.5f);
}
catch(Exception e )
{
	e.printStackTrace();
}


        //2. Draw GroupHeader 2
        handle = designer.insert(report, Designer.BOX, "Box1");
        designer.set(handle, "TopAttachPosX", 0f);
        designer.set(handle, "TopAttachPosY", 0.18f);
        designer.set(handle, "BottomAttachPosX", 4.8f);
        designer.set(handle, "BottomAttachPosY", 0.48f);
        designer.set(handle, "BorderColor", new Color(0x00007f));
        designer.set(handle, "BorderWidth", 0.01f);

        
        handle = designer.insert(header, Designer.LABEL, "Label6");
        designer.set(handle, "Width", 0.60f);
        designer.set(handle, "Height", 0.13f);
        designer.set(handle, "Text", "Cust ID");
        designer.set(handle, "Y", 0.18f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.09f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField6", "Customers_Customer ID");
        designer.set(handle, "Width", 0.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 0.02f);
        designer.set(handle, "Y", 0.31f);
        designer.set(handle, "Format", "#####");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label7");
        designer.set(handle, "Width", 0.60f);
        designer.set(handle, "Height", 0.13f);
        designer.set(handle, "Text", "No.");
        designer.set(handle, "X", 0.60f);
        designer.set(handle, "Y", 0.18f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.09f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField7", "Orders_Order ID");
        designer.set(handle, "Width", 0.60f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 0.60f);
        designer.set(handle, "Y", 0.31f);
        designer.set(handle, "Format", "#####");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label8");
        designer.set(handle, "Width", 0.70f);
        designer.set(handle, "Height", 0.13f);
        designer.set(handle, "Text", "Order Date");
        designer.set(handle, "X", 1.2f);
        designer.set(handle, "Y", 0.18f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.09f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField8", "Order Date");
        designer.set(handle, "Width", 0.70f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 1.2f);
        designer.set(handle, "Y", 0.31f);
        designer.set(handle, "Format", "MM/dd/yyyy");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label9");
        designer.set(handle, "Width", 0.70f);
        designer.set(handle, "Height", 0.13f);
        designer.set(handle, "Text", "Required Date");
        designer.set(handle, "X", 1.9f);
        designer.set(handle, "Y", 0.18f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.09f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField9", "Order Date");
        designer.set(handle, "Width", 0.70f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 1.9f);
        designer.set(handle, "Y", 0.31f);
        designer.set(handle, "Format", "MM/dd/yyyy");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label10");
        designer.set(handle, "Width", 0.70f);
        designer.set(handle, "Height", 0.13f);
        designer.set(handle, "Text", "Ship Date");
        designer.set(handle, "X", 2.6f);
        designer.set(handle, "Y", 0.18f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.09f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField10", "Order Date");
        designer.set(handle, "Width", 0.70f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 2.6f);
        designer.set(handle, "Y", 0.31f);
        designer.set(handle, "Format", "MM/dd/yyyy");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label11");
        designer.set(handle, "Width", 1f);
        designer.set(handle, "Height", 0.13f);
        designer.set(handle, "Text", "Ship Via");
        designer.set(handle, "X", 3.3f);
        designer.set(handle, "Y", 0.18f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.09f);

        handle = designer.insert(header, Designer.DBFIELD, "DBField12", "Payment Received");
        designer.set(handle, "Width", 1f);
        designer.set(handle, "Height", 0.18f);
        designer.set(handle, "X", 3.3f);
        designer.set(handle, "Y", 0.31f);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label12");
        designer.set(handle, "Width", 0.51f);
        designer.set(handle, "Height", 0.13f);
        designer.set(handle, "Text", "Terms");
        designer.set(handle, "X", 4.3f);
        designer.set(handle, "Y", 0.18f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.09f);

        handle = designer.insert(header, Designer.LABEL, "Label13");
        designer.set(handle, "Width", 0.51f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "Text", "30 days");
        designer.set(handle, "X", 4.3f);
        designer.set(handle, "Y", 0.3f);
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(header, Designer.LABEL, "Label14");
        designer.set(handle, "Width", 1.05f);
        designer.set(handle, "Height", 0.16f);
        designer.set(handle, "Text", "Item");
        designer.set(handle, "Y", 0.71f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        handle = designer.insert(header, Designer.LABEL, "Label15");
        designer.set(handle, "Width", 0.9f);
        designer.set(handle, "Height", 0.16f);
        designer.set(handle, "Text", "Product No.");
        designer.set(handle, "X", 1.05f);
        designer.set(handle, "Y", 0.71f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        handle = designer.insert(header, Designer.LABEL, "Label16");
        designer.set(handle, "Width", 0.95f);
        designer.set(handle, "Height", 0.16f);
        designer.set(handle, "Text", "Unit Price");
        designer.set(handle, "X", 1.95f);
        designer.set(handle, "Y", 0.71f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        handle = designer.insert(header, Designer.LABEL, "Label17");
        designer.set(handle, "Width", 0.95f);
        designer.set(handle, "Height", 0.16f);
        designer.set(handle, "Text", "Quantity");
        designer.set(handle, "X", 2.9f);
        designer.set(handle, "Y", 0.71f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        handle = designer.insert(header, Designer.LABEL, "Label18");
        designer.set(handle, "Width", 0.95f);
        designer.set(handle, "Height", 0.16f);
        designer.set(handle, "Text", "Amount");
        designer.set(handle, "X", 3.8f);
        designer.set(handle, "Y", 0.71f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        // get group2.detail
        detail = designer.getHandles(groupOrd, Designer.DETAIL)[0];
        designer.set(detail, "Label", "Detail");
        designer.set(detail, "Width", 7.5f);
        designer.set(detail, "Height", 0.25f);

        //3. Draw detail panel
        handle = designer.insert(detail, Designer.DBFIELD, "DBField13", "Product Name");
        designer.set(handle, "Width", 1f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "X", 0.02f);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(detail, Designer.DBFIELD, "DBField14", "Products_Product ID");
        designer.set(handle, "Width", 0.85f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "X", 1.05f);
        designer.set(handle, "Format", "#####");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(detail, Designer.DBFIELD, "DBField15", "Unit Price");
        designer.set(handle, "Width", 0.94f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "X", 1.95f);
        designer.set(handle, "Format", "$#,##0.00");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(detail, Designer.DBFIELD, "DBField16", "Quantity");
        designer.set(handle, "Width", 0.9f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "X", 2.9f);
        designer.set(handle, "Format", "#####");
        designer.set(handle, "FontSize", 0.11f);

        handle = designer.insert(detail, Designer.FORMULA, "Formula17", "Total");
        designer.set(handle, "Width", 0.95f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "X", 3.8f);
        designer.set(handle, "Format", "$#,##0.00");
        designer.set(handle, "Alignment", "RIGHTCENTER");
        designer.set(handle, "FontSize", 0.11f);

        // get group2.footer
        footer = designer.getHandles(groupOrd, Designer.GROUPFOOTER)[0];
        designer.set(footer, "Label", "GroupFooterPanel(OrderID)");
        designer.set(footer, "FillWholePage", true);
        designer.set(footer, "Height", 0.21f);

        //4. Draw groupfooter panel
        handle = designer.insert(footer, Designer.LABEL, "Label19");
        designer.set(handle, "Width", 0.52f);
        designer.set(handle, "Height", 0.21f);
        designer.set(handle, "Text", "Total:");
        designer.set(handle, "X", 2.84f);
        designer.set(handle, "Background", new Color(0x000090));
        designer.set(handle, "Foreground", new Color(0xffffff));
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.13f);

        handle = designer.insert(footer,  Designer.SUMMARY, "Summary1", "Sum_ProductSales_DynamicSummary");
        // handle = designer.insert(footer,  Designer.SUMMARY, "Summary1", "Sum_ProductSales_StaticGroupID");
        designer.set(handle, "Width", 0.95f);
        designer.set(handle, "Height", 0.21f);
        designer.set(handle, "X", 3.8f);
        designer.set(handle, "Alignment", "RIGHTCENTER");
        designer.set(handle, "Format", "$#,##0.00");
        designer.set(handle, "FontSize", 0.13f);
        
        //5. Draw pagefooter panel
        footer = designer.getHandles(report, Designer.PAGEFOOTER, -1)[0];
        designer.set(footer, "Height", 0.6f);
        handle = designer.insert(footer, Designer.DBFIELD, "DBField18", "Customer Name");
        designer.set(handle, "Width", 4.15f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        handle = designer.insert(footer, Designer.FORMULA, "formula19", "CustomerCityStateZip");
        designer.set(handle, "Width", 4.15f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "Y", 0.2f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        handle = designer.insert(footer, Designer.DBFIELD, "DBField19", "Customers_Phone");
        designer.set(handle, "Width", 4.15f);
        designer.set(handle, "Height", 0.2f);
        designer.set(handle, "Y", 0.4f);
        designer.set(handle, "Alignment", Designer.LEFTCENTER);
        designer.set(handle, "Format", "XXXXXXXXXX..");
        designer.set(handle, "Bold", true);
        designer.set(handle, "FontSize", 0.10f);

        fSuccess = true;  // report was made successfully
        return fSuccess;
    }

    // parameter tokens
    private final static String pathArg = "-path=";
    private final static String catArg = "-catalog=";
    private final static String rptArg = "-report=";
    private final static String logArg = "-log=";

    private final static void showUsage() {
        System.out.println( "TestDesignInvoice - programattically define an invoice report, registered into a report catalog.");
        System.out.println( "Usage:  java TestDesignInvoice <-options>");
        System.out.println("\nwhere options include:");
        System.out.println("    " + pathArg +   "<path to catalog>  indicates the full path name to the report catalog directory");
        System.out.println("    " + catArg +    "<catalog name>  indicates the name of the report catalog");
        System.out.println("    " + rptArg +    "<report name>  indicates the name of the report to create");
        System.out.println("    " + logArg +    "<log file path and name>   indicates to log to this file, defined by full path name");
        System.out.println("\nExample to create and define a report from the command line:");
        System.out.println("java TestDesignInvoice -path=C:\\JReport\\Designer\\Demo\\Reports\\TutorialReports  -catalog=JinfonetGourmetJava.cat  -report=TestInvoice.cls -log=C:\\JReport\\Designer\\logs\\designer.log");
    }

    public static void main(String argv[])
    {
        // values of command line arguments.
        String path = "";
        String catalog = "";
        String report = "";
        String log = "";

        // get values of command line arguments
        for (int i = 0; i < argv.length; i++) {
            if (argv[i].toLowerCase().startsWith(catArg)) {
                catalog = argv[i].substring(catArg.length(), argv[i].length());
            } else if (argv[i].toLowerCase().startsWith(rptArg)) {
                report = argv[i].substring(rptArg.length(), argv[i].length());
            } else if (argv[i].toLowerCase().startsWith(pathArg)) {
                path = argv[i].substring(pathArg.length(), argv[i].length());
            } else if (argv[i].toLowerCase().startsWith(logArg)) {
                log = argv[i].substring(logArg.length(), argv[i].length());
            } else {
                // is not a valid command line argument
                showUsage();
                System.exit(1);
            }
         }    

        // log file is an optional parameter.
        // path, catalog, and report are required parameters.
        if ( (path.length() == 0) || (catalog.length() == 0) || report.length() == 0 ) {
                showUsage();
                System.exit(1);
        }

        System.out.println("Catalog = " + catalog);
        System.out.println("   Path = " + path);
        System.out.println(" Report = " + report);
        System.out.println("    Log = " + ((log==null) ? "" : log));

        String stage = "init";  // stage holds processing step so exception handling will know where the exception was thrown
        TestDesignInvoice invoiceDesigner = null;
        try {
            // This is the licensed user for JReport DesignAPI.
            String user = "Jinfonet";
            String password ="XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX";

            // setup an authorized report designer, to design a report inside the given report catalog.
            invoiceDesigner = new TestDesignInvoice(user, password, path, catalog, log );
             
            // our local object may have failed the setup steps.
            if (! invoiceDesigner.isReady()) {
                System.out.println("Setup error: " + invoiceDesigner.getErrorMsg());
                System.exit(1);
            }
 
            stage = "create";
            // Define the report, using our local create() method, which creates an invoice report with specific layouts and datasource. 
            // On failure within create(), an error message will be in invoiceDesigner's errorMsg string. 
            Boolean fSuccess = invoiceDesigner.create(report);

            stage = "close";
            // Close the local designer object.  If the create stage was a success, save the edits, otherwise, drop the edits.
            invoiceDesigner.close(fSuccess);

            stage = "closed";
            // On success, the report will be defined and be inside the given catalog for viewing with the Designer.
            // On failure, the report will not exist.
            if (fSuccess) {
                System.out.println("Success:  Page Report " + report + " was created in catalog " + catalog + " at " + path);
                System.out.println("Success:  You can view it in the Designer now.");
            } else {
                System.out.println("Failure:  Unable to create Page Report " + report); 
                System.out.println("Failure:  " + invoiceDesigner.getErrorMsg());
                if ( log != null && log.length() > 0 ) {
                    System.out.println("Failure:  Check the log file " + log);
                } else {
                    System.out.println("Failure:  Run again with a log file set, to get more information.");
                }
            }
        } catch (Exception e) {
            System.out.println("Failure:  Exception in main() during stage " + stage);
            e.printStackTrace();
            if (stage.equals("create")) {
                invoiceDesigner.close(false);  // discard edits while closing
            }
            System.exit(1);
        }

        System.exit(0);
    }
}
