package com.tandbergtv.neptune.ui.realm.client.tab.role;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.framework.client.i18n.NeptuneConstants;
import com.tandbergtv.neptune.ui.realm.client.event.RoleAddedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.RoleDeletedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.RoleUpdatedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.UserAddedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.UserDeletedEvent;
import com.tandbergtv.neptune.ui.realm.client.event.UserUpdatedEvent;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventListener;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventListenerRegistry;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventSink;
import com.tandbergtv.neptune.widgettoolkit.client.event.NeptuneEvent;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem.AnchorChangeListener;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.KeySerializer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelIntegerView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class RoleTableProvider {
	private final EventSink eventSink;
	private final RoleUiServiceAsync roleService = GWT.create(RoleUiService.class);

	private final RoleDataProvider dataProvider;
	private final PageFeatureImpl<UiRoleKey, RoleRecord> pageFeature;
	private final RoleDetailFeature detailFeature;
	private final SortFeatureImpl<UiRoleKey, RoleRecord> sortFeature;
	private final BookmarkFeatureImpl<UiRoleKey, RoleRecord> bookmarkFeature;

	private final Table<UiRoleKey, RoleRecord> table;
	
	private NeptuneSecurity security;
	private static String CREATE_PERMISSION = "RoleManagement_Create";
	private static String DELETE_PERMISSION = "RoleManagement_Delete";

	public RoleTableProvider(EventSink eventSink, EventListenerRegistry eventListenerRegistry, NeptuneSecurity security) {
		this.eventSink = eventSink;
		this.security = security;

		dataProvider = new RoleDataProvider();
		table = new Table<UiRoleKey, RoleRecord>(dataProvider);

		pageFeature = new PageFeatureImpl<UiRoleKey, RoleRecord>(table);
		detailFeature = new RoleDetailFeature();
		sortFeature = new SortFeatureImpl<UiRoleKey, RoleRecord>(dataProvider.nameColumn, SortOrder.ASCENDING);
		bookmarkFeature = new BookmarkFeatureImpl<UiRoleKey, RoleRecord>(table, new KeySerializer<UiRoleKey>() {
			public UiRoleKey fromString(String keyString) {
				return UiRoleKey.parse(keyString);
			}

			public String toString(UiRoleKey key) {
				return key.toString();
			}
		});
		
		sortFeature.addSortableColumn(dataProvider.nameColumn);

		eventListenerRegistry.registerEventListener(UserAddedEvent.class, new UserAddedListener());
		eventListenerRegistry.registerEventListener(UserUpdatedEvent.class, new UserUpdatedListener());
		eventListenerRegistry.registerEventListener(UserDeletedEvent.class, new UserDeletedListener());
	}

	public final Table<?, ?> getTable(AnchorChangeListener reverseAnchorChangeListener) {
		bookmarkFeature.addDetailFeature(reverseAnchorChangeListener, table, dataProvider, detailFeature);
		bookmarkFeature.addPageFeature(reverseAnchorChangeListener, table, pageFeature);

		table.addDetailFeature(detailFeature);
		table.addSortFeature(sortFeature);
		table.addPageFeature(pageFeature);
		table.addBookmarkFeature(bookmarkFeature);
		table.init();

		return table;
	}

	public final class RoleDataProvider implements DataProvider<UiRoleKey, RoleRecord> {
		private final List<Column<?, RoleRecord>> columns;
		private ColumnBase<String, RoleRecord> nameColumn;
		private ColumnBase<Integer, RoleRecord> userCountColumn;
		private ColumnBase<String, RoleRecord> descriptionColumn;
		private ColumnBase<Integer, RoleRecord> permissionsColumn;

		public RoleDataProvider() {
			columns = new ArrayList<Column<?, RoleRecord>>();
			nameColumn = new ColumnBase<String, RoleRecord>("name", "Name") {
				public View<String> getView(RoleRecord record) {
					return new LabelStringView(record.getRole().getName());
				}
			};
			columns.add(nameColumn);
			userCountColumn = new ColumnBase<Integer, RoleRecord>("userCount", "User Count") {
				public View<Integer> getView(RoleRecord record) {
					return new LabelIntegerView(record.getRole().getUserCount());
				}
			};
			columns.add(userCountColumn);
			descriptionColumn = new ColumnBase<String, RoleRecord>("description", "Description") {
				public View<String> getView(RoleRecord record) {
					return new LabelStringView(record.getRole().getDescription());
				}
			};
			columns.add(descriptionColumn);
			permissionsColumn = new ColumnBase<Integer, RoleRecord>("permissions", "Permission Count") {
				public View<Integer> getView(RoleRecord record) {
					return new LabelIntegerView(record.getRole().getPermissionIds().size());
				}
			};
			columns.add(permissionsColumn);
		}

		public List<Column<?, RoleRecord>> getColumns() {
			return columns;
		}

		public void getRecords(final AsyncCallback<List<RoleRecord>> callback) {
			roleService.listRoles(pageFeature.getStart(), pageFeature.getLength(), sortFeature.getSortColumnName(),
					sortFeature.isAscending(), new AsyncCallback<UiRoleList>() {
						public void onFailure(Throwable caught) {
							callback.onFailure(caught);
						}

						public void onSuccess(UiRoleList result) {
							List<RoleRecord> records = new ArrayList<RoleRecord>();
							for (UiRole role : result.getRoles())
								records.add(new RoleRecord(role));
							pageFeature.setTotalSize(result.getTotal());
							callback.onSuccess(records);
						}
					});
		}

		public void initialize(final AsyncCallback<Void> callback) {
			roleService.getModules(new NeptuneAsyncCallback<List<UiModule>>() {
				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onNeptuneSuccess(List<UiModule> modules) {
					detailFeature.modules = modules;

					callback.onSuccess(null);
				}
			});
		}

		public boolean isCheckboxEnabled() {
			return security.isUserInRole(DELETE_PERMISSION);
		}

		public boolean isRecordCountEnabled() {
			return true;
		}

		public void getRecord(UiRoleKey key, final AsyncCallback<RoleRecord> callback) {
			roleService.getRole(key, new NeptuneAsyncCallback<UiRole>() {
				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onNeptuneSuccess(UiRole result) {
					callback.onSuccess(new RoleRecord(result));
				}

			});
		}

	}

	public final class RoleDetailFeature implements DetailFeature<UiRoleKey, RoleRecord> {
		protected List<UiModule> modules;
		private Listener<UiRoleKey, RoleRecord> listener;

		private final class RoleRecordDetailView implements DetailView<UiRoleKey, RoleRecord> {
			private static final String STYLE_VALIDATION_MESSAGE = "realm-validationFailure-message";
			
			private final RoleRecord record;
			private final LabelWidget errorMessageLabel;
			private final TextBoxWidget nameBox;
			private final TextBoxWidget descriptionBox;
			private final TextBoxWidget userCountBox;
			private final ModulePanel modulePanel;

			private final FormContainer mainContainer = new FormContainer(HasHorizontalAlignment.ALIGN_LEFT);
			private final VerticalContainer headerAndMainContainer = new VerticalContainer();

			NeptuneConstants constants = GWT.create(NeptuneConstants.class);
			
			public RoleRecordDetailView(final RoleRecord record,
					final DetailViewCallback<UiRoleKey, RoleRecord> callback) {
				this.record = record;

				errorMessageLabel = new LabelWidget();
				errorMessageLabel.setStyleName(STYLE_VALIDATION_MESSAGE);
				
				nameBox = new TextBoxWidget();
				descriptionBox = new TextBoxWidget();
				userCountBox = new TextBoxWidget();
				userCountBox.setReadOnly(true);
				modulePanel = new ModulePanel(modules);

				mainContainer.addRow("Name:", nameBox, true);
				mainContainer.addRow("User Count:", userCountBox);
				mainContainer.addRow("Description:", descriptionBox);
				mainContainer.addRow("Permissions:", modulePanel);

				// make the form buttons call back the table
				ButtonWidget saveButton = new ButtonWidget("Save", new ClickHandler() {
					public void onClick(ClickEvent event) {
						errorMessageLabel.setText(null);
						callback.save(RoleRecordDetailView.this);
					}
				});
				saveButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
				mainContainer.addButton(saveButton);
				
				ButtonWidget cancelButton = new ButtonWidget("Cancel", new ClickHandler() {
					public void onClick(ClickEvent event) {
						errorMessageLabel.setText(null);
						callback.cancel(RoleRecordDetailView.this);
					}
				});
				cancelButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
				mainContainer.addButton(cancelButton);
				
				ButtonWidget revertButton = new ButtonWidget("Revert", new ClickHandler() {
					public void onClick(ClickEvent event) {
						errorMessageLabel.setText(null);
						callback.revert(RoleRecordDetailView.this);
					}
				});
				revertButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
				mainContainer.addButton(revertButton);
				revert();
			}

			public void commit() {
				record.getRole().setName(nameBox.getText());
				record.getRole().setDescription(descriptionBox.getText());
				record.getRole().setPermissionIds(modulePanel.getData());
			}

			public RoleRecord getRecord() {
				return record;
			}

			public void revert() {
				nameBox.setText(record.getRole().getName());
				descriptionBox.setText(record.getRole().getDescription());
				int userCount = record.getRole().getUserCount() != null ? record.getRole().getUserCount() : 0;
				userCountBox.setText(String.valueOf(userCount));
				modulePanel.setData(record.getRole().getPermissionIds());
			}

			@Override
			public Widget getWidget() {
				String roleName = record.getRole().getName();
				if(roleName == null) {
					roleName = "";
				}
				headerAndMainContainer.add(new HeaderPanel(constants.roleDetails() + " " + roleName));
				headerAndMainContainer.add(errorMessageLabel);
				headerAndMainContainer.add(mainContainer);
				return headerAndMainContainer;
			}

			public void release() {
			}

			public String getStyleName() {
			    return null;
			}

			@Override
			public void saveFailed(Throwable throwable) {
				if(throwable instanceof ValidationException) {
					ValidationException ve = (ValidationException) throwable;
					StringBuffer messagesSB = new StringBuffer();
					for(String message : ve.getValidationMessages()) {
						messagesSB.append(message).append("\n");
					}
					errorMessageLabel.setText(messagesSB.toString());
				} else {
					errorMessageLabel.setText(throwable.getLocalizedMessage());
				}
			}
		}

		public void delete(final List<RoleRecord> records, final AsyncCallback<Void> callback) {
			List<UiRoleKey> roleKeys = new ArrayList<UiRoleKey>();
			for (RoleRecord record : records)
				roleKeys.add(record.getRole().getKey());
			roleService.deleteRoles(roleKeys, new AsyncCallback<Void>() {
				public void onFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onSuccess(Void result) {
					for (RoleRecord record : records)
						eventSink.fireEvent(new RoleDeletedEvent(record.getRole()));
					callback.onSuccess(null);
				}
			});

		}

		public void getNew(AsyncCallback<RoleRecord> callback) {
			callback.onSuccess(new RoleRecord(new UiRole()));
		}

		public DetailView<UiRoleKey, RoleRecord> getView(RoleRecord record,
				DetailFeature.DetailViewCallback<UiRoleKey, RoleRecord> callback) {
			final RoleRecordDetailView roleRecordDetailView = new RoleRecordDetailView(record, callback);
			listener.detailViewShown(roleRecordDetailView);
			return roleRecordDetailView;
		}

		public boolean hasDetailLink(Column<?, RoleRecord> column) {
			return column.getName().equals("name");
		}

		public void save(final RoleRecord record, final AsyncCallback<Void> callback) {
			final UiRole role = record.getRole();
			roleService.save(role, new NeptuneAsyncCallback<UiRoleKey>() {

				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				public void onNeptuneSuccess(UiRoleKey userId) {
					NeptuneEvent event = role.getId() == null ? new RoleAddedEvent(role) : new RoleUpdatedEvent(record
							.getRole());
					record.getRole().setKey(userId);
					eventSink.fireEvent(event);
					callback.onSuccess(null);
				}
			});
		}

		public void registerListener(DetailFeature.Listener<UiRoleKey, RoleRecord> listener) {
			this.listener = listener;
		}

		@Override
		public boolean showCreateButton() {
			return security.isUserInRole(CREATE_PERMISSION);
		}

		@Override
		public boolean showDeleteButton() {
			return security.isUserInRole(DELETE_PERMISSION);
		}

	}

	public final static class RoleRecord implements Record<UiRoleKey> {
		private final UiRole role;

		public RoleRecord(UiRole role) {
			this.role = role;
		}

		public UiRole getRole() {
			return role;
		}

		public UiRoleKey getKey() {
			return role.getKey();
		}
	}

	private final class UserAddedListener implements EventListener<UserAddedEvent> {
		public void eventOccured(UserAddedEvent event) {
			table.refresh();
		}
	}

	private final class UserUpdatedListener implements EventListener<UserUpdatedEvent> {

		public void eventOccured(UserUpdatedEvent event) {
			table.refresh();
		}
	}

	private final class UserDeletedListener implements EventListener<UserDeletedEvent> {
		public void eventOccured(UserDeletedEvent event) {
			table.refresh();
		}
	}

	public AnchorChangeListener getAnchorChangeListener() {
		return new AnchorChangeListener() {
			public void anchorChanged(String anchor) {
				bookmarkFeature.anchorChanged(anchor);
			}

		};
	}

}
