package com.tandbergtv.neptune.ui.framework.client;

import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_CENTER;
import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_RIGHT;
import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_TOP;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.framework.client.i18n.NeptuneConstants;
import com.tandbergtv.neptune.ui.framework.client.impl.EventManager;
import com.tandbergtv.neptune.ui.framework.client.impl.NeptuneSecurityImpl;
import com.tandbergtv.neptune.ui.framework.client.impl.ServiceLoaderImpl;
import com.tandbergtv.neptune.ui.home.client.HomeTabPanel;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.application.PortletFactory;
import com.tandbergtv.neptune.widgettoolkit.client.application.ServiceLoader;
import com.tandbergtv.neptune.widgettoolkit.client.application.WidgetFactory;
import com.tandbergtv.neptune.widgettoolkit.client.component.Component;
import com.tandbergtv.neptune.widgettoolkit.client.menu.ActionMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.menu.GroupMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.menu.MenuItemBase;
import com.tandbergtv.neptune.widgettoolkit.client.menu.MenuItemProxy;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.security.InsufficientRolesException;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.security.UserNotLoggedInException;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.PasswordTextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.TreeTabPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.TreeTabPanel.MenuNode;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.TreeTabPanel.TabNode;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DockContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class NeptuneApplicationImpl extends NeptuneApplication {
	private static final String STATE_INITIAL = "Home";

	private final List<Component> components;
	private final EventManager eventManager;
	private ServiceLoader serviceLoader;
	private final Map<Class<? extends Throwable>, Runnable> throwableToHandlerMap;
	private final DockContainer mainContainer;
	private TextBox usernameTextBox;
	private final DockContainer imageContainer;
	private boolean loginPageShown;

	private NeptuneApplicationImpl(List<Component> components) {
				
		imageContainer = new DockContainer();
	
		
		ImageWidget imageWPLogo = new ImageWidget("images/hdr_wplogo.png");
		ImageWidget imageGradient = new ImageWidget("images/hdr_gradient.png");
		ImageWidget imageTTVLogo = new ImageWidget("images/hdr_tandberglogo.png");
		
		imageWPLogo.setStyleName("wplogo");
		imageGradient.setStyleName("headergradient");
		imageTTVLogo.setStyleName("tandberglogo");
		
		imageContainer.add(imageWPLogo, DockContainer.WEST);
		imageContainer.add(imageGradient, DockContainer.CENTER);
		imageContainer.add(imageTTVLogo, DockContainer.EAST);
		


		this.components = components;
		this.eventManager = new EventManager();

		// initialize main panel
		mainContainer = new DockContainer();
		mainContainer.setVerticalAlignment(ALIGN_TOP);
		mainContainer.setHorizontalAlignment(ALIGN_CENTER);
		mainContainer.setHeight("100%");
		mainContainer.setStyleName("neptune-main-container");
		roleCheck(mainContainer);

		throwableToHandlerMap = new HashMap<Class<? extends Throwable>, Runnable>();
		throwableToHandlerMap.put(UserNotLoggedInException.class, new Runnable() {
			public void run() {
				if(!isLoginPageShown()) {
					mainContainer.clear();
					populateWithLoginForm(mainContainer);
					usernameTextBox.setFocus(true);
				}
			}
		});

		throwableToHandlerMap.put(InsufficientRolesException.class, new Runnable() {
			public void run() {
				Window.alert("You do not have the necessary permission for this operation");
			}
		});
	}

	public static void init(List<Component> components) {
		NeptuneApplication.INSTANCE = new NeptuneApplicationImpl(components);		 
	}

	@Override
	public Widget getMainContainer() {
		return mainContainer;
	}

	@Override
	public ServiceLoader getServiceLoader() {
		return serviceLoader;
	}
	
	private synchronized boolean isLoginPageShown() {
		return loginPageShown;
	}

	private synchronized void setLoginPageShown(boolean loginPageShown) {
		this.loginPageShown = loginPageShown;
	}

	private void roleCheck(final DockContainer mainContainer) {
		LoginUiServiceAsync async = GWT.create(LoginUiService.class);
		async.getCredentials(new AsyncCallback<Credentials>() {
			public void onFailure(Throwable caught) {
				if(!isLoginPageShown()) {
					populateWithLoginForm(mainContainer);
					usernameTextBox.setFocus(true);
				}
			}

			public void onSuccess(Credentials result) {
				NeptuneSecurity security = new NeptuneSecurityImpl(result.getUsername(), result.getRoles());
				loadServerInfo(mainContainer, security);
				DOM.createIFrame();
			}
		});
	}

	private void populateWithLoginForm(final DockContainer mainContainer) {
		setLoginPageShown(true);
		
		final TextBoxWidget usernameWidget = new TextBoxWidget();
		usernameTextBox = usernameWidget;

		final PasswordTextBoxWidget passwordWidget = new PasswordTextBoxWidget();

		NeptuneConstants constants = GWT.create(NeptuneConstants.class);

		final ButtonWidget loginButton = new ButtonWidget(constants.login(), new ClickHandler() {
			public void onClick(ClickEvent event) {
				LoginUiServiceAsync async = GWT.create(LoginUiService.class);
				async.login(usernameWidget.getText(), passwordWidget.getText(), new AsyncCallback<List<String>>() {

					public void onFailure(Throwable caught) {
						Window.alert("Invalid username/password");
					}

					public void onSuccess(List<String> result) {
						setLoginPageShown(false);
						NeptuneSecurity security = new NeptuneSecurityImpl(usernameWidget.getText(), result);
						mainContainer.clear();
						loadServerInfo(mainContainer, security);
					}
				});
			}
		});
		
		//Pressing "Enter" key on password field should trigger login button "Click"
		passwordWidget.addKeyPressHandler(new KeyPressHandler() {
			@Override
			public void onKeyPress(KeyPressEvent event) {
				if(event.getCharCode() == KeyCodes.KEY_ENTER) {
					loginButton.click();
				}
			}
		});
		
		loginButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);

		FormContainer formContainer = new FormContainer(ALIGN_RIGHT);
		formContainer.addRow(new LabelWidget(constants.username()), usernameWidget);
		formContainer.addRow(new LabelWidget(constants.password()), passwordWidget);
		formContainer.addButton(loginButton);

		mainContainer.add(imageContainer, DockContainer.NORTH);
		HorizontalContainer decoratedLoginForm = new HorizontalContainer();
		decoratedLoginForm.add(new ImageWidget("images/lock.gif"));
		decoratedLoginForm.add(formContainer);
		mainContainer.add(decoratedLoginForm, DockContainer.CENTER);
		mainContainer.setCellHorizontalAlignment(decoratedLoginForm, ALIGN_CENTER);

		usernameTextBox.setFocus(true);
	}

	private void loadServerInfo(final DockContainer mainContainer,
			final NeptuneSecurity security) {
		ServerInfoProviderServiceAsync async = GWT
				.create(ServerInfoProviderService.class);
		async.getInfo(new AsyncCallback<ServerInfo>() {

			public void onFailure(Throwable caught) {
				Window.alert(caught.getLocalizedMessage());
			}

			public void onSuccess(ServerInfo result) {
				populateWithTabPage(mainContainer, security, result);
			}
		});
	}
	
	private void populateWithTabPage(final DockContainer mainContainer,
			final NeptuneSecurity security, ServerInfo info) {
		serviceLoader = new ServiceLoaderImpl(components, security);

		// initialize all the components
		for (Component component : components)
			component.init(eventManager, eventManager, serviceLoader, info.getInfoMap());
		
		/* if there were any errors while preparing for this component  */
		if (info.getErrorMsgs() != null && !info.getErrorMsgs().isEmpty()) {
			StringBuilder sb = new StringBuilder();
			for (String e : info.getErrorMsgs()) {
				if (sb.length() > 0) {
					sb.append(",");
				}
				sb.append(e);
			}
			Window.alert(sb.toString());
		}

		// this listener set the current history token to whatever is provided by the TreeTabPanel
		final TreeTabPanel.AnchorChangeListener frameworkAnchorChangeListener = new TreeTabPanel.AnchorChangeListener() {
			public void anchorChanged(String anchor) {
				History.newItem(anchor, false);
			}
		};
		
		final TreeTabPanel treeTabPanel = new TreeTabPanel(frameworkAnchorChangeListener, security.getUsername());
		treeTabPanel.addListener(new TreeTabPanel.Listener() {
			public void tabChanged(TreeTabPanel.TabNode tabNode) {
				History.newItem(tabNode.getToken(), false);
			}

		});

		final TabHistoryListener tabHistoryListener = new TabHistoryListener(treeTabPanel);
		final HandlerRegistration registration = History.addValueChangeHandler(tabHistoryListener);

		final String token = History.getToken().length() == 0 ? STATE_INITIAL : History.getToken();

		// create the tab producer
		TreeTabPanelNodeProducer producer = new TreeTabPanelNodeProducer(treeTabPanel.getRootMenu());

		// add all component portlets
		final List<PortletFactory> portletFactories = new ArrayList<PortletFactory>();
		for (Component component : components) {
			List<PortletFactory> componentPortletFactories = component.listPortlets(security);
			if(componentPortletFactories != null)
				portletFactories.addAll(componentPortletFactories);
		}

		// create tabs
		List<MenuItemBase> tabs = new ArrayList<MenuItemBase>();

		// add home tab
		WidgetMenuItem homeTabFactory = new WidgetMenuItem("Home", new WidgetFactory() {
			public Widget getInstance() {
				return new HomeTabPanel(security, portletFactories);
			}

			public void release(Widget widget) {
			}
		});
		
		tabs.add(homeTabFactory);

		// add component tabs
		for (Component component : components) {
			tabs.addAll(component.listTabs(security));
		}
		
		Collections.sort(tabs, new Comparator<MenuItemBase>() {
			public int compare(MenuItemBase o1, MenuItemBase o2) {
				if (o1 instanceof MenuItemProxy && !(o2 instanceof MenuItemProxy))
					return 1;
				return 0;
			}
		});

		// add logout action
		tabs.add(new ActionMenuItem("Logout", new Runnable() {
			public void run() {
				((LoginUiServiceAsync) GWT.create(LoginUiService.class)).logout(new AsyncCallback<Void>() {
					public void onFailure(Throwable caught) {
						Window.alert("Logout failed. Reason: " + caught.getMessage());
					}

					public void onSuccess(Void result) {
						// clean up before moving away from this
						// page
						registration.removeHandler();
						mainContainer.clear();
						populateWithLoginForm(mainContainer);
						usernameTextBox.setFocus(true);
					}
				});
			}
		}));

		// get application specific panels
		for (MenuItemBase tab : tabs)
			tab.accept(producer);

		treeTabPanel.setActiveTab(token);

		VerticalContainer tabContainer = new VerticalContainer();
		tabContainer.add(imageContainer);
		tabContainer.setCellHorizontalAlignment(imageContainer, VerticalContainer.ALIGN_LEFT);
		tabContainer.add(treeTabPanel);
		treeTabPanel.doLayout();
		mainContainer.add(tabContainer, DockContainer.CENTER);
	}

	private final class TreeTabPanelNodeProducer implements MenuItemBase.Visitor {
		private final class ForwardingAnchorChangeListener implements TreeTabPanel.AnchorChangeListener {
			private final WidgetMenuItem menuitem;

			private ForwardingAnchorChangeListener(WidgetMenuItem panel) {
				this.menuitem = panel;
			}

			public void anchorChanged(String anchor) {
				menuitem.getAnchorChangeListener().anchorChanged(anchor);
			}
		}

		private final TreeTabPanel.MenuNode menuNode;

		private TreeTabPanelNodeProducer(TreeTabPanel.MenuNode menuNode) {
			this.menuNode = menuNode;
		}

		public void visit(WidgetMenuItem menuitem) {
			final TabNode tabNode = menuNode.addTabNode(menuitem.getName(), menuitem.getWidgetFactory(),
					new ForwardingAnchorChangeListener(menuitem));
			menuitem.setReverseAnchorChangeListener(new WidgetMenuItem.AnchorChangeListener() {
				public void anchorChanged(String anchor) {
					tabNode.getReverseAnchorChangeListener().anchorChanged(anchor);
				}
			});
		}

		public void visit(GroupMenuItem menu) {
			TreeTabPanel.MenuNode subMenuNode = menuNode.addMenuNode(menu.getName());
			
			for (MenuItemBase menuitem : menu.getSubPanels())
				menuitem.accept(new TreeTabPanelNodeProducer(subMenuNode));
		}

		public void visit(ActionMenuItem action) {
			menuNode.addActionNode(action.getName(), action.getRunnable());
		}

		@Override
		public void visit(MenuItemProxy proxy) {
			/* This menuNode is guaranteed to be the root */
			MenuNode node = menuNode.getMenuNode(proxy.getName());
			
			/* Mighty odd, someone is trying to attach to a non-existent parent */
			if (node == null)
				node = menuNode.addMenuNode(proxy.getName()); 
			
			for (MenuItemBase item : proxy.getItems())
				item.accept(new TreeTabPanelNodeProducer(node));
		}
	}

	private final class TabHistoryListener implements ValueChangeHandler<String> {
		private final TreeTabPanel tabPanel;

		private TabHistoryListener(TreeTabPanel tabPanel) {
			this.tabPanel = tabPanel;
		}

		@Override
		public void onValueChange(ValueChangeEvent<String> event) {
			tabPanel.setActiveTab(event.getValue());
		}
	}

	@Override
	public boolean handleThrowable(Throwable caught) {
		Runnable runnable = throwableToHandlerMap.get(caught.getClass());
		if (runnable == null)
			return false;

		runnable.run();
		return true;
	}

	@SuppressWarnings("unchecked")
    @Override
	public <T extends Component> T getComponent(Class<T> componentClass) {
	    if (this.components == null)
	    	return null;
	    
	    for (Component component : this.components) {
	    	if (component.getClass().equals(componentClass)) {
	    		return (T) component;
	    	}
	    }
	    
	    return null;
	}
}
