package com.tandbergtv.neptune.ui.home.client;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.CloseHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.tandbergtv.neptune.ui.realm.client.tab.userpref.UiUserPreference;
import com.tandbergtv.neptune.ui.realm.client.tab.userpref.UserPreferenceService;
import com.tandbergtv.neptune.ui.realm.client.tab.userpref.UserPreferenceServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.application.PortletFactory;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public class HomeTabPanel extends Composite {

	public static final String PORTLET_ORDER_TYPE = "portlet.order";
	public static final String HOME_COMPONENT = "Home";

	/* State */
	private final List<PortletRecord> allPortletRecords;

	/* Widgets and Handlers */
	private final SimpleContainer mainContainer;
	private VerticalContainer contentsPanel;
	private LabelWidget addPortletsLink;
	private HomeTabPortal portal;
	private BusyIndicator busyIndicator = new BusyIndicator();
	private final PortletEventHandler eventHandler = new PortletEventHandler();
	private final List<HandlerRegistration> registrations;

	/* Services and messages */
	private final UserPreferenceServiceAsync service = GWT.create(UserPreferenceService.class);
	private HomeComponentConstants constants = GWT.create(HomeComponentConstants.class);

	/* Styles */
	private static final String STYLE_NAME = "home-HomeTabPanel";
	private static final String STYLE_ADD_PORTLETS_LINK = "home-HomeTabPanel-addPortletsLabel";

	/**
	 * Constructor
	 * 
	 * @param portletFactories the list of all portlet factories
	 */
	public HomeTabPanel(List<PortletFactory> portletFactories) {
		this.allPortletRecords = getAllPortletRecords(portletFactories);
		this.registrations = new ArrayList<HandlerRegistration>();

		mainContainer = new SimpleContainer();
		mainContainer.addStyleName(STYLE_NAME);
		initWidget(mainContainer);

		initialize();
	}

	/*
	 * Build a list of all portlet records for each portlet factory
	 */
	private List<PortletRecord> getAllPortletRecords(List<PortletFactory> factories) {
		List<PortletRecord> records = new ArrayList<PortletRecord>();
		if (factories != null) {
			for (PortletFactory factory : factories) {
				Portlet portlet = factory.getInstance();
				records.add(new PortletRecord(portlet, factory.getComponent()));
			}
		}

		return records;
	}

	/*
	 * Initialize the widgets
	 */
	private void initialize() {
		contentsPanel = new VerticalContainer();

		/* Build the 'Add Portlet' link */
		this.addPortletsLink = new LabelWidget(constants.addPortletsLink());
		this.addPortletsLink.addStyleName(STYLE_ADD_PORTLETS_LINK);
		this.addPortletsLink.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleAddPortletLinkClicked();
			}
		});
		this.addPortletsLink.setVisible(false);
		contentsPanel.add(this.addPortletsLink);
		contentsPanel.setCellHorizontalAlignment(this.addPortletsLink, VerticalPanel.ALIGN_RIGHT);

		/* Build the portal widget */
		this.portal = new HomeTabPortal(this);
		contentsPanel.add(this.portal);
		contentsPanel.setCellHorizontalAlignment(this.portal, VerticalPanel.ALIGN_CENTER);

		/* Set the contents of the main container */
		this.mainContainer.setWidget(contentsPanel);

		/* Load the user preferences, and decide which portlets to display */
		loadUserPreferences();
	}

	/**
	 * Shows additional selected portlets
	 * 
	 * @param records The records to show
	 */
	void showPortlets(List<PortletRecord> records) {
		/* Save the preferences */
		if (records != null && records.size() > 0) {
			for (PortletRecord record : records) {
				portal.showPortlet(record.getPortlet());
			}

			this.addPortletsLink.setVisible((getHiddenPortletRecords().size() != 0));
			saveUserPreferences();
		}

		this.mainContainer.setWidget(this.contentsPanel);
	}

	/**
	 * Handle the change in the ordering of the portlets
	 */
	void updatePortletOrder() {
		saveUserPreferences();
	}

	/*
	 * Add all close handlers for the portlets
	 */
	@Override
	protected void onLoad() {
		super.onLoad();

		/* Add close handler for each portlet */
		for (PortletRecord record : this.allPortletRecords) {
			HandlerRegistration registration = record.getPortlet().addCloseHandler(eventHandler);
			this.registrations.add(registration);
		}
	}

	/*
	 * Remove all close handlers for the portlets
	 */
	@Override
	protected void onUnload() {
		/* Remove all close handlers for each portlet */
		for (HandlerRegistration registration : this.registrations) {
			registration.removeHandler();
		}
		registrations.clear();

		super.onUnload();
	}

	// ========================================================================
	// ===================== PREFERENCE MANAGEMENT
	// ========================================================================

	/*
	 * Load the user preferences for portlet display
	 */
	private void loadUserPreferences() {
		busyIndicator.center();
		service.getPreferences(HOME_COMPONENT, PORTLET_ORDER_TYPE,
		        new NeptuneAsyncCallback<List<UiUserPreference>>() {

			        @Override
			        public void onNeptuneFailure(Throwable caught) {
				        busyIndicator.hide();
				        Window.alert(constants.errorLoadingState());
				        handleGetPreferencesSuccess(new ArrayList<UiUserPreference>());
			        }

			        @Override
			        public void onNeptuneSuccess(List<UiUserPreference> result) {
				        busyIndicator.hide();
				        handleGetPreferencesSuccess(result);
			        }
		        });
	}

	/*
	 * Show the portlets based on the user preferences
	 */
	private void handleGetPreferencesSuccess(List<UiUserPreference> userPreferences) {
		boolean canAddPortlets = false;
		int preferenceCount = (userPreferences != null) ? userPreferences.size() : 0;
		if (preferenceCount == 0) {
			int index = 0;
			for (PortletRecord record : this.allPortletRecords) {
				record.setPreferenceIndex(index++);
				portal.showPortlet(record.getPortlet(), record.getPreferenceIndex());
			}
		} else {
			List<PortletRecord> visibleRecords = getVisiblePortletRecords(userPreferences);
			canAddPortlets = (visibleRecords.size() != this.allPortletRecords.size());
			for (PortletRecord record : visibleRecords) {
				portal.showPortlet(record.getPortlet(), record.getPreferenceIndex());
			}
		}

		/* Show the add portlet link only if all portlets are not showing */
		if (this.addPortletsLink != null)
			this.addPortletsLink.setVisible(canAddPortlets);
	}

	/*
	 * Saves the current portal state as user preferences
	 */
	private void saveUserPreferences() {
		busyIndicator.center();
		service.updatePreferences(HOME_COMPONENT, PORTLET_ORDER_TYPE, createUserPreferences(),
		        new NeptuneAsyncCallback<Void>() {

			        @Override
			        public void onNeptuneFailure(Throwable caught) {
				        busyIndicator.hide();
				        Window.alert(constants.errorSavingState());
			        }

			        @Override
			        public void onNeptuneSuccess(Void result) {
				        busyIndicator.hide();
			        }
		        });
	}

	/*
	 * Create preferences for all portlets
	 */
	private List<UiUserPreference> createUserPreferences() {
		List<UiUserPreference> preferences = new ArrayList<UiUserPreference>();

		Map<Integer, Portlet> portletIndices = portal.getPortletIndexMap();
		preferences.add(createUserPreference("___CONTROL___", -1));
		for (Integer key : portletIndices.keySet()) {
			Portlet portlet = portletIndices.get(key);
			preferences.add(createUserPreference(portlet.getPortletId(), key));
		}

		return preferences;
	}

	/*
	 * Create a preference for a portlet
	 */
	private UiUserPreference createUserPreference(String key, int index) {
		UiUserPreference uiUserPreference = new UiUserPreference();
		uiUserPreference.setComponent(HOME_COMPONENT);
		uiUserPreference.setType(PORTLET_ORDER_TYPE);
		uiUserPreference.setKey(key);
		uiUserPreference.setValue(String.valueOf(index));

		return uiUserPreference;
	}

	// ========================================================================
	// ===================== RECORD SEARCHING
	// ========================================================================

	/*
	 * Get the list of visible portlet records in the order defined by the preferences
	 */
	private List<PortletRecord> getVisiblePortletRecords(List<UiUserPreference> userPreferences) {
		List<PortletRecord> visibleRecords = new ArrayList<PortletRecord>();
		for (PortletRecord record : this.allPortletRecords) {
			/* Find matching preference */
			UiUserPreference userPreference = null;
			for (UiUserPreference preference : userPreferences) {
				String portletId = record.getPortlet().getPortletId();
				if (portletId.equals(preference.getKey())) {
					userPreference = preference;
					break;
				}
			}

			if (userPreference != null) {
				String value = userPreference.getValue();
				Integer index = null;
				try {
					index = (value != null) ? Integer.parseInt(value) : null;
				} catch (Exception ignore) {
				}

				if (index != null) {
					record.setPreferenceIndex(index);
					visibleRecords.add(record);
				}
			}
		}

		/* Sort the list of portlet records */
		Collections.sort(visibleRecords, new Comparator<PortletRecord>() {
			@Override
			public int compare(PortletRecord o1, PortletRecord o2) {
				return o1.getPreferenceIndex().compareTo(o2.getPreferenceIndex());
			}
		});

		return visibleRecords;
	}

	/*
	 * Find all portlet records that are hidden
	 */
	private List<PortletRecord> getHiddenPortletRecords() {
		List<PortletRecord> records = new ArrayList<PortletRecord>();
		for (PortletRecord record : this.allPortletRecords) {
			if (!portal.isPortletShowing(record.getPortlet())) {
				records.add(record);
			}
		}

		return records;
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Show the 'Add Portlets' Panel widget as the contents
	 */
	private void handleAddPortletLinkClicked() {
		/* Get the list of hidden portlets */
		List<PortletRecord> hiddenRecords = getHiddenPortletRecords();
		if (hiddenRecords.size() == 0) {
			return;
		}

		/* Change the contents of the panel to the add portlet panel */
		mainContainer.setWidget(new AddPortletPanel(HomeTabPanel.this, hiddenRecords));
	}

	/*
	 * Handle Portlet close event
	 */
	private void handlePortletClosed(Portlet portlet) {
		/* Remove from list of portlets and fire detach event if required */
		if (portal.removePortlet(portlet)) {
			/* Save the preferences */
			saveUserPreferences();
			this.addPortletsLink.setVisible(true);
		}
	}

	/*
	 * Handler for the close handler
	 */
	private class PortletEventHandler implements CloseHandler<Portlet> {

		@Override
		public void onClose(CloseEvent<Portlet> event) {
			handlePortletClosed(event.getTarget());
		}
	}
}
