/*
 * Created on Jan 19, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.neptune.ui.framework.client.application;

import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_MIDDLE;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions;
import com.tandbergtv.neptune.ui.realm.client.RealmUIEvents;
import com.tandbergtv.neptune.ui.realm.client.networkelements.view.RedundancyStatusButton;
import com.tandbergtv.neptune.ui.realm.client.networkelements.view.RedundancyStatusStateHolder;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.ServerTimeLabel;

/**
 * The header widget used by the application
 * 
 * @author Vijay Silva
 */
public class HeaderWidget extends Composite {

	/* Styles */
	private static final String STYLE_NAME = "nfw-HeaderWidget";
	private static final String STYLE_DESCRIPTION_CONTAINER = "nfw-HeaderWidget-description";
	private static final String STYLE_LOGO = "nfw-HeaderWidget-logo";
	private static final String STYLE_TITLE_CONTAINER = "nfw-HeaderWidget-titleSection";
	private static final String STYLE_TITLE_CONTAINER_LABEL = "nfw-HeaderWidget-titleSectionLabel";
	private static final String STYLE_TITLE_LABEL = "nfw-HeaderWidget-titleLabel";
	private static final String STYLE_SUBTITLE_LABEL = "nfw-HeaderWidget-subTitleLabel";
	private static final String STYLE_USER_CONTAINER = "nfw-HeaderWidget-userSection";
	private static final String STYLE_DETAIL_CONTAINER = "nfw-HeaderWidget-detailSection";
	private static final String STYLE_DETAIL_SEPARATOR_LABEL = "nfw-HeaderWidget-detailSeparatorLabel";
	private static final String STYLE_DETAIL_SECTION_LABEL = "nfw-HeaderWidget-detailSectionLabel";
	private static final String STYLE_BUTTON_CONTAINER = "nfw-HeaderWidget-buttonPanel";
	private static final String STYLE_BUTTON_ABOUT = "nfw-HeaderWidget-aboutButton";
	private static final String STYLE_BUTTON_GUIDE = "nfw-HeaderWidget-guideButton";
	private static final String STYLE_BUTTON_LOGOUT = "nfw-HeaderWidget-logoutButton";

	/* Constants */
	private static final String IMAGE_LOGO_PATH = "neptune_framework/images/header-logo.png";

	private static final int TIMER_REFRESH_PERIOD = 60*1000;

	/* Widgets */
	private final ApplicationMessages messages;
	private HTMLWidget titleLabel, subTitleLabel, userLabel;
	private ServerTimeLabel serverTimeLabel;
	private RedundancyStatusButton clusterStatusButton;

	private HorizontalContainer buttonContainer;
	private ButtonWidget aboutButton, guideButton, logoutButton;

	/* Event Handlers */
	private List<HeaderWidget.EventHandler> handlers = new ArrayList<HeaderWidget.EventHandler>();

	/* The timer to trigger updates of the header widgets. */
	private Timer timer;
	/* Handler for the redundancy status widget event. */
	private HandlerRegistration redundancyStatusEventHandler;
	
	/**
	 * The header widget used by the application
	 */
	public HeaderWidget() {
		messages = GWT.create(ApplicationMessages.class);
		initializeWidget();
	}

	/*
	 * Initialize the widget contents / layout
	 */
	private void initializeWidget() {
		/* Main container */
		SimpleContainer mainContainer = new SimpleContainer();
		mainContainer.setStyleName(STYLE_NAME);

		/* The main table container */
		HorizontalContainer container = new HorizontalContainer();
		mainContainer.setWidget(container);

		/* Container for logo and application title */
		HorizontalContainer descriptionContainer = new HorizontalContainer();
		descriptionContainer.setStylePrimaryName(STYLE_DESCRIPTION_CONTAINER);
		container.add(descriptionContainer);

		/* Add the logo */
		ImageWidget image = new ImageWidget(IMAGE_LOGO_PATH);
		image.addStyleName(STYLE_LOGO);
		descriptionContainer.add(image);

		/* Add the application title container */
		VerticalContainer titleContainer = new VerticalContainer();
		titleContainer.setStylePrimaryName(STYLE_TITLE_CONTAINER);
		descriptionContainer.add(titleContainer);
		descriptionContainer.setCellVerticalAlignment(titleContainer, ALIGN_MIDDLE);

		titleLabel = new HTMLWidget();
		titleLabel.addStyleName(STYLE_TITLE_CONTAINER_LABEL);
		titleLabel.addStyleName(STYLE_TITLE_LABEL);
		titleContainer.add(titleLabel);

		subTitleLabel = new HTMLWidget();
		subTitleLabel.addStyleName(STYLE_TITLE_CONTAINER_LABEL);
		subTitleLabel.addStyleName(STYLE_SUBTITLE_LABEL);
		titleContainer.add(subTitleLabel);

		/* Add the user controls section */
		VerticalContainer userControlsContainer = new VerticalContainer();
		userControlsContainer.setStylePrimaryName(STYLE_USER_CONTAINER);
		container.add(userControlsContainer);

		/* Add the server time and user name information panel */
		HorizontalContainer detailContainer = new HorizontalContainer();
		detailContainer.setStylePrimaryName(STYLE_DETAIL_CONTAINER);
		userControlsContainer.add(detailContainer);

		/* Add the server time label */
		serverTimeLabel = new ServerTimeLabel();
		serverTimeLabel.addStyleName(STYLE_DETAIL_SECTION_LABEL);
		detailContainer.add(serverTimeLabel);

		/* Add a separator label */
		HTMLWidget separatorLabel = new HTMLWidget(messages.headerWidgetDetailSeparator());
		separatorLabel.addStyleName(STYLE_DETAIL_SECTION_LABEL);
		separatorLabel.addStyleName(STYLE_DETAIL_SEPARATOR_LABEL);
		detailContainer.add(separatorLabel);

		/* Add the user label */
		userLabel = new HTMLWidget();
		userLabel.addStyleName(STYLE_DETAIL_SECTION_LABEL);
		detailContainer.add(userLabel);

		/* Add the buttons */
		buttonContainer = new HorizontalContainer();
		buttonContainer.setStylePrimaryName(STYLE_BUTTON_CONTAINER);
		userControlsContainer.add(buttonContainer);
		
		clusterStatusButton = new RedundancyStatusButton();
		/* Should be visible only for users with HA_VIEW permission.
		 * Ideally shouldn't add the button to the page, but since ClientAuthorizationManager.isAuthorized 
		 * check doesn't work from here use the visible/invisible trick.*/
		buttonContainer.add(clusterStatusButton);
		
		aboutButton = new ButtonWidget();
		aboutButton.addStyleName(STYLE_BUTTON_ABOUT);
		aboutButton.setText(messages.headerWidgetAboutButton());
		aboutButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleAboutButtonClick(event);
			}
		});
		buttonContainer.add(aboutButton);

		guideButton = new ButtonWidget();
		guideButton.addStyleName(STYLE_BUTTON_GUIDE);
		guideButton.setText(messages.headerWidgetGuideButton());
		guideButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGuideButtonClick(event);
			}
		});
		buttonContainer.add(guideButton);

		logoutButton = new ButtonWidget();
		logoutButton.addStyleName(STYLE_BUTTON_LOGOUT);
		logoutButton.setText(messages.headerWidgetLogoutButton());
		logoutButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleLogoutButtonClick(event);
			}
		});
		buttonContainer.add(logoutButton);

		/* Initialize the timer */
		this.timer = new Timer() {
			@Override
			public void run() {
				fireTimeElapsedEvent();
			}
		};
		initWidget(mainContainer);
	}

	/**
	 * 		Notifies the timer listeners
	 */
	protected void fireTimeElapsedEvent() {
		// publishes the time elapsed event to the event bus
		RealmUIEvents.EVENT_BUS.fireEvent(new TimedEvent());
	}

	/*
	 * Start the timer when the widget is attached to the DOM to make server calls
	 */
	@Override
	protected void onLoad() {
		super.onLoad();

		clusterStatusButton.setVisible(false);
		/*Subscribes the redundancy status event handler to the timer event. 
		  so that every 60 seconds the redundancy status handler will be called */
		if (ClientAuthorizationManager.isAuthorized(RealmComponentPermissions.HA_VIEW)) {
			redundancyStatusEventHandler = RealmUIEvents.EVENT_BUS.addHandler(TimedEvent.TYPE, new RedundancyStatusStateHolder());
			RedundancyStatusStateHolder.refreshCurrentStatus(true);
		}
		fireTimeElapsedEvent();
		this.timer.scheduleRepeating(TIMER_REFRESH_PERIOD);
	}

	/*
	 * Stop the timer (ensuring that no server calls are made)
	 */
	@Override
	protected void onUnload() {
		/* Stop the timer. */
		this.timer.cancel();
		super.onUnload();
		if (redundancyStatusEventHandler != null) {
			redundancyStatusEventHandler.removeHandler();
			redundancyStatusEventHandler = null;
		}
	}
	
	// ========================================================================
	// ===================== API
	// ========================================================================

	/**
	 * Set the pattern to use for the server time display
	 * 
	 * @param pattern The date time format pattern to use when displaying the server time
	 */
	public void setServerTimeDisplayPattern(String pattern) {
		serverTimeLabel.setDisplayPattern(pattern);
	}

	/**
	 * Set the Application Title to display in the header widget
	 * 
	 * @param title The application title
	 */
	public void setApplicationTitle(String title) {
		if (title != null && title.trim().length() > 0)
			title = messages.headerWidgetTitleLabel(title);

		titleLabel.setHTML(title);
	}

	/**
	 * Set the Application sub-title to display in the header widget
	 * 
	 * @param subtitle The application sub-title
	 */
	public void setApplicationSubtitle(String subtitle) {
		if (subtitle != null && subtitle.trim().length() > 0)
			subtitle = messages.headerWidgetSubtitleLabel(subtitle);

		subTitleLabel.setHTML(subtitle);
	}

	/**
	 * Set the currently logged-in user in the header widget
	 * 
	 * @param userName the user name for the currently logged in user
	 */
	public void setCurrentUser(String userName) {
		if (userName != null && userName.trim().length() > 0)
			userName = messages.headerWidgetCurrentUserLabel(userName);

		userLabel.setHTML(userName);
	}

	/**
	 * Remove the currently logged-in user.
	 */
	public void clearCurrentUser() {
		userLabel.setHTML("");
	}

	/**
	 * Determine if the logout button is visible
	 * 
	 * @return true if visible, false otherwise
	 */
	public boolean isLogoutButtonVisible() {
		return logoutButton.isVisible();
	}

	/**
	 * Set the logout button visible / invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setLogoutButtonVisible(boolean visible) {
		logoutButton.setVisible(visible);
	}

	/**
	 * Determine if the user guide button is visible
	 * 
	 * @return true if visible, false otherwise
	 */
	public boolean isUserGuideButtonVisible() {
		return guideButton.isVisible();
	}

	/**
	 * Set the user guide button visible / invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setUserGuideButtonVisible(boolean visible) {
		guideButton.setVisible(visible);
	}

	/**
	 * Determine if the user guide button is visible
	 * 
	 * @return true if visible, false otherwise
	 */
	public boolean isAboutButtonVisible() {
		return guideButton.isVisible();
	}

	/**
	 * Set the about button visible / invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setAboutButtonVisible(boolean visible) {
		aboutButton.setVisible(visible);
	}

	/**
	 * Set all the buttons to be visible or invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setButtonsVisible(boolean visible) {
		setLogoutButtonVisible(visible);
		setUserGuideButtonVisible(visible);
		setAboutButtonVisible(visible);
	}

	/**
	 * Add an event handler
	 * 
	 * @param handler The event handler
	 */
	public void addEventHandler(HeaderWidget.EventHandler handler) {
		if (handler != null && !handlers.contains(handler))
			this.handlers.add(handler);
	}

	/**
	 * Remove a previously registered event handler
	 * 
	 * @param handler The event handler
	 */
	public void removeEventHandler(HeaderWidget.EventHandler handler) {
		this.handlers.remove(handler);
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the 'About' button click
	 */
	private void handleAboutButtonClick(ClickEvent event) {
		for (HeaderWidget.EventHandler handler : handlers)
			handler.onAboutButtonClick(new HeaderWidget.Event(this));
	}

	/*
	 * Handle the 'Guide' button click
	 */
	private void handleGuideButtonClick(ClickEvent event) {
		for (HeaderWidget.EventHandler handler : handlers)
			handler.onUserGuideButtonClick(new HeaderWidget.Event(this));
	}

	/*
	 * Handle the 'Log Out' button click
	 */
	private void handleLogoutButtonClick(ClickEvent event) {
		for (HeaderWidget.EventHandler handler : handlers)
			handler.onLogoutButtonClick(new HeaderWidget.Event(this));
	}

	// ========================================================================
	// ===================== WIDGET EVENT
	// ========================================================================

	/**
	 * The event fired by this widget
	 */
	public static class Event {

		private HeaderWidget source;

		/**
		 * Constructor
		 * 
		 * @param source The event source
		 */
		public Event(HeaderWidget source) {
			this.source = source;
		}

		/**
		 * Get the event source
		 * 
		 * @return The source
		 */
		public HeaderWidget getSource() {
			return source;
		}
	}

	// ========================================================================
	// ===================== WIDGET EVENT HANDLER
	// ========================================================================

	/**
	 * Event handler for the events fired by the Header Widget
	 * 
	 * @author Vijay Silva
	 */
	public interface EventHandler {

		/**
		 * Triggered by the 'About' button is clicked on the header widget
		 * 
		 * @param event The event
		 */
		void onAboutButtonClick(HeaderWidget.Event event);

		/**
		 * Triggered by the 'User Guide' button is clicked on the header widget
		 * 
		 * @param event The event
		 */
		void onUserGuideButtonClick(HeaderWidget.Event event);

		/**
		 * Triggered by the 'Log Out' button is clicked on the header widget
		 * 
		 * @param event The event
		 */
		void onLogoutButtonClick(HeaderWidget.Event event);
	}
}
