/*
 * Created on Jan 3, 2012
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view;

import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_CREATE;
import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_MODIFY;
import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_VIEW;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmMessages;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUser;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserConfiguration;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserKey;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiService;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiServiceAsync;
import com.tandbergtv.neptune.ui.realm.client.tab.user.view.external.ExternalUserRecord;
import com.tandbergtv.neptune.ui.realm.client.tab.user.view.external.ExternalUserView;
import com.tandbergtv.neptune.ui.realm.client.tab.user.view.external.ExternalUserViewEvent;
import com.tandbergtv.neptune.ui.realm.client.tab.user.view.external.ExternalUserViewEventHandler;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.LazyView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitHandler;

/**
 * View for the users
 * 
 * @author Vijay Silva
 */
public class UserView extends LazyView {

	private VerticalPanel mainPanel;
	private HeaderPanel headerPanel;
	private SimplePanel widgetPanel;
	private UserTableView tableView;
	private BusyIndicator busyIndicator = new BusyIndicator(this);
	private UserUiServiceAsync service;
	private RealmMessages messages;
	private RealmConstants constants;
	private UiUserConfiguration configuration;

	/* Styles / constants */
	private static final String STYLE_NAME = "realm-UserView";
	private static final String STYLE_LIST_VIEW = "listView";
	private static final String STYLE_DETAIL_VIEW = "detailView";
	private static final String STYLE_HEADER_PANEL = STYLE_NAME + "-headerPanel";
	private static final String STYLE_WIDGET_PANEL = STYLE_NAME + "-widgetPanel";
	private static final String CREATE_VIEW_TOKEN = "Create";
	private static final String EDIT_VIEW_TOKEN = "Edit";
	private static final String CREATE_EXTERNAL_VIEW_TOKEN = "CreateExternal";
	private static final String VIEW_TOKEN_SEPARATOR = "?";
	private static final String RECORD_KEY_TOKEN = "id";
	private static final String USERNAME_TOKEN = "userName";
	private ResizableContainer container;
	private UserForm form;
	private ExternalUserView view;
	

	/**
	 * Constructor
	 */
	public UserView() {
		addStyleName(STYLE_NAME);
	}

	/**
	 * Reset the view to the initial state. Does not fire an anchor change event.
	 */
	public void reset() {
		/* Still lazy, do nothing */
		if (getWidget() == null)
			return;

		/* Reset the displayed widget to a blank widget */
		setWidgetInPanel(null);
		if (tableView != null)
			tableView.reset();
		updateButtonForListView();
		container.reset();
		container.updateSize();
	}

	/**
	 * Show table view, keeping the current state of the features. The records are re-fetched from
	 * the server. Does not fire an anchor change event.
	 * 
	 * @param showFirstPage Flag indicating the table should change to show the first page or
	 *        continue to show the current page of records
	 */
	public void showTable(final boolean showFirstPage) {
		/* Ensure the widget is not lazy */
		ensureWidget();

		/* Hide the error message and error panel */
		hideErrorMessage();
		hideHeaderPanel();

		/* Get the user configuration */
		getUserConfiguration(new NeptuneAsyncCallback<UiUserConfiguration>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Do nothing, the default implementation handles this */
			}

			@Override
			public void onNeptuneSuccess(UiUserConfiguration result) {
				/* Got the configuration, update the table */
				showTableView();
				tableView.refresh(showFirstPage);
			}
		});
	}

	/**
	 * Check if this view is showing the table of external users. If the view is lazy and is not
	 * initialized, this method will return true
	 * 
	 * @return true if the table view is showing, false otherwise
	 */
	public boolean isShowingTable() {
		Widget widget = getWidgetInPanel();
		return (widget == null || widget == tableView);
	}

	/*
	 * Create the widget
	 */
	@Override
	protected Widget createWidget() {
		mainPanel = new VerticalPanel();
		mainPanel.setStyleName(STYLE_LIST_VIEW);

		/* At the very top is the header panel */
		headerPanel = new HeaderPanel("");
		headerPanel.setVisible(false);
		headerPanel.addStyleName(STYLE_HEADER_PANEL);
		mainPanel.add(headerPanel);

		/* Add the widget panel */
		widgetPanel = new SimpleContainer();
		widgetPanel.addStyleName(STYLE_WIDGET_PANEL);
		mainPanel.add(widgetPanel);

		container = new ResizableContainer(widgetPanel, "Users", null);
		mainPanel.add(container);
		container.addHeightOffsettingWidget(headerPanel);
		return mainPanel;
	}

	// ========================================================================
	// ===================== INTERNAL STUFF
	// ========================================================================

	/*
	 * If the widget is attached to the DOM, show the busy indicator
	 */
	private void showBusyIndicator() {
		busyIndicator.center();
	}

	/*
	 * Hide the busy indicator
	 */
	private void hideBusyIndicator() {
		busyIndicator.hide();
	}

	/*
	 * Show the error message in the view
	 */
	private void showErrorMessage(String message) {
		if (message == null)
			message = "";
		container.setErrorMessage(message);
	}

	/*
	 * Clear the error message displayed in the view
	 */
	private void hideErrorMessage() {
		container.reset();
	}

	/*
	 * Show the header panel
	 */
	private void showHeaderPanel(String heading, String styleName) {
		headerPanel.setLabel(heading);
		headerPanel.setVisible(true);
		mainPanel.setStyleName(styleName);
	}

	/*
	 * Hide the header panel
	 */
	private void hideHeaderPanel() {
		headerPanel.setLabel("");
		headerPanel.setVisible(false);
		mainPanel.setStyleName(STYLE_LIST_VIEW);
	}

	/*
	 * Get the user service
	 */
	private UserUiServiceAsync getService() {
		if (service == null)
			service = GWT.create(UserUiService.class);

		return service;
	}

	/*
	 * Get the constants
	 */
	private RealmConstants getConstants() {
		if (constants == null)
			constants = GWT.create(RealmConstants.class);

		return constants;
	}

	/*
	 * Get the messages
	 */
	private RealmMessages getMessages() {
		if (messages == null)
			messages = GWT.create(RealmMessages.class);

		return messages;
	}

	// ========================================================================
	// ===================== USER CONFIGURATION LOADING
	// ========================================================================

	/*
	 * Load the user configuration if not already cached and build the widgets that depend on it
	 */
	private void getUserConfiguration(final AsyncCallback<UiUserConfiguration> callback) {
		/* Get the user configuration from cache first */
		if (this.configuration != null) {
			callback.onSuccess(this.configuration);
			return;
		}

		/* Load from the server */
		showBusyIndicator();
		getService().getUserConfiguration(new NeptuneAsyncCallback<UiUserConfiguration>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Hide the busy indicator */
				hideBusyIndicator();

				/* Failed to get configuration */
				String errorMessage = caught.getLocalizedMessage();
				showErrorMessage(getMessages().getUserConfigurationFailure(errorMessage));
				setWidgetInPanel(null);
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UiUserConfiguration result) {
				/* Hide the busy indicator */
				hideBusyIndicator();

				/* Update the cached configuration */
				configuration = result;

				/* Build the table view */
				tableView = new UserTableView(result, busyIndicator,container);
				TableEventHandler eventHandler = new TableEventHandler();
				tableView.addRecordClickHandler(eventHandler);
				tableView.addUserTableViewEventHandler(eventHandler);
				tableView.addViewAnchorChangeHandler(eventHandler);
				setWidgetInPanel(tableView);

				callback.onSuccess(configuration);
			}
		});
	}

	// ========================================================================
	// ===================== ANCHOR MANAGEMENT
	// ========================================================================

	/*
	 * Get the widget anchor
	 */
	@Override
	public String getAnchor() {
		String anchor = null;
		Widget widget = getWidgetInPanel();

		if (tableView != null && widget == tableView) {
			/* Showing the list view, get table anchor */
			anchor = tableView.getAnchor();
		} else if (widget instanceof UserForm) {
			/* This is either the create or edit screen */
			UserForm form = (UserForm) widget;
			UiUserKey key = form.getUser().getKey();
			if (key.isValueAssigned()) {
				Map<String, String> tokens = new HashMap<String, String>();
				tokens.put(RECORD_KEY_TOKEN, key.toString());
				anchor = EDIT_VIEW_TOKEN + VIEW_TOKEN_SEPARATOR
				        + new AnchorTokenizer().buildAnchor(tokens);
			} else {
				anchor = CREATE_VIEW_TOKEN;
			}
		} else if (widget instanceof ExternalUserView) {
			/* Get the anchor from the external user view */
			ExternalUserView view = (ExternalUserView) widget;
			anchor = CREATE_EXTERNAL_VIEW_TOKEN + VIEW_TOKEN_SEPARATOR + view.getAnchor();
			return anchor;
		} else {
			anchor = "";
		}

		return anchor;
	}

	/*
	 * Set the widget anchor
	 */
	@Override
	protected void setWidgetAnchor(final String anchor) {
		/* Reset the widget state / view */
		reset();

		/* Load the user configuration data */
		getUserConfiguration(new NeptuneAsyncCallback<UiUserConfiguration>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Do nothing, the default implementation handles this */
			}

			@Override
			public void onNeptuneSuccess(UiUserConfiguration result) {
				/* Got the configuration, build the view with the anchor */
				showWidgetForAnchor(anchor);
			}
		});
	}

	/*
	 * Show the widget based on the anchor (the configuration is loaded
	 */
	private void showWidgetForAnchor(String anchor) {
		
		/* Test the view token in the anchor */
		if (isAnchorForView(CREATE_EXTERNAL_VIEW_TOKEN, anchor)) {
			clearButtons();
			/* Create External anchor */
			showWidgetForCreateExternalAnchor(anchor);
		} else if (isAnchorForView(CREATE_VIEW_TOKEN, anchor)) {
			clearButtons();
			/* Create Anchor */
			showWidgetForCreateAnchor(anchor);
		} else if (isAnchorForView(EDIT_VIEW_TOKEN, anchor)) {
			clearButtons();
			showWidgetForEditAnchor(anchor);
		} else {
			/* Default to table view */
			showTableView();
			tableView.setAnchor(anchor);
		}
	}

	/*
	 * Determine if the anchor is for a particular view
	 */
	private boolean isAnchorForView(String viewToken, String anchor) {
		return (anchor != null && (anchor.equals(viewToken) || anchor.startsWith(viewToken
		        + VIEW_TOKEN_SEPARATOR)));
	}

	/*
	 * Show the widget for the 'CreateExternal' anchor
	 */
	private void showWidgetForCreateExternalAnchor(String anchor) {
		/* Creating external user - verify permissions */
		if (!verifyAuthorization(USER_VIEW, USER_CREATE)) {
			return;
		}

		/* Verify that the mode is correct */
		if (!configuration.isExternalMode()) {
			showErrorMessage(getConstants().createExternalUserInvalidModeError());
			return;
		}

		/* Show the widget */
		anchor = getAnchorForView(CREATE_EXTERNAL_VIEW_TOKEN, anchor);
		ExternalUserView view = showExternalUserView();
		view.setAnchor(anchor);
	}

	/*
	 * Show the widget for the 'Create' anchor
	 */
	private void showWidgetForCreateAnchor(String anchor) {
		/* Creating user - verify permissions */
		if (!verifyAuthorization(USER_VIEW, USER_CREATE))
			return;

		/* Verify that the mode is correct */
		if (configuration.isExternalMode() && !configuration.isCreateInternalUserAllowed()) {
			showErrorMessage(getConstants().createUserInvalidModeError());
			return;
		}

		/* Show the widget */
		showUserFormForCreate();
	}

	/*
	 * Show the widget for the 'Edit' anchor
	 */
	private void showWidgetForEditAnchor(String anchor) {
		/* Editing / viewing user - verify permissions */
		if (!verifyAuthorization(USER_VIEW))
			return;

		/* Parse the anchor */
		anchor = getAnchorForView(EDIT_VIEW_TOKEN, anchor);
		Map<String, String> tokens = new AnchorTokenizer().parseAnchor(anchor);
		if (tokens.containsKey(RECORD_KEY_TOKEN)) {
			try {
				String keyValue = tokens.get(RECORD_KEY_TOKEN);
				UiUserKey key = UiUserKey.parse(keyValue);
				showUserByKey(key);
			} catch (RuntimeException e) {
				showErrorMessage(getConstants().viewUserInvalidAnchorError());
			}
		} else if (tokens.containsKey(USERNAME_TOKEN)) {
			String userName = tokens.get(USERNAME_TOKEN);
			showUserByUserName(userName);
		} else {
			showErrorMessage(getConstants().viewUserInvalidAnchorError());
		}
	}

	/*
	 * Show the user given the key
	 */
	private void showUserByKey(final UiUserKey key) {
		showBusyIndicator();
		getService().getUser(key, new NeptuneAsyncCallback<UiUser>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				hideBusyIndicator();
				String idValue = (key.getId() != null) ? key.getId().toString() : "";
				String errorMessage = caught.getLocalizedMessage();
				showErrorMessage(getMessages().getUserFailure(idValue, errorMessage));
			}

			@Override
			public void onNeptuneSuccess(UiUser result) {
				hideBusyIndicator();
				showUserFormForEdit(result);
			}
		});
	}

	/*
	 * Show the user given the user name
	 */
	private void showUserByUserName(final String userName) {
		showBusyIndicator();
		getService().getUser(userName, new NeptuneAsyncCallback<UiUser>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				hideBusyIndicator();
				String errorMessage = caught.getLocalizedMessage();
				showErrorMessage(getMessages().getUserByNameFailure(userName, errorMessage));
			}

			@Override
			public void onNeptuneSuccess(UiUser result) {
				hideBusyIndicator();
				showUserFormForEdit(result);
			}
		});
	}

	/*
	 * Verify that the user is authorized based on permissions, updating the error message if not
	 * authorized.
	 */
	private boolean verifyAuthorization(String... permissions) {
		/* Verify permission to access this view */
		if (!ClientAuthorizationManager.isAuthorized(permissions)) {
			showErrorMessage(getConstants().unauthorizedAccessError());
			return false;
		}

		return true;
	}

	/*
	 * Get the anchor without the view token
	 */
	private String getAnchorForView(String viewToken, String anchor) {
		anchor = anchor.substring(viewToken.length());
		if (anchor.startsWith(VIEW_TOKEN_SEPARATOR))
			anchor = anchor.substring(VIEW_TOKEN_SEPARATOR.length());

		return anchor;
	}

	/*
	 * Fire the view anchor change event
	 */
	private void fireViewAnchorChange() {
		fireViewAnchorChangeEvent(getAnchor());
	}

	// ========================================================================
	// ===================== WIDGET PANEL DISPLAY
	// ========================================================================

	/*
	 * Get the widget being displayed in the widget panel
	 */
	private Widget getWidgetInPanel() {
		if (widgetPanel == null)
			return null;

		return widgetPanel.getWidget();
	}

	/*
	 * Set the panel widget
	 */
	private void setWidgetInPanel(Widget widget) {
		widgetPanel.setWidget(widget);
	}

	/*
	 * Show the table of user records as a result of an event
	 */
	private void returnToTableView(boolean showFirstPage) {
		/* Hide the error message */
		hideErrorMessage();

		/* Return to table and refresh the table records */
		showTableView();
		tableView.refresh(showFirstPage);

		/* Update the widget anchor */
		fireViewAnchorChange();
		
		/* update buttons */
		updateButtonForListView();
	}

	/*
	 * Show the table view in the widget panel
	 */
	private void showTableView() {
		setWidgetInPanel(tableView);
		showHeaderPanel(getConstants().userListViewHeading(), STYLE_LIST_VIEW);
		
	}

	/*
	 * Create and show a new external user view
	 */
	private ExternalUserView showExternalUserView() {
		view = new ExternalUserView(configuration,container);
		ExternalUserViewHandler handler = new ExternalUserViewHandler();
		view.addViewCommitHandler(handler);
		view.addViewCancelHandler(handler);
		view.addViewAnchorChangeHandler(handler);
		view.addViewEventHandler(handler);
		setWidgetInPanel(view);
		hideHeaderPanel();
		return view;
	}

	/*
	 * Create and show a user form for a newly created user
	 */
	private UserForm showUserFormForCreate() {
		/* Create the user to create */
		UiUser userToCreate = new UiUser();
		userToCreate.setExternallyAuthenticated(configuration.isHybridMode());

		/* Build the form */
		form = new UserForm(configuration, userToCreate, false, container);
		FormEventHandler eventHandler = new FormEventHandler();
		form.addViewCancelHandler(eventHandler);
		form.addViewCommitHandler(eventHandler);
		setWidgetInPanel(form);
		showHeaderPanel(getConstants().userCreateHeading(), STYLE_DETAIL_VIEW);
		return form;
	}

	/*
	 * Create and show the user form for the user being edited / viewed
	 */
	private UserForm showUserFormForEdit(UiUser user) {
		boolean readOnly = isReadOnly(user);
		form = new UserForm(configuration, user, readOnly,container);
		FormEventHandler eventHandler = new FormEventHandler();
		form.addViewCancelHandler(eventHandler);
		form.addViewCommitHandler(eventHandler);
		setWidgetInPanel(form);
		showHeaderPanel(getMessages().userEditHeading(user.getUserName()), STYLE_DETAIL_VIEW);
		return form;
	}

	/*
	 * Determine if the user must be shown as read-only
	 */
	private boolean isReadOnly(UiUser user) {
		if (!ClientAuthorizationManager.isAuthorized(USER_VIEW, USER_MODIFY)) {
			/* No permission to edit user */
			return true;
		}

		/* If admin user, the user must be the same as logged in user */
		if (user.isAdminUser()) {
			String currentUser = ClientAuthorizationManager.getSecurity().getUsername();
			if (currentUser == null || !currentUser.equals(user.getUserName())) {
				return true;
			}
		}

		return false;
	}

	// ========================================================================
	// ===================== TABLE VIEW EVENTS
	// ========================================================================

	/*
	 * Internal event handler for the table
	 */
	private final class TableEventHandler implements RecordClickHandler<UserRecord>,
	        UserTableViewEventHandler, ViewAnchorChangeHandler {

		/*
		 * Handle the table record click
		 */
		@Override
		public void onRecordClick(RecordClickEvent<UserRecord> event) {
			/* Hide the error message */
			hideErrorMessage();

			/* Verify that the user permissions authorize this action */
			if (!ClientAuthorizationManager.isAuthorized(USER_VIEW))
				return;

			/* Get the complete record from the server */
			final UserRecord record = event.getRecord();
			showBusyIndicator();
			getService().getUser(record.getKey(), new NeptuneAsyncCallback<UiUser>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					hideBusyIndicator();
					UiUserKey key = record.getKey();
					String idValue = (key.getId() != null) ? key.getId().toString() : "";
					String msg = caught.getLocalizedMessage();
					Window.alert(getMessages().getUserFailure(idValue, msg));
				}

				@Override
				public void onNeptuneSuccess(UiUser result) {
					hideBusyIndicator();

					
					/* Show the form */
					showUserFormForEdit(result);

					/* Update the anchor */
					fireViewAnchorChange();
				}
			});
		}

		/*
		 * Handle the table create external button click
		 */
		@Override
		public void onCreateExternalButtonClick(UserTableViewEvent event) {
			/* Hide the error message */
			hideErrorMessage();

			/* Create and show the external user view */
			ExternalUserView view = showExternalUserView();
			view.showTable(false);

			/* Update the anchor */
			fireViewAnchorChange();
		}

		/*
		 * Handle the table create button click
		 */
		@Override
		public void onCreateButtonClick(UserTableViewEvent event) {
			/* Hide the error message */
			hideErrorMessage();

			/* Show the form */
			showUserFormForCreate();

			/* Update the anchor */
			fireViewAnchorChange();
		}

		/*
		 * Handle the table delete button click
		 */
		@Override
		public void onDeleteButtonClick(UserTableViewEvent event) {
			/* Hide the error message */
			hideErrorMessage();

			/* Check if records are selected */
			List<UserRecord> records = tableView.getSelectedRecords();
			if (records.size() == 0) {
				container.setErrorMessage((getConstants().noUserCheckBoxSelectedError()));
				return;
			}

			/* Request confirmation from the user */
			if (!Window.confirm(getConstants().userDeleteConfirmation())) {
				return;
			}

			/* Collect the record keys */
			List<UiUserKey> userKeys = new ArrayList<UiUserKey>();
			for (UserRecord record : records)
				userKeys.add(record.getKey());

			/* Delete the selected records */
			showBusyIndicator();
			getService().deleteUsers(userKeys, new NeptuneAsyncCallback<Void>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					/* Hide the busy indicator and report error to user */
					hideBusyIndicator();
					container.setErrorMessage((getMessages().deleteUsersFailure(caught.getLocalizedMessage())));
				}

				@Override
				public void onNeptuneSuccess(Void result) {
					/* Hide the busy indicator */
					hideBusyIndicator();

					/* Refresh the table records */
					tableView.refresh(false);
				}
			});
		}

		/*
		 * Handle the table anchor change
		 */
		@Override
		public void onAnchorChange(ViewAnchorChangeEvent event) {
			/* Hide the error message */
			hideErrorMessage();

			/* If the table view is being displayed, fire the new anchor change */
			if (getWidgetInPanel() == tableView)
				fireViewAnchorChange();
		}
	}

	// ========================================================================
	// ===================== EXTERNAL USER VIEW EVENTS
	// ========================================================================

	/*
	 * Internal event handler for the external user view
	 */
	private final class ExternalUserViewHandler implements ExternalUserViewEventHandler,
	        ViewCommitHandler<ExternalUserRecord>, ViewCancelHandler, ViewAnchorChangeHandler {

		/* User saved, return to first page of the table */
		@Override
		public void onCommit(ViewCommitEvent<ExternalUserRecord> event) {
			returnToTableView(false);
		}

		/* Action canceled, return to the table */
		@Override
		public void onCancel(ViewCancelEvent event) {
			returnToTableView(false);
		}

		/* Anchor of the view changed */
		@Override
		public void onAnchorChange(ViewAnchorChangeEvent event) {
			if (getWidgetInPanel() instanceof ExternalUserView) {
				fireViewAnchorChange();
			}
		}

		/* Blank widget in the external user view, update header */
		@Override
		public void onShowBlankWidget(ExternalUserViewEvent event) {
			if (getWidgetInPanel() instanceof ExternalUserView) {
				hideHeaderPanel();
			}
		}

		/* Table in the external user view, update header */
		@Override
		public void onShowTable(ExternalUserViewEvent event) {
			if (getWidgetInPanel() instanceof ExternalUserView) {
				showHeaderPanel(getConstants().externalUserListViewHeading(), STYLE_LIST_VIEW);
			}
		}

		/* User Form in the external user view, update header */
		@Override
		public void onShowExternalUser(ExternalUserViewEvent event) {
			if (getWidgetInPanel() instanceof ExternalUserView) {
				showHeaderPanel(getConstants().userCreateExternalHeading(), STYLE_DETAIL_VIEW);
			}
		}
	}

	// ========================================================================
	// ===================== USER FORM EVENTS
	// ========================================================================

	/*
	 * Internal event handler for the user form
	 */
	private final class FormEventHandler implements ViewCommitHandler<UiUser>, ViewCancelHandler {

		/* User saved, return to first page of the table */
		@Override
		public void onCommit(ViewCommitEvent<UiUser> event) {
			returnToTableView(false);
		}

		/* Action canceled, return to the table */
		@Override
		public void onCancel(ViewCancelEvent event) {
			returnToTableView(false);
		}
	}
	
	private void updateButtonForListView(){
		if(tableView != null){
			tableView.showCreateButton();
			tableView.showCreateExternalButton();
			tableView.showDeleteButton();
		}
		if(form != null){
			form.getCancelButton().setVisible(false);
			form.getSaveButton().setVisible(false);
		}
		if(view != null && view.getExternalUserForm() != null){
			if(view.getExternalUserForm().getSaveButton() != null)
				view.getExternalUserForm().getSaveButton().setVisible(false);
			if(view.getExternalUserForm().getCancelButton() != null)
				view.getExternalUserForm().getCancelButton().setVisible(false);
		}
		if(view != null && view.getExternalTableView().getCancelButton() != null)
				view.getExternalTableView().getCancelButton().setVisible(false);
		
	}
	
	private void clearButtons(){
		if(tableView != null){
			tableView.hideCreateButton();
			tableView.hideCreateExternalButton();
			tableView.hideDeleteButton();
		}
		if(form != null){
			form.getCancelButton().setVisible(false);
			form.getSaveButton().setVisible(false);
		}
		if(view != null && view.getExternalUserForm() != null){
			if(view.getExternalUserForm().getSaveButton() != null)
				view.getExternalUserForm().getSaveButton().setVisible(false);
			if(view.getExternalUserForm().getCancelButton() != null)
				view.getExternalUserForm().getCancelButton().setVisible(false);
		}
		if(view != null && view.getExternalTableView().getCancelButton() != null)
				view.getExternalTableView().getCancelButton().setVisible(false);
		}
	
	
}
