package com.tandbergtv.neptune.ui.framework.client.application;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.MenuBar;
import com.google.gwt.user.client.ui.MenuItem;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.application.WidgetFactory;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.MenuBarWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.InvalidPagePanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * The main widget containing the menu bar structure used by the Neptune application.
 * 
 * @author trybak
 * @author Vijay Silva
 */
public class TreeTabPanel extends Composite {

	/* Style Names */
	private static final String STYLE_NAME = "nfw-TreeTabPanel";
	private static final String STYLE_MENU = "nfw-TreeTabPanel-menu";
	private static final String STYLE_MENU_LEVEL1 = "nfw-TreeTabPanel-menu1";
	private static final String STYLE_MENU_LEVEL2 = "nfw-TreeTabPanel-menu2";
	private static final String STYLE_SEPARATOR = "nfw-TreeTabPanel-separator";
	private static final String STYLE_MENU_ITEM = "nfw-TreeTabPanel-menuItem";
	private static final String STYLE_WIDGET_CONTAINER = "nfw-TreeTabPanel-widgetContainer";

	/* Widgets */
	private final VerticalContainer mainContainer;
	private final MenuBarWidget tabMenuBar;
	private final SimpleContainer centerContainer;
	private final MenuNode rootNode;

	/* Widget Factory for the currently selected menu item */
	private WidgetFactory currentFactory;

	/* Event listeners */
	private final List<Listener> listeners;
	private final AnchorChangeListener frameworkListener;

	/* Constants */
	private ApplicationMessages constants = GWT.create(ApplicationMessages.class);

	/**
	 * Instantiate a tree tab panel with a given {@link AnchorChangeListener}. Anchor change
	 * messages initiated by the managed tabs will be forwarded to the framework listener, after the
	 * tab anchor is prepended.
	 * 
	 * @param frameworkListener an external {@link AnchorChangeListener} that will be informed of
	 *        any anchor change messages initiated by any of the managed tabs.
	 */
	public TreeTabPanel(AnchorChangeListener frameworkListener) {
		this.frameworkListener = frameworkListener;
		listeners = new ArrayList<Listener>();
		rootNode = new MenuNode("", "");
		tabMenuBar = new MenuBarWidget();
		tabMenuBar.setStylePrimaryName(STYLE_MENU);
		tabMenuBar.setAutoOpen(true);

		mainContainer = new VerticalContainer();
		mainContainer.setStyleName(STYLE_NAME);

		MenuItem menuItem = new MenuItem(constants.loading(), new Command() {
			public void execute() {
				return;
			}
		});
		menuItem.setStylePrimaryName(STYLE_MENU_ITEM);
		tabMenuBar.addItem(menuItem);

		HorizontalContainer menuContainer = new HorizontalContainer();
		menuContainer.addStyleName(STYLE_MENU_LEVEL1);

		menuContainer.add(tabMenuBar);
		menuContainer.setCellHorizontalAlignment(tabMenuBar, HorizontalContainer.ALIGN_LEFT);

		mainContainer.add(menuContainer);

		/* Add secondary menu */
		SimpleContainer secondaryMenuContainer = new SimpleContainer();
		secondaryMenuContainer.setStylePrimaryName(STYLE_MENU_LEVEL2);
		mainContainer.add(secondaryMenuContainer);

		/* Add separator container */
		SimpleContainer separatorContainer = new SimpleContainer();
		separatorContainer.setStylePrimaryName(STYLE_SEPARATOR);
		mainContainer.add(separatorContainer);

		centerContainer = new MainWidgetContainer(this);
		centerContainer.setStyleName(STYLE_WIDGET_CONTAINER);
		mainContainer.add(centerContainer);
		mainContainer.setCellHorizontalAlignment(centerContainer, HorizontalContainer.ALIGN_CENTER);
		initWidget(mainContainer);
	}

	/**
	 * Hide the menu popup, if showing
	 */
	public void hideMenu() {
		tabMenuBar.closeAllChildren(false);
	}

	/**
	 * Add listener for event indicating that a menu tab was selected
	 * 
	 * @param listener The listener
	 */
	public void addListener(Listener listener) {
		listeners.add(listener);
	}

	/**
	 * Add an event handler for the menu item selection event
	 * 
	 * @param handler The event handler
	 */
	public void addMenuItemSelectionHandler(MenuItemSelectionHandler handler) {
		addHandler(handler, MenuItemSelectionEvent.getType());
	}

	/**
	 * Get the root menu node
	 * 
	 * @return The root menu node
	 */
	public MenuNode getRootMenu() {
		return rootNode;
	}

	/**
	 * Layout the menu items from the set of built menu nodes
	 */
	public void doLayout() {
		tabMenuBar.clearItems();
		for (Node node : rootNode.subNodes) {
			MenuItem menuItem = node.createMenuItem();
			if (menuItem != null)
				tabMenuBar.addItem(menuItem);
		}
	}

	/**
	 * Get the content widget displayed in the content area
	 * 
	 * @return The widget displayed in the content area
	 */
	public Widget getContentWidget() {
		return centerContainer.getWidget();
	}

	/*
	 * Determine if the menu action can be performed
	 */
	private boolean cancelMenuAction() {
		/* Allow the parent to decide if the action should be prevented */
		MenuItemSelectionEvent event = new MenuItemSelectionEvent();
		fireEvent(event);
		return event.isCanceled();
	}

	public abstract class Node {
		protected final String name;
		protected final String displayName;
		protected MenuNode parentNode;

		protected Node(String name) {
			this.name = name;
			this.displayName = name;
		}

		protected Node(String name, String displayName) {
			this.name = name;
			this.displayName = displayName;
		}

		protected abstract MenuItem createMenuItem();

		public abstract void accept(NodeVisitor visitor);

		/**
		 * @return the displayName
		 */
		public String getDisplayName() {
			return displayName;
		}
	}

	public interface NodeVisitor {
		void visit(MenuNode node);

		void visit(TabNode node);

		void visit(ActionNode node);
	}

	public final class MenuNode extends Node {
		private final List<Node> subNodes;
		private final Map<String, Node> nameToSubNode;

		private MenuNode(String name, String displayName) {
			super(name, displayName);
			subNodes = new ArrayList<Node>();
			nameToSubNode = new HashMap<String, Node>();
		}

		/**
		 * Returns a top-level menu node whose name matches the specified one
		 * 
		 * @param name
		 * @return
		 */
		public MenuNode getMenuNode(String name) {
			for (Node node : subNodes) {
				if (node instanceof MenuNode && node.name.equals(name))
					return (MenuNode) node;
			}

			return null;
		}

		@Override
		protected MenuItem createMenuItem() {
			MenuBar subMenu = new MenuBarWidget(true);
			subMenu.setStylePrimaryName(STYLE_MENU);
			subMenu.setAutoOpen(true);

			int childItemCount = 0;
			for (Node node : subNodes) {
				MenuItem menuItem = node.createMenuItem();
				if (menuItem != null) {
					subMenu.addItem(menuItem);
					childItemCount++;
				}
			}

			MenuItem item = null;
			if (childItemCount > 0) {
				item = new MenuItem(displayName, subMenu);
				item.setStylePrimaryName(STYLE_MENU_ITEM);
			}

			return item;
		}

		public MenuNode addMenuNode(String name, String displayName) {
			MenuNode subMenuNode = new MenuNode(name, displayName);
			addNode(subMenuNode);
			return subMenuNode;
		}

		public TabNode addTabNode(String name, String displayName, WidgetFactory widgetFactory,
				AnchorChangeListener tabListener) {
			TabNode panelTabNode = new TabNode(name, displayName, widgetFactory, tabListener);
			addNode(panelTabNode);

			return panelTabNode;

		}

		public TabNode addTabNode(String name, WidgetFactory widgetFactory) {
			return addTabNode(name, name, widgetFactory, null);
		}

		public ActionNode addActionNode(String name, String displayName, Runnable runnable) {
			ActionNode actionNode = new ActionNode(name, displayName, runnable);
			addNode(actionNode);
			return actionNode;
		}

		private void addNode(Node node) {
			subNodes.add(node);
			node.parentNode = this;
			nameToSubNode.put(node.name, node);
		}

		@Override
		public void accept(NodeVisitor visitor) {
			visitor.visit(this);
		}
	}

	public final class ActionNode extends Node {
		private final Runnable runnable;

		public ActionNode(String name, String displayName, Runnable runnable) {
			super(name, displayName);
			this.runnable = runnable;
		}

		@Override
		public void accept(NodeVisitor visitor) {
			visitor.visit(this);
		}

		@Override
		protected MenuItem createMenuItem() {
			MenuItem item = new MenuItem(displayName, new Command() {
				public void execute() {
					if (cancelMenuAction()) {
						return;
					}

					runnable.run();
				}
			});
			item.setStylePrimaryName(STYLE_MENU_ITEM);
			return item;
		}
	}

	public final class TabNode extends Node {
		private final WidgetFactory widgetFactory;
		private final AnchorChangeListener tabAnchorListener;
		private final AnchorChangeListener reverseAnchorListener;

		private TabNode(String name, String displayName, WidgetFactory widgetFactory,
				AnchorChangeListener tabListener) {
			super(name, displayName);
			this.widgetFactory = widgetFactory;
			this.tabAnchorListener = tabListener;

			reverseAnchorListener = new AnchorChangeListener() {
				public void anchorChanged(final String anchor) {
					StringBuilder realAnchor = new StringBuilder();
					realAnchor.append(getToken());
					if (!anchor.startsWith("?"))
						realAnchor.append('.');
					realAnchor.append(anchor);
					frameworkListener.anchorChanged(realAnchor.toString());
				}
			};
		}

		private TabNode(String name, WidgetFactory widgetFactory) {
			this(name, name, widgetFactory, null);
		}

		public String getToken() {
			StringBuilder token = new StringBuilder();
			constructToken(this, token);
			return token.toString();
		}

		public AnchorChangeListener getReverseAnchorChangeListener() {
			return reverseAnchorListener;
		}

		public String getName() {
			return name;
		}

		public WidgetFactory getWidgetFactory() {
			return widgetFactory;
		}

		private void constructToken(Node node, StringBuilder builder) {
			if (node == null)
				return;

			constructToken(node.parentNode, builder);

			if (builder.length() > 0)
				builder.append('.');
			builder.append(node.name);
		}

		@Override
		protected MenuItem createMenuItem() {
			MenuItem item = new MenuItem(displayName, new Command() {
				public void execute() {
					if (cancelMenuAction()) {
						return;
					}

					activatePanel(true);
				}
			});
			item.setStylePrimaryName(STYLE_MENU_ITEM);
			return item;
		}

		public void activatePanel(Queue<String> queue, String params, boolean notifyListener) {
			StringBuilder anchor = new StringBuilder();
			for (String bit : queue)
				anchor.append(bit).append('.');
			if (anchor.length() > 0)
				anchor.setLength(anchor.length() - 1);
			if (params != null)
				anchor.append(params);

			if (currentFactory != null && centerContainer.getWidget() != null) {
				currentFactory.release(centerContainer.getWidget());
			}

			currentFactory = widgetFactory;
			centerContainer.clear();
			centerContainer.add(widgetFactory.getInstance());
			if (tabAnchorListener != null)
				tabAnchorListener.anchorChanged(anchor.toString());
			if (notifyListener)
				for (Listener listener : listeners)
					listener.tabChanged(TabNode.this);
		}

		public void activatePanel(boolean notifyListener) {
			activatePanel(new LinkedList<String>(), null, notifyListener);
		}

		@Override
		public void accept(NodeVisitor visitor) {
			visitor.visit(this);
		}

	}

	public interface AnchorChangeListener {
		void anchorChanged(String anchor);
	}

	public void setActiveTab(String token) {
		int indexOfParams = token.indexOf('?');
		String params = null;
		if (indexOfParams != -1) {
			params = token.substring(indexOfParams);
			token = token.substring(0, indexOfParams);
		}
		Queue<String> queue = new LinkedList<String>();
		String[] tokenBits = token.split("\\.", -1);
		for (String tokenBit : tokenBits)
			queue.add(tokenBit);

		final NodeActivator locator = new NodeActivator(queue, params);
		rootNode.accept(locator);
	}

	public interface Listener {
		void tabChanged(TabNode tabNode);
	}

	private final class NodeActivator implements NodeVisitor {
		private final Queue<String> queue;
		private final String params;

		public NodeActivator(Queue<String> queue, String params) {
			this.queue = queue;
			this.params = params;
		}

		public void visit(MenuNode menuNode) {
			if (queue.peek() == null)
				return;
			String name = queue.remove();
			Node node = menuNode.nameToSubNode.get(name);
			if (node != null) {
				NodeActivator subLocator = new NodeActivator(queue, params);
				node.accept(subLocator);
			} else {
				/*
				 * If the sub node is not found for any reason, show the invalid URL page rather
				 * than doing nothing.
				 */
				centerContainer.clear();
				centerContainer.add(new InvalidPagePanel(constants.invalidURL()));
			}
		}

		public void visit(TabNode panelTabNode) {
			panelTabNode.activatePanel(queue, params, false);
		}

		public void visit(ActionNode node) {
		}
	}

}

class MainWidgetContainer extends SimpleContainer {
	private TreeTabPanel treeTabPanel;
	public MainWidgetContainer(TreeTabPanel treeTabPanel) {
		super();
		sinkEvents(Event.ONMOUSEOVER);
		this.treeTabPanel = treeTabPanel;
	}
	@Override
	public void onBrowserEvent(Event event) {
		super.onBrowserEvent(event);
		if (DOM.eventGetType(event) == Event.ONMOUSEOVER) {
			treeTabPanel.hideMenu();
		}
	}
	
}
