package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.INeptuneWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.ColumnCommandsFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.CheckBox;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.ColumnCommandsFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.DetailFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.FeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.PageFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.SortFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.TableInternal;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedCornerContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * Table that supports a simple view by default, but that can be extended with
 * the addition on features.
 * 
 * Supported Features: SortFeature DetailFeature PageFeature
 * ColumnCommandsFeature
 * 
 * TODO, continue re-factor to properly encapsulate the built-in table
 * functionality. The TableInternal interface should not allow access to the
 * internal column structure to support the additional features.
 */
public class Table<K, R extends Record<K>> extends Composite implements TableInternal<K, R>,
        TableConstants, TableExternal<K, R>, INeptuneWidget {
	private final SimplePanel mainPanel;
	private final DataProvider<K, R> dataProvider;
	private final ListViewPanel<K, R> listViewPanel;
	private final List<FeatureHandler> featureHandlers;
	private DetailFeatureHandler<K, R> detailFeatureHandler;

	private boolean initialized;
	
	private synchronized boolean isInitialized() {
		return initialized;
	}
	
	private synchronized void setInitialized(boolean initialized) {
		this.initialized = initialized;
	}
	
	public Table(final DataProvider<K, R> dataProvider) {
		super();

		featureHandlers = new ArrayList<FeatureHandler>();
		mainPanel = new SimplePanel();
		this.dataProvider = dataProvider;
		listViewPanel = new ListViewPanel<K, R>(dataProvider);

		this.initWidget(mainPanel);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableExternal
	 * #init()
	 */
	public void init(final AsyncCallback<Void> callback) {
		if(isInitialized()) {
			listViewPanel.refresh(dataProvider);
			mainPanel.setWidget(listViewPanel);
			callback.onSuccess(null);
			return;
		}

		dataProvider.initialize(new AsyncCallback<Void>() {
			public void onFailure(Throwable caught) {
				Window.alert(caught.getLocalizedMessage());
				callback.onFailure(caught);
			}

			public void onSuccess(Void result) {
				for (FeatureHandler handler : featureHandlers)
					handler.init();

				listViewPanel.refresh(dataProvider);
				mainPanel.setWidget(listViewPanel);
				setInitialized(true);
				callback.onSuccess(result);
			}
		});
	}

	/**
	 * Installs the Page Feature.
	 * 
	 * @param pageFeature
	 *            the page feature
	 */
	public void addPageFeature(PageFeature pageFeature) {
		PageFeatureHandler<K, R> pageFeatureHandler = new PageFeatureHandler<K, R>(dataProvider, this, pageFeature);
		featureHandlers.add(pageFeatureHandler);
	}

	/**
	 * Installs the Sort Feature.
	 * 
	 * @param sortFeature
	 *            the sort feature
	 */
	public void addSortFeature(final SortFeature<K, R> sortFeature) {
		SortFeatureHandler<K, R> sortFeatureHandler = new SortFeatureHandler<K, R>(dataProvider, this, sortFeature);
		featureHandlers.add(sortFeatureHandler);
	}

	/**
	 * Installs the Detail Feature
	 * 
	 * @param detailFeature
	 *            the detail feature
	 */
	public void addDetailFeature(final DetailFeature<K, R> detailFeature) {
		DetailFeatureHandler<K, R> detailFeatureHandler = new DetailFeatureHandler<K, R>(dataProvider, this,
				detailFeature);
		featureHandlers.add(detailFeatureHandler);
		this.detailFeatureHandler = detailFeatureHandler;
	}

	/**
	 * Install the Column Command feature.
	 * 
	 * @param columnCommandFeature
	 *            the column command feature
	 */
	public void addColumnCommandFeature(final ColumnCommandsFeature<R> columnCommandFeature) {
		ColumnCommandsFeatureHandler<K, R> columnCommandHandler = new ColumnCommandsFeatureHandler<K, R>(
				columnCommandFeature, this);
		featureHandlers.add(columnCommandHandler);
	}

	/**
	 * Install the Bookmark Feature
	 * 
	 * @param bookmarkFeature
	 *            the bookmark feature
	 */
	public void addBookmarkFeature(final BookmarkFeatureImpl<?, R> bookmarkFeature) {
		featureHandlers.add(new FeatureHandler() {
			public void reload() {
			}
			
			public void init() {
			}
		});
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableExternal
	 * #revertToInitialState()
	 */
	public void revertToInitialState() {
		for (FeatureHandler handler : featureHandlers)
			handler.reload();
		refresh();
		setListView();
	}

	private final static class ListViewPanel<K, R extends Record<K>> extends RoundedCornerContainer {
		private final HorizontalContainer paginationPanel;
		private final HorizontalContainer actionContainer;
		private final ColumnPanel columnPanel;
		private final List<Listener> listeners;
		private final VerticalContainer mainContainer;

		public ListViewPanel(DataProvider<K, R> dataProvider) {
			super(ALL, ROUNDED_PANEL_CORNER_HEIGHT);
			listeners = new ArrayList<Listener>();
			mainContainer = new VerticalContainer();
			mainContainer.setStyleName(STYLE_TABLE_PANEL);
			// create the placeholder panels even if the feature is not enabled
			// as
			// this allows features to be added dynamically without worry of
			// order
			actionContainer = new HorizontalContainer();
			paginationPanel = new HorizontalContainer();
			columnPanel = new ColumnPanel(dataProvider);

			mainContainer.setSpacing(PANEL_SPACING);
			mainContainer.setSize("100%", "100%");
			mainContainer.setHorizontalAlignment(VerticalContainer.ALIGN_CENTER);
			
			columnPanel.setStyleName(STYLE_TABLE_BODY);
			mainContainer.setCellHorizontalAlignment(columnPanel, VerticalContainer.ALIGN_LEFT);
			mainContainer.add(columnPanel);

			if (dataProvider.isRecordCountEnabled()) {
				SimpleContainer numRecordsPanel = new SimpleContainer();
				LabelWidget numRecordsLabel = new LabelWidget("");
				numRecordsLabel.setStyleName(STYLE_GENERAL_TEXT);
				numRecordsPanel.setWidget(numRecordsLabel);
				mainContainer.add(numRecordsPanel);
			}
			mainContainer.add(actionContainer);
			mainContainer.setCellHorizontalAlignment(actionContainer, VerticalContainer.ALIGN_LEFT);

			paginationPanel.setWidth("100%");
			mainContainer.add(paginationPanel);
			mainContainer.setCellHorizontalAlignment(paginationPanel, VerticalContainer.ALIGN_CENTER);
			
			setWidget(mainContainer);
			
			setSize("100%", "100%");
			setCornerStyleName(STYLE_TABLE_PANEL);
		}

		public HorizontalPanel getHeaderColPanelForColumn(Column<?, R> column) {
			return columnPanel.visibleColumnToHeaderColPanel.get(column);
		}

		public Label getHeaderLabelForColumn(Column<?, R> column) {
			return columnPanel.visibleColumnToHeaderLabel.get(column);
		}

		/**
		 * a refresh will cause an asynchronous update of the current data
		 */
		public void refresh(final DataProvider<K, R> dataProvider) {
			// create data rows
			columnPanel.dataRowsCheckBoxes.clear();
			dataProvider.getRecords(new AsyncCallback<List<R>>() {
				public void onFailure(Throwable caught) {
					Window.alert(caught.getLocalizedMessage());
				}

				public void onSuccess(List<R> result) {
					columnPanel.updateRecords(dataProvider, result);
				}
			});
		}

		public HorizontalContainer getPaginationPanel() {
			return paginationPanel;
		}

		public List<Column<?, R>> listVisibleColumns() {
			return columnPanel.listVisibleColumns();
		}

		public List<Widget> listWidgetsForVisibleColumn(Column<?, R> visibleColumn) {
			return columnPanel.listWidgetsForVisibleColumn(visibleColumn);
		}

		public R getRecordForWidget(Widget widget) {
			return columnPanel.getRecordForWidget(widget);
		}

		public void addListener(Listener listener) {
			listeners.add(listener);
		}

		private void notifyAllListeners() {
			for (Listener listener : listeners)
				listener.updated();
		}

		public interface Listener {
			void updated();
		}

		private final class ColumnPanel extends FlexTable {
			private int colIndex;
			private final CheckBox<R> headerRowCheckbox;
			private final List<CheckBox<R>> dataRowsCheckBoxes;
			private final List<Column<?, R>> visibleColumns;
			private final Map<Column<?, R>, List<Widget>> visibleColumnToWidgetListMap;
			private final Map<Widget, R> widgetToRecordMap;
			private final Map<Column<?, R>, Label> visibleColumnToHeaderLabel;
			private final Map<Column<?, R>, HorizontalPanel> visibleColumnToHeaderColPanel;

			public ColumnPanel(DataProvider<K, R> dataProvider) {
				setSize("100%", "100%");
				headerRowCheckbox = new CheckBox<R>();
				dataRowsCheckBoxes = new ArrayList<CheckBox<R>>();
				visibleColumns = new ArrayList<Column<?, R>>();
				visibleColumnToWidgetListMap = new HashMap<Column<?, R>, List<Widget>>();
				widgetToRecordMap = new HashMap<Widget, R>();
				visibleColumnToHeaderColPanel = new HashMap<Column<?, R>, HorizontalPanel>();
				visibleColumnToHeaderLabel = new HashMap<Column<?, R>, Label>();

				// create header row
				getRowFormatter().setStyleName(0, STYLE_HEADER_ROW);
				colIndex = 0;
				if (dataProvider.isCheckboxEnabled()) {
					// selecting all datarow checkboxes when header row checkbox
					// is
					// selected
					headerRowCheckbox.addClickHandler(new ClickHandler() {
						public void onClick(ClickEvent event) {
							CheckBox<R> cb = headerRowCheckbox;
							for (CheckBox<R> dataRowCB : dataRowsCheckBoxes) {
								dataRowCB.setValue(cb.getValue());
							}
						}
					});
					setWidget(0, colIndex++, headerRowCheckbox);
				}

				for (final Column<?, R> column : dataProvider.getColumns()) {
					addColumn(column);
				}
			}

			private void addColumn(final Column<?, R> column) {
				visibleColumns.add(column);
				HorizontalPanel headerColPanel = new HorizontalPanel();
				headerColPanel.setSpacing(3);
				headerColPanel.setVerticalAlignment(HorizontalPanel.ALIGN_MIDDLE);
				Label headerLabel = new Label(column.getDisplayName());
				headerColPanel.add(headerLabel);

				headerLabel.setStyleName(STYLE_HEADER_NONSORTABLE_COL_TEXT);
				setWidget(0, colIndex++, headerColPanel);
				visibleColumnToHeaderLabel.put(column, headerLabel);
				visibleColumnToHeaderColPanel.put(column, headerColPanel);
			}

			public void updateRecords(DataProvider<K, R> dataProvider, List<R> result) {
				// clean up the maps since all the widgets will be remade
				visibleColumnToWidgetListMap.clear();
				widgetToRecordMap.clear();

				// clean out existing rows
				final int rowCount = getRowCount();
				for (int i = rowCount - 1; i > 0; --i)
					removeRow(i);

				int rowIndex = 1;
				for (final R record : result) {
					String styleName = (rowIndex % 2 == 0) ? STYLE_DATA_EVEN_ROW : STYLE_DATA_ODD_ROW;
					getRowFormatter().setStyleName(rowIndex, styleName);
					int colIndex = 0;
					if (dataProvider.isCheckboxEnabled()) {
						final CheckBox<R> dataRowCheckBox = new CheckBox<R>();
						dataRowCheckBox.setRecord(record);
						// unselecting the header row checkbox when any datarow
						// checkbox is unselected
						dataRowCheckBox.addClickHandler(new ClickHandler() {
							public void onClick(ClickEvent event) {
								if (headerRowCheckbox.getValue()) {
									if (!dataRowCheckBox.getValue())
										headerRowCheckbox.setValue(false);
								}
							}
						});
						dataRowsCheckBoxes.add(dataRowCheckBox);
						setWidget(rowIndex, colIndex++, dataRowCheckBox);
					}
					for (Column<?, R> field : visibleColumns) {
						final View<?> listView = field.getView(record);
						String viewStyleName = listView.getStyleName();
						if (viewStyleName == null)
							viewStyleName = STYLE_DATA_TEXT;
						Widget widget = listView.getWidget();
						widget.setStyleName(viewStyleName);
						setWidget(rowIndex, colIndex, widget);
						colIndex++;

						// add widget to list
						getInitializedWidgetList(field).add(widget);
						widgetToRecordMap.put(widget, record);
					}
					rowIndex++;
				}

				// update all listeners that cell view has been updated
				notifyAllListeners();

			}

			private List<Widget> getInitializedWidgetList(Column<?, R> visibleColumn) {
				List<Widget> list = visibleColumnToWidgetListMap.get(visibleColumn);
				if (list == null) {
					list = new ArrayList<Widget>();
					visibleColumnToWidgetListMap.put(visibleColumn, list);
				}
				return list;
			}

			public List<Column<?, R>> listVisibleColumns() {
				return visibleColumns;
			}

			public List<Widget> listWidgetsForVisibleColumn(Column<?, R> column) {
				return visibleColumnToWidgetListMap.get(column);
			}

			public R getRecordForWidget(Widget widget) {
				return widgetToRecordMap.get(widget);
			}

			public List<R> listSelectedRecords() {
				List<R> records = new ArrayList<R>();
				for (CheckBox<R> checkBox : dataRowsCheckBoxes)
					if (checkBox.getValue())
						records.add(checkBox.getRecord());
				return records;
			}
		}
	}

	public List<Column<?, R>> listVisibleColumns() {
		return listViewPanel.listVisibleColumns();
	}

	public void setListView() {
		mainPanel.setWidget(listViewPanel);
		listViewPanel.notifyAllListeners();
		
	}

	public void setRootWidget(Widget widget) {
		mainPanel.setWidget(widget);
	}

	public void addListener(final Listener<K, R> listener) {
		listViewPanel.addListener(new ListViewPanel.Listener() {
			public void updated() {
				listener.recordsUpdated();
			}
		});
	}

	public R getRecordForWidget(Widget widget) {
		return listViewPanel.getRecordForWidget(widget);
	}

	public List<Widget> listWidgetsForVisibleColumn(Column<?, R> visibleColumn) {
		return listViewPanel.listWidgetsForVisibleColumn(visibleColumn);
	}

    /*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableExternal
	 * #refresh()
	 */
	public void refresh() {
		listViewPanel.refresh(dataProvider);
	}

	public HorizontalPanel getHeaderColPanelForColumn(Column<?, R> column) {
		return listViewPanel.getHeaderColPanelForColumn(column);
	}

	public Label getHeaderLabelForColumn(Column<?, R> column) {
		return listViewPanel.getHeaderLabelForColumn(column);
	}

	public HorizontalContainer getPaginationPanel() {
		return listViewPanel.getPaginationPanel();
	}

	public List<R> getSelectedRecords() {
		return listViewPanel.columnPanel.listSelectedRecords();
	}

	public void updateView(R record) {
		// the record has been modified, need to refresh the view
		refresh();
	}

	public void addColumn(Column<?, R> column) {
		listViewPanel.columnPanel.addColumn(column);
	}

	public void registerWidgetOnActionContainer(Widget widget) {
		listViewPanel.actionContainer.add(widget);
	}

	public void showCreatePage() {
		detailFeatureHandler.getRequester().showCreatePage();
	}

	public void showRecord(R record) {
		detailFeatureHandler.getRequester().showDetailViewForRecord(record);
	}

	@Override
	public boolean isDisplayed() {
	    if (!this.isAttached())
			return false;

		Widget current = this;
		while (current != null && current.isVisible())
			current = current.getParent();
		return (current == null);
	}

	@SuppressWarnings("unchecked")
	@Override
	public void refreshFirstPage() {
		for(FeatureHandler handler : featureHandlers) {
			if(handler instanceof PageFeatureHandler) {
				((PageFeatureHandler<K, R>)handler).setPageStart(0);
				((PageFeatureHandler<K, R>)handler).setPageNumber(null);
				break;
			}
		}
		refresh();
		setListView();
	}
}
