package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import java.util.HashMap;
import java.util.Map;

import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;

public class PageFeatureHandler<K, R extends Record<K>> implements TableConstants, FeatureHandler {

	/* State */
	private final TableInternal<K, R> table;
	private final PageFeature feature;
	private final PageFeatureWidget widget;
	private static final int DEFAULT_PAGE_SIZE = 10;

	/* The anchor tokens for paging */
	private static final String PAGE_START_ANCHOR_TOKEN = "PageStart";
	private static final String PAGE_SIZE_ANCHOR_TOKEN = "PageSize";

	/**
	 * Constructor
	 * 
	 * @param table The table
	 * @param dataProvider the data provider
	 * @param feature The feature
	 */
	public PageFeatureHandler(final TableInternal<K, R> table, final PageFeature pageFeature) {
		this.table = table;
		this.feature = pageFeature;
		this.widget = new PageFeatureWidget(this);

		/* Add the widget to the table */
		this.widget.setWidth("100%");
		this.table.getPaginationPanel().setWidget(widget);

		/* Add a listener for when the table is updated */
		table.addListener(new TableInternal.Listener<K, R>() {
			@Override
			public void recordsUpdated() {
				widget.refresh();
			}
		});

		reset();
	}

	@Override
	public void init() {
		resetFeature();
		widget.setPageNumberText(null);

	}

	/**
	 * Reset the starting and page size to the default value
	 */
	@Override
	public void reset() {
		resetFeature();
		widget.setPageNumberText(null);
	}

	/*
	 * Reset the state maintained by the feature
	 */
	private void resetFeature() {
		int defaultPageSize = feature.getDefaultPageSize();
		if (defaultPageSize <= 0)
			defaultPageSize = DEFAULT_PAGE_SIZE;

		feature.setStart(0);
		feature.setLength(defaultPageSize);
	}

	/**
	 * Get the tokens to add to the anchor to re-initialize the state of this feature
	 * 
	 * @see com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.FeatureHandler#getAnchorTokens()
	 */
	@Override
	public Map<String, String> getAnchorTokens() {
		Map<String, String> tokenMap = new HashMap<String, String>();

		/* Put the start page and the page size in the anchor */
		tokenMap.put(PAGE_START_ANCHOR_TOKEN, Integer.toString(feature.getStart()));
		tokenMap.put(PAGE_SIZE_ANCHOR_TOKEN, Integer.toString(feature.getLength()));

		return tokenMap;
	}

	public void setPageStart(int start) {
		feature.setStart(start);
	}

	public void setPageNumber(Integer pageNumber) {
		widget.setPageNumberText(pageNumber);
	}

	/*
	 * Get the table
	 */
	TableInternal<K, R> getTable() {
		return this.table;
	}

	/*
	 * Allow the page feature widget to access the feature state
	 */
	PageFeature getFeature() {
		return this.feature;
	}

	/*
	 * Update the page. This is triggered by a user event on the widget
	 */
	void updatePage(int start, int length) {
		feature.setStart(start);
		feature.setLength(length);
		table.refresh();
	}
}
