/*
 * Created on Sep 14, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedCornerContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * The table list view panel
 * 
 * @author Vijay Silva
 */
class TableListViewPanel<K, R extends Record<K>> extends RoundedCornerContainer {

	/* Properties */
	private final DataProvider<K, R> dataProvider;
	private SimpleContainer paginationPanel;
	private HorizontalContainer actionContainer;
	private TableColumnPanel<K, R> columnPanel;
	private List<Listener> listeners;
	private VerticalContainer mainContainer;

	/**
	 * Constructor
	 * 
	 * @param dataProvider The data provider
	 */
	public TableListViewPanel(DataProvider<K, R> dataProvider) {
		super(ALL, TableConstants.ROUNDED_PANEL_CORNER_HEIGHT);
		this.dataProvider = dataProvider;

		initializeWidget();
	}

	// ========================================================================
	// ===================== INITIALIZATION
	// ========================================================================

	/*
	 * Initialize the widget
	 */
	private void initializeWidget() {
		listeners = new ArrayList<Listener>();
		mainContainer = new VerticalContainer();
		mainContainer.setStyleName(TableConstants.STYLE_TABLE_PANEL);

		/* create place holder panels even if the feature is not enabled */
		actionContainer = new HorizontalContainer();
		paginationPanel = new SimpleContainer();
		columnPanel = new TableColumnPanel<K, R>(dataProvider);

		mainContainer.setSpacing(TableConstants.PANEL_SPACING);
		mainContainer.setSize("100%", "100%");
		mainContainer.setHorizontalAlignment(VerticalContainer.ALIGN_CENTER);

		columnPanel.setStyleName(TableConstants.STYLE_TABLE_BODY);
		mainContainer.setCellHorizontalAlignment(columnPanel, VerticalContainer.ALIGN_LEFT);
		mainContainer.add(columnPanel);

		if (dataProvider.isRecordCountEnabled()) {
			SimpleContainer numRecordsPanel = new SimpleContainer();
			LabelWidget numRecordsLabel = new LabelWidget("");
			numRecordsLabel.setStyleName(TableConstants.STYLE_GENERAL_TEXT);
			numRecordsPanel.setWidget(numRecordsLabel);
			mainContainer.add(numRecordsPanel);
		}

		mainContainer.add(actionContainer);
		mainContainer.setCellHorizontalAlignment(actionContainer, VerticalContainer.ALIGN_LEFT);

		paginationPanel.setWidth("100%");
		mainContainer.add(paginationPanel);
		mainContainer.setCellHorizontalAlignment(paginationPanel, VerticalContainer.ALIGN_CENTER);

		setWidget(mainContainer);

		setSize("100%", "100%");
		setCornerStyleName(TableConstants.STYLE_TABLE_PANEL);
	}

	// ========================================================================
	// ===================== API
	// ========================================================================

	public HorizontalPanel getHeaderColPanelForColumn(Column<?, R> column) {
		return columnPanel.getHeaderColumnPanel(column);
	}

	public Label getHeaderLabelForColumn(Column<?, R> column) {
		return columnPanel.getHeaderColumnLabel(column);
	}

	/**
	 * Refresh will cause an asynchronous update of the current data
	 */
	public void refresh() {
		/* Clear the previous records, and request for the new records */
		dataProvider.getRecords(new AsyncCallback<List<R>>() {
			public void onFailure(Throwable caught) {
				Window.alert(caught.getLocalizedMessage());

				/* Clear all the records, notify listeners */
				columnPanel.updateRecords(new ArrayList<R>());
				notifyAllListeners();
			}

			public void onSuccess(List<R> result) {
				/* Display all the records and notify listeners */
				columnPanel.updateRecords(result);
				notifyAllListeners();
			}
		});
	}

	public void clearRecords() {
		columnPanel.clearRecords();
	}
	
	public SimpleContainer getPaginationPanel() {
		return paginationPanel;
	}

	public List<Column<?, R>> listVisibleColumns() {
		return columnPanel.listVisibleColumns();
	}

	public List<Widget> listWidgetsForVisibleColumn(Column<?, R> visibleColumn) {
		return columnPanel.listWidgetsForVisibleColumn(visibleColumn);
	}

	public R getRecordForWidget(Widget widget) {
		return columnPanel.getRecordForWidget(widget);
	}

	public HorizontalContainer getActionContainer() {
		return this.actionContainer;
	}
	
	public List<R> getRecords() {
		return columnPanel.listSelectedRecords();
	}

	public List<R> getSelectedRecords() {
		return columnPanel.listSelectedRecords();
	}
	
	public int getDataRowCount() {
		return columnPanel.getDataRowCount();
	}

	public void addListener(Listener listener) {
		if (listener != null && !listeners.contains(listener))
			listeners.add(listener);
	}

	public void removeListener(Listener listener) {
		listeners.remove(listener);
	}

	private void notifyAllListeners() {
		for (Listener listener : listeners)
			listener.updated();
	}

	public interface Listener {
		void updated();
	}
}
