/*
 * Created on Sep 14, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table;

import static com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants.STYLE_DATA_EVEN_ROW;
import static com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants.STYLE_DATA_ODD_ROW;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.CheckBox;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;

/**
 * @author Vijay Silva
 */
class TableColumnPanel<K, R extends Record<K>> extends FlexTable {

	/* Properties */
	private final DataProvider<K, R> dataProvider;
	private int colIndex;
	private List<R> records;
	private CheckBox<R> headerRowCheckbox;
	private List<CheckBox<R>> dataRowsCheckBoxes;
	private List<Column<?, R>> visibleColumns;
	private Map<Column<?, R>, List<Widget>> visibleColumnToWidgetListMap;
	private Map<Widget, R> widgetToRecordMap;
	private Map<Column<?, R>, Label> visibleColumnToHeaderLabel;
	private Map<Column<?, R>, HorizontalPanel> visibleColumnToHeaderColPanel;

	/**
	 * Constructor
	 * 
	 * @param dataProvider The data provider
	 */
	public TableColumnPanel(DataProvider<K, R> dataProvider) {
		this.dataProvider = dataProvider;
		initializeWidget();
	}

	// ========================================================================
	// ===================== WIDGET INITIALIZATION
	// ========================================================================

	/*
	 * Initialize the table widget
	 */
	private void initializeWidget() {
		/* Initialize the cache state */
		records = new ArrayList<R>();
		headerRowCheckbox = new CheckBox<R>();
		dataRowsCheckBoxes = new ArrayList<CheckBox<R>>();
		visibleColumns = new ArrayList<Column<?, R>>();
		visibleColumnToWidgetListMap = new HashMap<Column<?, R>, List<Widget>>();
		widgetToRecordMap = new HashMap<Widget, R>();
		visibleColumnToHeaderColPanel = new HashMap<Column<?, R>, HorizontalPanel>();
		visibleColumnToHeaderLabel = new HashMap<Column<?, R>, Label>();

		/* update size */
		setSize("100%", "100%");

		/* create header row */
		getRowFormatter().setStyleName(0, TableConstants.STYLE_HEADER_ROW);
		colIndex = 0;
		if (dataProvider.isCheckboxEnabled()) {
			headerRowCheckbox.addClickHandler(new ClickHandler() {
				public void onClick(ClickEvent event) {
					handleHeaderCheckBoxClick(event);
				}
			});

			setWidget(0, colIndex++, headerRowCheckbox);
		}

		/* Create header row columns */
		for (final Column<?, R> column : dataProvider.getColumns()) {
			addColumn(column);
		}
	}

	/* Add a column to the header row */
	private void addColumn(final Column<?, R> column) {
		visibleColumns.add(column);
		HorizontalPanel headerColPanel = new HorizontalPanel();
		headerColPanel.setSpacing(3);
		headerColPanel.setVerticalAlignment(HorizontalPanel.ALIGN_MIDDLE);
		Label headerLabel = new Label(column.getDisplayName());
		headerColPanel.add(headerLabel);

		headerLabel.setStyleName(TableConstants.STYLE_HEADER_NONSORTABLE_COL_TEXT);
		setWidget(0, colIndex++, headerColPanel);
		visibleColumnToHeaderLabel.put(column, headerLabel);
		visibleColumnToHeaderColPanel.put(column, headerColPanel);
	}

	// ========================================================================
	// ===================== API
	// ========================================================================

	public void clearRecords() {
		records.clear();
		headerRowCheckbox.setValue(false);
		visibleColumnToWidgetListMap.clear();
		widgetToRecordMap.clear();
		dataRowsCheckBoxes.clear();

		// clean out existing rows
		final int rowCount = getRowCount();
		for (int i = rowCount - 1; i > 0; --i)
			removeRow(i);
	}

	public void updateRecords(List<R> result) {
		clearRecords();

		/* Add the records to render */
		if (result != null)
			records.addAll(result);

		int rowIndex = 1;
		for (final R record : records) {
			String styleName = (rowIndex % 2 == 0) ? STYLE_DATA_EVEN_ROW : STYLE_DATA_ODD_ROW;
			getRowFormatter().setStyleName(rowIndex, styleName);
			int colIndex = 0;
			if (dataProvider.isCheckboxEnabled()) {
				final CheckBox<R> dataRowCheckBox = new CheckBox<R>();
				dataRowCheckBox.setRecord(record);
				dataRowCheckBox.addClickHandler(new ClickHandler() {
					public void onClick(ClickEvent event) {
						handleDataCheckBoxClick(dataRowCheckBox, event);
					}
				});
				dataRowsCheckBoxes.add(dataRowCheckBox);
				setWidget(rowIndex, colIndex++, dataRowCheckBox);
			}

			for (Column<?, R> field : visibleColumns) {
				final View<?> listView = field.getView(record);
				String viewStyleName = listView.getStyleName();
				if (viewStyleName == null)
					viewStyleName = TableConstants.STYLE_DATA_TEXT;
				Widget widget = listView.getWidget();
				widget.setStyleName(viewStyleName);
				setWidget(rowIndex, colIndex, widget);
				colIndex++;

				// add widget to list
				getInitializedWidgetList(field).add(widget);
				widgetToRecordMap.put(widget, record);
			}
			rowIndex++;
		}
	}

	/*
	 * Get the list of widgets for a column, or initialize if the list is not present
	 */
	private List<Widget> getInitializedWidgetList(Column<?, R> visibleColumn) {
		List<Widget> list = visibleColumnToWidgetListMap.get(visibleColumn);
		if (list == null) {
			list = new ArrayList<Widget>();
			visibleColumnToWidgetListMap.put(visibleColumn, list);
		}
		return list;
	}

	public List<Column<?, R>> listVisibleColumns() {
		return visibleColumns;
	}

	public List<Widget> listWidgetsForVisibleColumn(Column<?, R> column) {
		return visibleColumnToWidgetListMap.get(column);
	}

	public R getRecordForWidget(Widget widget) {
		return widgetToRecordMap.get(widget);
	}

	public List<R> getRecords() {
		List<R> visibleRecords = new ArrayList<R>();
		visibleRecords.addAll(this.records);
		return visibleRecords;
	}

	public List<R> listSelectedRecords() {
		List<R> records = new ArrayList<R>();
		for (CheckBox<R> checkBox : dataRowsCheckBoxes)
			if (checkBox.getValue())
				records.add(checkBox.getRecord());
		return records;
	}

	public int getDataRowCount() {
		return records.size();
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/* Handle the click event on the header check box */
	private void handleHeaderCheckBoxClick(ClickEvent event) {
		/* selecting all data row check boxes when header row check box is selected */
		Boolean headerValue = headerRowCheckbox.getValue();
		for (CheckBox<R> dataRowCB : dataRowsCheckBoxes) {
			dataRowCB.setValue(headerValue);
		}
	}

	/* Handle the click event on the data row check box */
	private void handleDataCheckBoxClick(CheckBox<R> source, ClickEvent event) {
		if (headerRowCheckbox.getValue()) {
			if (!source.getValue())
				headerRowCheckbox.setValue(false);
		}
	}

	// ========================================================================
	// ===================== ACCESSORS
	// ========================================================================

	/**
	 * Get the header column panel for the column
	 */
	public HorizontalPanel getHeaderColumnPanel(Column<?, R> column) {
		return visibleColumnToHeaderColPanel.get(column);
	}

	/**
	 * Get the header column label for the column
	 * 
	 * @return header column label
	 */
	public Label getHeaderColumnLabel(Column<?, R> column) {
		return visibleColumnToHeaderLabel.get(column);
	}

	/**
	 * Get the header row check box
	 * 
	 * @return The header row check box
	 */
	public CheckBox<R> getHeaderRowCheckBox() {
		return headerRowCheckbox;
	}
}
