/*
 * Created on Sep 14, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * The table list view panel
 * 
 * @author Vijay Silva
 */
class TableListViewPanel<K, R extends Record<K>> extends Composite {

	/* Properties */
	private final DataProvider<K, R> dataProvider;
	private TableColumnPanel<K, R> columnPanel;
	private SimpleContainer recordCountPanel;
	private LabelWidget numRecordsLabel;
	private HorizontalContainer actionContainer;
	private SimpleContainer paginationPanel;
	private List<Listener> listeners;
	private VerticalContainer mainContainer;

	/**
	 * Constructor
	 * 
	 * @param dataProvider The data provider
	 */
	public TableListViewPanel(DataProvider<K, R> dataProvider) {
		this.dataProvider = dataProvider;
		initializeWidget();
	}

	// ========================================================================
	// ===================== INITIALIZATION
	// ========================================================================

	/*
	 * Initialize the widget
	 */
	private void initializeWidget() {
		listeners = new ArrayList<Listener>();
		mainContainer = new VerticalContainer();

		/* create place holder panels even if the feature is not enabled */
		recordCountPanel = new SimpleContainer();
		numRecordsLabel = new LabelWidget("");
		recordCountPanel.setWidget(numRecordsLabel);
		actionContainer = new HorizontalContainer();
		paginationPanel = new SimpleContainer();
		columnPanel = new TableColumnPanel<K, R>(dataProvider);

		/* set style information for the list view and its sections */ 
		mainContainer.setStylePrimaryName(TableConstants.STYLE_TABLE_PANEL);
		mainContainer.setSpacing(TableConstants.PANEL_SPACING);
		recordCountPanel.setStylePrimaryName(TableConstants.STYLE_TABLE_RECORDCOUNT_PANEL);
		recordCountPanel.setVisible(dataProvider.isRecordCountEnabled());
		numRecordsLabel.setStyleName(TableConstants.STYLE_TABLE_RECORDCOUNT_LABEL);
		actionContainer.setStylePrimaryName(TableConstants.STYLE_TABLE_ACTION_PANEL);
		paginationPanel.setStylePrimaryName(TableConstants.STYLE_TABLE_PAGING_PANEL);

		/* add widgets to the panel */
		mainContainer.add(columnPanel);
		mainContainer.add(recordCountPanel);
		mainContainer.add(actionContainer);
		mainContainer.add(paginationPanel);

		initWidget(mainContainer);
	}

	// ========================================================================
	// ===================== API
	// ========================================================================

	public HorizontalPanel getHeaderColPanelForColumn(Column<?, R> column) {
		return columnPanel.getHeaderColumnPanel(column);
	}

	public Label getHeaderLabelForColumn(Column<?, R> column) {
		return columnPanel.getHeaderColumnLabel(column);
	}

	/**
	 * Refresh will cause an asynchronous update of the current data
	 */
	public void refresh() {
		/* Clear the previous records, and request for the new records */
		dataProvider.getRecords(new AsyncCallback<List<R>>() {
			public void onFailure(Throwable caught) {
				Window.alert(caught.getLocalizedMessage());

				/* Clear all the records, notify listeners */
				columnPanel.updateRecords(new ArrayList<R>());
				notifyAllListeners();
			}

			public void onSuccess(List<R> result) {
				/* Display all the records and notify listeners */
				columnPanel.updateRecords(result);
				notifyAllListeners();
			}
		});
	}

	public void clearRecords() {
		columnPanel.clearRecords();
	}
	
	public SimpleContainer getPaginationPanel() {
		return paginationPanel;
	}

	public List<Column<?, R>> listVisibleColumns() {
		return columnPanel.listVisibleColumns();
	}

	public List<Widget> listWidgetsForVisibleColumn(Column<?, R> visibleColumn) {
		return columnPanel.listWidgetsForVisibleColumn(visibleColumn);
	}

	public R getRecordForWidget(Widget widget) {
		return columnPanel.getRecordForWidget(widget);
	}

	public HorizontalContainer getActionContainer() {
		return this.actionContainer;
	}
	
	public List<R> getRecords() {
		return columnPanel.listSelectedRecords();
	}

	public List<R> getSelectedRecords() {
		return columnPanel.listSelectedRecords();
	}
	
	public int getDataRowCount() {
		return columnPanel.getDataRowCount();
	}

	public void addListener(Listener listener) {
		if (listener != null && !listeners.contains(listener))
			listeners.add(listener);
	}

	public void removeListener(Listener listener) {
		listeners.remove(listener);
	}

	private void notifyAllListeners() {
		for (Listener listener : listeners)
			listener.updated();
	}

	public interface Listener {
		void updated();
	}
}
