/*
 * Created on Feb 8, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.dnd;

import java.util.ArrayList;
import java.util.List;

import com.allen_sauer.gwt.dnd.client.DragContext;
import com.allen_sauer.gwt.dnd.client.drop.AbstractPositioningDropController;
import com.allen_sauer.gwt.dnd.client.util.CoordinateLocation;
import com.allen_sauer.gwt.dnd.client.util.DOMUtil;
import com.allen_sauer.gwt.dnd.client.util.LocationWidgetComparator;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.IndexedPanel;
import com.google.gwt.user.client.ui.Widget;

/**
 * Drop Controller for handling drop events triggered using the {@link TableRowDragController} to
 * support moving of table rows using drag and drop.
 * 
 * @author Vijay Silva
 */
public class TableRowMoveController extends AbstractPositioningDropController {

	/* Event Handlers */
	private List<TableRowMoveHandler> handlers = new ArrayList<TableRowMoveHandler>();
	private final IndexedTablePanel tableWrapper;

	/**
	 * Constructor
	 * 
	 * @param table The table that is supporting the row move operations using drag and drop.
	 */
	public TableRowMoveController(FlexTable table) {
		super(table);
		this.tableWrapper = new IndexedTablePanel(table);
	}

	/*
	 * Overriding the signature of 'getDropTarget' to indicate that the drop target must be a table
	 * 
	 * @see com.allen_sauer.gwt.dnd.client.drop.AbstractDropController#getDropTarget()
	 */
	@Override
	public FlexTable getDropTarget() {
		return (FlexTable) super.getDropTarget();
	}

	/**
	 * Handle the drop action by moving the table row to new location.
	 */
	@Override
	public void onDrop(DragContext context) {
		/* Verify if the drag controller is of the expected type */
		if (!(context.dragController instanceof TableRowDragController)) {
			return;
		}

		/* Verify that row being dragged is being dropped in the same table */
		TableRowDragController dragController = (TableRowDragController) context.dragController;
		FlexTable table = getDropTarget();
		if (!table.equals(dragController.getTable())) {
			return;
		}

		/* Determine the index of the row being dragged, and the index of the drop location */
		int index = dragController.getDraggedRowIndex();
		int targetIndex = getTargetRowIndex(context);
		if (!isValidIndex(index) || !isValidIndex(targetIndex))
			return;

		/* Fire the before move event */
		TableRowBeforeMoveEvent event = new TableRowBeforeMoveEvent(table, index, targetIndex);
		for (TableRowMoveHandler handler : this.handlers) {
			handler.onBeforeRowMove(event);
			if (event.isCanceled())
				return;
		}

		/* Move the row */
		moveRow(index, targetIndex);

		/* Fire the move event */
		TableRowMoveEvent moveEvent = new TableRowMoveEvent(table, index, targetIndex);
		for (TableRowMoveHandler handler : this.handlers) {
			handler.onRowMove(moveEvent);
		}

		super.onDrop(context);
	}

	/**
	 * Register a new handler for table row move events
	 * 
	 * @param handler The event handler
	 */
	public void addTableRowMoveHandler(TableRowMoveHandler handler) {
		if (handler != null && !handlers.contains(handler))
			handlers.add(handler);
	}

	/**
	 * Remove a previously registered handler for table row move events
	 * 
	 * @param handler The event handler
	 */
	public void removeTableRowMoveHandler(TableRowMoveHandler handler) {
		handlers.remove(handler);
	}

	/*
	 * Get the row index that is the drop target
	 */
	private int getTargetRowIndex(DragContext context) {
		CoordinateLocation location = new CoordinateLocation(context.mouseX, context.mouseY);
		LocationWidgetComparator comparator = LocationWidgetComparator.BOTTOM_HALF_COMPARATOR;
		int targetIndex = DOMUtil.findIntersect(tableWrapper, location, comparator) - 1;
		return targetIndex;
	}

	/*
	 * Determine if the row index is valid for the table
	 */
	private boolean isValidIndex(int rowIndex) {
		return (rowIndex >= 0 && rowIndex < getDropTarget().getRowCount());
	}

	/*
	 * Move the table row from the source index to the target index
	 */
	private void moveRow(int source, int target) {
		FlexTable table = getDropTarget();

		/* no change in index, no move required */
		if (target == source)
			return;

		/* Since target is after source, increment target since source will be removed */
		if (target > source)
			target++;

		table.insertRow(target);

		/* If new row is before source, the source index has increased by 1 */
		if (target < source)
			source++;

		/* Copy the source to the target */
		for (int column = 0; column < table.getCellCount(source); column++) {
			Widget w = table.getWidget(source, column);
			if (w != null) {
				table.setWidget(target, column, w);
			} else {
				HTML html = new HTML(table.getHTML(source, column));
				table.setWidget(target, column, html);
			}

			/* Copy all styles and span for the cell */
			String style = table.getCellFormatter().getStyleName(source, column);
			table.getCellFormatter().setStyleName(target, column, style);
			int columnSpan = table.getFlexCellFormatter().getColSpan(source, column);
			table.getFlexCellFormatter().setColSpan(target, column, columnSpan);
		}

		/* Copy row style to target row */
		String rowStyle = table.getRowFormatter().getStyleName(source);
		table.getRowFormatter().setStyleName(target, rowStyle);

		/* Remove the source row completing the 'move' */
		table.removeRow(source);
	}

	/*
	 * Wrapper widget to allow indexed access of the table rows
	 */
	private static final class IndexedTablePanel implements IndexedPanel {

		private final FlexTable table;

		public IndexedTablePanel(FlexTable table) {
			this.table = table;
		}

		public Widget getWidget(int index) {
			return table.getWidget(index, 0);
		}

		public int getWidgetCount() {
			return table.getRowCount();
		}

		public int getWidgetIndex(Widget child) {
			throw new UnsupportedOperationException();
		}

		public boolean remove(int index) {
			throw new UnsupportedOperationException();
		}
	};
}
