/*
 * Created on Jun 3, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.datatype;

import java.util.Date;

import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.event.dom.client.HasClickHandlers;
import com.google.gwt.event.dom.client.HasKeyDownHandlers;
import com.google.gwt.event.dom.client.HasKeyPressHandlers;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.HasMouseDownHandlers;
import com.google.gwt.event.dom.client.HasMouseMoveHandlers;
import com.google.gwt.event.dom.client.HasMouseOutHandlers;
import com.google.gwt.event.dom.client.HasMouseOverHandlers;
import com.google.gwt.event.dom.client.HasMouseUpHandlers;
import com.google.gwt.event.dom.client.HasMouseWheelHandlers;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.dom.client.MouseWheelEvent;
import com.google.gwt.event.dom.client.MouseWheelHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DeferredCommand;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.datepicker.client.DateBox;
import com.google.gwt.user.datepicker.client.DateBox.Format;
import com.google.gwt.user.datepicker.client.DatePicker;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.DateBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.DatePickerWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IDateConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.ValueFormatException;

/**
 * A Date Box widget
 * 
 * @author Vijay Silva
 */
public class TypedDateBoxWidget extends DataTypeWidget<String> implements HasClickHandlers,
        HasKeyPressHandlers, HasKeyUpHandlers, HasKeyDownHandlers, HasMouseDownHandlers,
        HasMouseMoveHandlers, HasMouseOutHandlers, HasMouseOverHandlers, HasMouseUpHandlers,
        HasMouseWheelHandlers {

	/* The widget displaying the value */
	private DateBoxWidget widget;
	private Command updateCommand = new UpdateCommand();
	private IDateConverter dateConverter;

	/**
	 * Create a new date box widget that uses string parsing using the provided date pattern
	 * 
	 * @param datePattern The date pattern
	 * @param dateConverter The converter to transforms the display date text to the intermediate
	 *        date text value expected by the model
	 */
	public TypedDateBoxWidget(IDateConverter dateConverter) {
		this(new DatePickerWidget(), null, dateConverter);
	}

	/**
	 * Create a new date box widget.
	 * 
	 * @param date the default date.
	 * @param picker the picker to drop down from the date box
	 * @param format to use to parse and format dates
	 * @param dateConverter The converter to transforms the display date text to the intermediate
	 *        date text value expected by the model
	 */
	public TypedDateBoxWidget(DatePicker picker, Date date, IDateConverter dateConverter) {
		widget = new DateBoxWidget(picker, date, new StrictFormat(dateConverter));
		this.dateConverter = dateConverter;
		initializeWidget();
	}

	/*
	 * Perform any widget initialization
	 */
	private void initializeWidget() {
		this.initWidget(widget);
		EventHandler handler = new EventHandler();
		widget.addValueChangeHandler(handler);
	}

	// ========================================================================
	// ===================== EVENT HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addClickHandler(ClickHandler handler) {
		return addDomHandler(handler, ClickEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyPressHandler(KeyPressHandler handler) {
		return addDomHandler(handler, KeyPressEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyUpHandler(KeyUpHandler handler) {
		return addDomHandler(handler, KeyUpEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyDownHandler(KeyDownHandler handler) {
		return addDomHandler(handler, KeyDownEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseDownHandler(MouseDownHandler handler) {
		return addDomHandler(handler, MouseDownEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseMoveHandler(MouseMoveHandler handler) {
		return addDomHandler(handler, MouseMoveEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOutHandler(MouseOutHandler handler) {
		return addDomHandler(handler, MouseOutEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseOverHandler(MouseOverHandler handler) {
		return addDomHandler(handler, MouseOverEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseUpHandler(MouseUpHandler handler) {
		return addDomHandler(handler, MouseUpEvent.getType());
	}

	@Override
	public HandlerRegistration addMouseWheelHandler(MouseWheelHandler handler) {
		return addDomHandler(handler, MouseWheelEvent.getType());
	}

	// ========================================================================
	// ===================== HAS VALUE
	// ========================================================================

	/**
	 * Gets the typed value, or null if there was an error parsing the value
	 */
	@Override
	public String getValue() {
		return dateConverter.getInputValue(widget.getValue());
	}

	@Override
	public void setValue(String value) {
		this.setValue(value, false);
	}

	@Override
	public void setValue(String value, boolean fireEvents) {
		setTextValue(value, fireEvents);
	}

	@Override
	public HandlerRegistration addValueChangeHandler(ValueChangeHandler<String> handler) {
		return this.addHandler(handler, ValueChangeEvent.getType());
	}

	// ========================================================================
	// ===================== FOCUSABLE
	// ========================================================================

	/**
	 * @see DateBoxWidget#getTabIndex()
	 */
	@Override
	public int getTabIndex() {
		return widget.getTabIndex();
	}

	/**
	 * @see DateBoxWidget#setTabIndex(int)
	 */
	@Override
	public void setTabIndex(int index) {
		widget.setTabIndex(index);
	}

	/**
	 * @see DateBoxWidget#setAccessKey(char)
	 */
	@Override
	public void setAccessKey(char key) {
		widget.setAccessKey(key);
	}

	/**
	 * @see DateBoxWidget#setFocus(boolean)
	 */
	@Override
	public void setFocus(boolean focused) {
		widget.setFocus(focused);
	}

	// ========================================================================
	// ===================== HAS ALL FOCUS HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addFocusHandler(FocusHandler handler) {
		return this.addDomHandler(handler, FocusEvent.getType());
	}

	@Override
	public HandlerRegistration addBlurHandler(BlurHandler handler) {
		return this.addDomHandler(handler, BlurEvent.getType());
	}

	// ========================================================================
	// ===================== DATA TYPE WIDGET ABSTRACT METHODS
	// ========================================================================

	@Override
	public boolean isValidValue() {
		boolean valid = false;

		try {
			dateConverter.getTypedValue(widget.getTextBox().getText());
			valid = true;
		} catch (ValueFormatException vfe) {
		}

		return valid;
	}

	@Override
	public String getTextValue() {
		return this.getTextBox().getText();
	}

	@Override
	public void setTextValue(String value) {
		setTextValue(value, false);
	}

	/*
	 * Set the value to display for the widget. Updates the model of the date picker, the text in
	 * the text box and fires events if required. Also updates the widget state.
	 */
	private void setTextValue(String value, boolean fireEvents) {
		try {
			Date date = dateConverter.getDateForInputValue(value);
			widget.setValue(date, fireEvents);
		} catch (ValueFormatException e) {
			/* Update the widget model and text */
			widget.setValue(null, fireEvents);
			widget.getTextBox().setText(value);
		}

		/* Update the widget state */
		this.updateWidgetState();
	}

	@Override
	public boolean isReadOnly() {
		return !widget.getTextBox().isEnabled();
	}

	@Override
	public void setReadOnly(boolean readOnly) {
		widget.getTextBox().setReadOnly(readOnly);
		widget.setEnabled(!readOnly);
	}

	@Override
	public boolean isEnabled() {
		return widget.getTextBox().isEnabled();
	}

	@Override
	public void setEnabled(boolean enabled) {
		widget.setEnabled(enabled);
	}

	/**
	 * Add or remove the error style based on the error state
	 */
	@Override
	protected void updateStyle() {
		if (this.isInErrorState())
			addStyleDependentName(ERROR_STYLE);
		else
			removeStyleDependentName(ERROR_STYLE);
	}

	@Override
	public String getCurrentToolTip() {
		return this.getTitle();
	}

	@Override
	protected void setCurrentToolTip(String toolTip) {
		this.setTitle(toolTip);
	}

	// ========================================================================
	// ===================== DATE BOX WIDGET METHODS
	// ========================================================================

	/**
	 * @see DateBoxWidget#getCursorPos()
	 */
	public int getCursorPos() {
		return widget.getCursorPos();
	}

	/**
	 * @see DateBoxWidget#getDatePicker()
	 */
	public DatePicker getDatePicker() {
		return widget.getDatePicker();
	}

	/**
	 * @see DateBoxWidget#setFormat(Format)
	 */
	public Format getFormat() {
		return widget.getFormat();
	}

	/**
	 * @see DateBoxWidget#getTextBox()
	 */
	public TextBox getTextBox() {
		return widget.getTextBox();
	}

	/**
	 * @see DateBoxWidget#hideDatePicker()
	 */
	public void hideDatePicker() {
		widget.hideDatePicker();
	}

	/**
	 * @see DateBoxWidget#isDatePickerShowing()
	 */
	public boolean isDatePickerShowing() {
		return widget.isDatePickerShowing();
	}

	/**
	 * @see DateBoxWidget#showDatePicker()
	 */
	public void showDatePicker() {
		widget.showDatePicker();
	}

	// ========================================================================
	// ===================== DATE BOX WIDGET FORMAT
	// ========================================================================

	/**
	 * Format for the date box widget to use that ensures strict formatting of the date string
	 */
	private class StrictFormat implements Format {
		private final IDateConverter dateConverter;

		/**
		 * Constructor
		 */
		public StrictFormat(IDateConverter converter) {
			this.dateConverter = converter;
		}

		@Override
		public String format(DateBox box, Date date) {
			String value = (date != null) ? dateConverter.getDisplayValue(date) : "";
			DeferredCommand.addCommand(updateCommand);
			return value;
		}

		@Override
		public Date parse(DateBox dateBox, String dateText, boolean reportError) {
			if (reportError)
				updateWidgetState();

			Date date = null;
			try {
				date = dateConverter.getDateForDisplayValue(dateText);
			} catch (Exception e) {
			}

			if (date == null && reportError)
				ValueChangeEvent.fire(TypedDateBoxWidget.this, null);

			return date;
		}

		@Override
		public void reset(DateBox dateBox, boolean abandon) {
		}
	}

	// ========================================================================
	// ===================== EVENT DELEGATOR
	// ========================================================================

	/*
	 * Inner class that delegates the value change events for this widget
	 */
	private class EventHandler implements ValueChangeHandler<Date> {

		@Override
		public void onValueChange(ValueChangeEvent<Date> event) {
			Date dateValue = event.getValue();
			String value = dateConverter.getInputValue(dateValue);
			ValueChangeEvent.fire(TypedDateBoxWidget.this, value);
		}
	}

	// ========================================================================
	// ===================== COMMAND TO UPDATE WIDGET STATE
	// ========================================================================

	/*
	 * Command that updates the widget state
	 */
	private class UpdateCommand implements Command {

		@Override
		public void execute() {
			updateWidgetState();
		}
	}
}
