package com.tandbergtv.neptune.widgettoolkit.client.application;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.ShowAboutEvent;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.ShowAboutHandler;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.ShowUserGuideEvent;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.ShowUserGuideHandler;
import com.tandbergtv.neptune.widgettoolkit.client.component.Component;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventListenerRegistry;
import com.tandbergtv.neptune.widgettoolkit.client.event.EventSink;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;

/**
 * Application Widget that maintains a singleton for global client access. Maintains any shared
 * application state that the various component widgets can query.
 * 
 * @author trybak
 * @author Vijay Silva
 */
public class NeptuneApplication extends Composite {

	/* The Neptune application - load using deferred binding */
	private static NeptuneApplication INSTANCE = GWT.create(NeptuneApplication.class);

	/* Constants */
	private static final String STYLE_NAME = "nwt-Application";

	/* Application State */
	private List<Component> components;
	private NeptuneSecurity security;
	private Map<String, String> componentProperties;
	private EventListenerRegistry eventRegistry;
	private EventSink eventSink;
	private ServiceLoader serviceLoader;
	private String applicationTitle;
	private String applicationSubtitle;
	private String applicationVersion;
	private Map<Class<? extends Throwable>, Runnable> errorHandlerRegistry;
	private boolean initialized = false;

	/* Widgets */
	private SimpleContainer container;

	// ========================================================================
	// ===================== CONSTRUCTOR
	// ========================================================================

	/**
	 * Constructor
	 */
	public NeptuneApplication() {
		this.container = new SimpleContainer();
		this.container.setStylePrimaryName(STYLE_NAME);
		this.initWidget(container);

		this.errorHandlerRegistry = new HashMap<Class<? extends Throwable>, Runnable>();
	}

	// ========================================================================
	// ===================== APPLICATION ACCESSOR
	// ========================================================================

	/**
	 * Get the Neptune Application
	 * 
	 * @return The Neptune Application
	 */
	public static NeptuneApplication getApplication() {
		return INSTANCE;
	}

	// ========================================================================
	// ===================== STATE ACCESSORS
	// ========================================================================

	/**
	 * Get the application components
	 * 
	 * @return the components
	 */
	public List<Component> getComponents() {
		return this.components;
	}

	/**
	 * Set the application components. The components must be set before the application is
	 * initialized, and the list cannot be changed after initialization.
	 * 
	 * @param components The application components
	 */
	public void setComponents(List<Component> components) {
		if (initialized) {
			String msg = "The application has been initialized, cannot set components.";
			throw new RuntimeException(msg);
		}

		this.components = components;
	}

	/**
	 * Get the component instance given the component class. The class must exactly match the class
	 * of the component instance.
	 * 
	 * @param <T> The type of the component
	 * @param componentClass The class for the component
	 * @return The component with matching class, or null if no component matches the class.
	 */
	@SuppressWarnings("unchecked")
	public <T extends Component> T getComponent(Class<T> componentClass) {
		if (this.components == null)
			return null;

		for (Component component : this.components) {
			if (component.getClass().equals(componentClass)) {
				return (T) component;
			}
		}

		return null;
	}

	/**
	 * Get the current user security
	 * 
	 * @return the user security or null if no user is logged in
	 */
	public NeptuneSecurity getSecurity() {
		return this.security;
	}

	/**
	 * Set the current user security, or null to clear the current security
	 * 
	 * @param security The security to set (null to clear)
	 */
	protected void setSecurity(NeptuneSecurity security) {
		this.security = security;
	}

	/**
	 * Get the event registry
	 * 
	 * @return The event registry
	 */
	public EventListenerRegistry getEventRegistry() {
		return this.eventRegistry;
	}

	/**
	 * Set the event registry
	 * 
	 * @return The event registry
	 */
	protected void setEventRegistry(EventListenerRegistry eventRegistry) {
		this.eventRegistry = eventRegistry;
	}

	/**
	 * Get the event sink
	 * 
	 * @return The event sink
	 */
	public EventSink getEventSink() {
		return this.eventSink;
	}

	/**
	 * Set the event sink
	 * 
	 * @return The event sink
	 */
	protected void setEventSink(EventSink eventSink) {
		this.eventSink = eventSink;
	}

	/**
	 * Gets the service loader.
	 * 
	 * @return The service loader
	 */
	public ServiceLoader getServiceLoader() {
		return this.serviceLoader;
	}

	/**
	 * Set the service loader
	 * 
	 * @param serviceLoader The service loader
	 */
	protected void setServiceLoader(ServiceLoader serviceLoader) {
		this.serviceLoader = serviceLoader;
	}

	/**
	 * Get the component properties
	 * 
	 * @return The component properties
	 */
	public Map<String, String> getComponentProperties() {
		return this.componentProperties;
	}

	/**
	 * Set the component properties
	 * 
	 * @return The component properties
	 */
	protected void setComponentProperties(Map<String, String> componentProperties) {
		this.componentProperties = componentProperties;
	}

	/**
	 * Get the application title
	 * 
	 * @return The application title
	 */
	public String getApplicationTitle() {
		return this.applicationTitle;
	}

	/**
	 * Set the application title. Cannot change the title after the application is initialized
	 * 
	 * @param title The application title.
	 */
	public void setApplicationTitle(String title) {
		if (initialized) {
			String msg = "The application has been initialized, cannot set application title.";
			throw new RuntimeException(msg);
		}

		this.applicationTitle = title;
	}

	/**
	 * Get the application sub-title
	 * 
	 * @return The application sub-title
	 */
	public String getApplicationSubtitle() {
		return this.applicationSubtitle;
	}

	/**
	 * Set the application sub-title. Cannot change sub-title after application is initialized
	 * 
	 * @return The application sub-title
	 */
	public void setApplicationSubtitle(String subtitle) {
		if (initialized) {
			String msg = "The application has been initialized, cannot set application sub-title.";
			throw new RuntimeException(msg);
		}

		this.applicationSubtitle = subtitle;
	}

	/**
	 * Set the application version
	 * 
	 * @return The application version
	 */
	public String getApplicationVersion() {
		return this.applicationVersion;
	}

	/**
	 * Set the application version. Cannot change version after application is initialized
	 * 
	 * @param version The application version
	 */
	public void setApplicationVersion(String version) {
		if (initialized) {
			String msg = "The application has been initialized, cannot set application version.";
			throw new RuntimeException(msg);
		}

		this.applicationVersion = version;
	}

	/**
	 * Determine if the application is initialized
	 * 
	 * @return true if initialize, false otherwise
	 */
	public boolean isInitialized() {
		return initialized;
	}

	// ========================================================================
	// ===================== WIDGET ACCESSORS
	// ========================================================================

	/**
	 * Get the application container
	 * 
	 * @return the application container
	 */
	protected SimpleContainer getApplicationContainer() {
		return this.container;
	}

	// ========================================================================
	// ===================== ERROR HANDLING
	// ========================================================================

	/**
	 * Performs error handling for any errors resulting from asynchronous calls made to the server.
	 * If the exception is not handled by the component, the application will catch and handle this
	 * error.
	 * 
	 * @param caught The error resulting from a server request
	 * @return true if the error is handled, false otherwise
	 */
	public boolean handleThrowable(Throwable caught) {
		Runnable runnable = this.errorHandlerRegistry.get(caught.getClass());
		if (runnable == null)
			return false;

		runnable.run();
		return true;
	}

	/**
	 * Add a error handler for any errors that match the given error type
	 * 
	 * @param errorType The error type (throwable class)
	 * @param errorHandler The handler to execute when handling error of given error type
	 */
	protected void registerErrorHandler(Class<? extends Throwable> errorType, Runnable errorHandler) {
		this.errorHandlerRegistry.put(errorType, errorHandler);
	}

	// ========================================================================
	// ===================== APPLICATION INITIALIZATION / DESTRUCTION
	// ========================================================================

	/**
	 * Initialize the application after setting required application state. To override the
	 * initialization behavior, do not override this method. Instead override
	 * {@link NeptuneApplication#initializeApplication()} method.
	 */
	public final void initialize() {
		if (this.initialized)
			return;

		initializeApplication(new AsyncCallback<Void>() {
			public void onFailure(Throwable caught) {
				Window.alert("Failed to initialize the application.");
			};

			public void onSuccess(Void result) {
				initialized = true;
			};
		});
	}

	/**
	 * Perform any custom initialization of the application
	 */
	protected void initializeApplication(AsyncCallback<Void> callback) {
	}

	/**
	 * Destroy the application. To override behavior, do not override this method. Instead override
	 * {@link NeptuneApplication#destroyApplication()} method.
	 */
	public final void destroy() {
		if (!this.initialized)
			return;

		destroyApplication();
		this.initialized = false;
	}

	/**
	 * Perform any custom destruction of the application.
	 */
	protected void destroyApplication() {
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/**
	 * Add event handler for the 'show about' application event
	 * 
	 * @param handler The event handler
	 */
	public HandlerRegistration addShowAboutHandler(ShowAboutHandler handler) {
		return addHandler(handler, ShowAboutEvent.getType());
	}

	/**
	 * Add event handler for the 'show user guide' application event
	 * 
	 * @param handler The event handler
	 */
	public HandlerRegistration addShowUserGuideHandler(ShowUserGuideHandler handler) {
		return addHandler(handler, ShowUserGuideEvent.getType());
	}
	
	public void hideMenu(){
	};
}
