/*

 * Created on Jul 19, 2012
 * 
 * (C) Copyright Ericsson Television Inc.
 */
package com.tandbergtv.neptune.widgettoolkit.client.widget.networkelements;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Timer;
import com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetMessages;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;

/**
 * Displays the current redundancy status. It goes to the server N seconds to get the
 * application server cluster status, and then updates the button color/label.
 * 
 * @author Francisco B. S. Neto
 */
public class RedundancyStatusButton extends ButtonWidget {

	/**
	 * The default rate (in seconds) at which the time is synchronized with the server
	 */
	public static final int DEFAULT_SYNCHRONIZATION_RATE = 60; 

	/* The RPC service */
	private final INetworkElementsUIServiceAsync service = GWT.create(INetworkElementsUIService.class);

	/** Styles for the distinct statuses */
	private static final String ACTIVE_STYLE_NAME = "nfw-clusterIndicator-active";
	private static final String INACTIVE_STYLE_NAME = "nfw-clusterIndicator-inactive";
	private static final String FAILED_STYLE_NAME = "nfw-clusterIndicator-fail";

	/* The timer to trigger updates of the widget state */
	private final Timer timer;

	private int synchronizationRate = DEFAULT_SYNCHRONIZATION_RATE;

	private final NeptuneWidgetMessages messages;

	private NetworkElementStatus redundancyStatus;
	
	/**
	 * Constructor. Use the default pattern for display
	 */
	public RedundancyStatusButton() {
		/* Initialize the timer */
		this.timer = new Timer() {
			@Override
			public void run() {
				updateReduncancyStatus();
			}
		};
		setVisible(false);
		messages = GWT.create(NeptuneWidgetMessages.class);
	}


	// ========================================================================
	// ===================== WIDGET LOAD / UNLOAD AND ACCESS
	// ========================================================================

	/*
	 * Start the timer when the widget is attached to the DOM to make server calls
	 */
	@Override
	protected void onLoad() {
		super.onLoad();

		updateReduncancyStatus();
		this.timer.scheduleRepeating(synchronizationRate * 1000);
	}

	/*
	 * Stop the timer (ensuring that no server calls are made)
	 */
	@Override
	protected void onUnload() {
		/* Stop the timer and clear the timer state */
		this.timer.cancel();

		super.onUnload();
	}

	/**
	 * Get the rate (in seconds) at which the displayed time is synchronized with the server
	 * 
	 * @return The refresh rate at which to synchronize the time with the server (in seconds)
	 */
	public int getSynchronizationRate() {
		return this.synchronizationRate;
	}

	/**
	 * Set the rate (in seconds) at which the displayed time is synchronized with the server
	 * 
	 * @seconds The refresh rate at which to synchronize the time with the server (in seconds)
	 */
	public void setSynchronizationRate(int seconds) {
		this.synchronizationRate = seconds;
	}

	/*
	 * Update the state with the current time, and update the view
	 */
	private void updateReduncancyStatus() {
		service.getRedundancyStatus(new NeptuneAsyncCallback<NetworkElementStatus>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				// do nothing, keep last state
			}

			@Override
			public void onNeptuneSuccess(NetworkElementStatus result) {
				setRedundancyStatus(result);
			}
		});
	}

	/*
	 * Set the current time
	 */
	private void setRedundancyStatus(NetworkElementStatus redundancyStatus) {
		this.redundancyStatus = redundancyStatus;
		updateWidget();
	}

	public NetworkElementStatus getRedundancyStatus() {
		return redundancyStatus;
	}
	
	/*
	 * Update the text in this label to show the current time
	 */
	private void updateWidget() {
		if (redundancyStatus.isDisabled()) {
			setVisible(false);
		} else {
			String text = null;
			String styleName = null;
			setVisible(true);

			setStyleDependentName(ACTIVE_STYLE_NAME, false);
			setStyleDependentName(INACTIVE_STYLE_NAME, false);
			setStyleDependentName(FAILED_STYLE_NAME, false);
			
			if (redundancyStatus.isActive()) {
				styleName = ACTIVE_STYLE_NAME;
				text = messages.redundancyStatusActive();
			} else if (redundancyStatus.isInactive()) {
				styleName = INACTIVE_STYLE_NAME;
				text = messages.redundancyStatusInactive();
			} else if (redundancyStatus.isFailed()) {
				styleName = FAILED_STYLE_NAME;
				text = messages.redundancyStatusFailed();
			} 
			setStyleDependentName(styleName, true);
			setText(text);
		}

	}
}
