/*
 * Created on Jan 15, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.neptune.ui.framework.client.application;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.PasswordTextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * The default Login Form used by the Neptune Application
 * 
 * @author Vijay Silva
 */
public class LoginForm extends Composite {

	/* Styles */
	private static final String STYLE_NAME = "nfw-LoginForm";
	private static final String STYLE_LOGO_CONTAINER = "nfw-LoginForm-logo";
	private static final String STYLE_CONTENT_SECTION = "nfw-LoginForm-content";
	private static final String STYLE_CONTENT_LABEL = "nfw-LoginForm-label";
	private static final String STYLE_CONTENT_LABEL_TITLE = "nfw-LoginForm-label-title";
	private static final String STYLE_CONTENT_LABEL_VERSION = "nfw-LoginForm-label-version";
	private static final String STYLE_CONTENT_LABEL_COPYRIGHT = "nfw-LoginForm-label-copyright";
	private static final String STYLE_LOGINBOX = "nfw-LoginForm-loginBox";
	private static final String STYLE_LOGINBOX_LABEL = "nfw-LoginForm-loginBoxLabel";
	private static final String STYLE_LOGINBOX_TEXTBOX = "nfw-LoginForm-loginBoxTextBox";
	private static final String STYLE_LOGINBOX_TEXTBOX_USER = "nfw-LoginForm-loginBoxTextBox_user";
	private static final String STYLE_LOGINBOX_TEXTBOX_PASSWORD = "nfw-LoginForm-loginBoxTextBox_password";

	/* Constants */
	private static final String IMAGE_LOGO_PATH = "neptune_framework/images/loginform-logo.png";

	/* Widgets / Messages / Handlers */
	private final ApplicationMessages messages;
	private HTMLWidget titleLabel, versionLabel;
	private TextBoxWidget userTextBox;
	private PasswordTextBoxWidget passwordTextBox;
	private ButtonWidget loginButton;
	private List<LoginHandler> handlers = new ArrayList<LoginHandler>();
	private MessageArea messageArea;
	private VerticalContainer loginContainer;
	private boolean applicationLicensed;

	/**
	 * Constructor
	 */
	public LoginForm() {
		messages = GWT.create(ApplicationMessages.class);
		initializeWidget();
	}

	/*
	 * Initialize the widget / layout
	 */
	private void initializeWidget() {
		SimpleContainer mainContainer = new SimpleContainer();
		mainContainer.setStylePrimaryName(STYLE_NAME);

		VerticalContainer container = new VerticalContainer();
		mainContainer.setWidget(container);

		/* Build and add the logo image */
		SimpleContainer imageContainer = new SimpleContainer();
		imageContainer.setStylePrimaryName(STYLE_LOGO_CONTAINER);
		ImageWidget image = new ImageWidget(IMAGE_LOGO_PATH);
		imageContainer.setWidget(image);
		container.add(imageContainer);

		/* Build and add the application title section */
		loginContainer = new VerticalContainer();
		loginContainer.setStylePrimaryName(STYLE_CONTENT_SECTION);

		SimpleContainer messageContainer = new SimpleContainer();
		messageArea = new MessageArea();
		messageArea.setVisible(false);
		messageContainer.add(messageArea);
		container.add(messageContainer);

		/* Build the application name label */
		titleLabel = new HTMLWidget();
		titleLabel.addStyleName(STYLE_CONTENT_LABEL);
		titleLabel.addStyleName(STYLE_CONTENT_LABEL_TITLE);
		loginContainer.add(titleLabel);
		
		/* Build the application version label */
		versionLabel = new HTMLWidget();
		versionLabel.addStyleName(STYLE_CONTENT_LABEL);
		versionLabel.addStyleName(STYLE_CONTENT_LABEL_VERSION);
		loginContainer.add(versionLabel);

		
		SimpleContainer loginBox = new SimpleContainer();
		loginBox.setStylePrimaryName(STYLE_LOGINBOX);

		Widget loginFormBox = buildLoginFormBox();
		loginBox.add(loginFormBox);
		loginContainer.add(loginBox);
		
		/* Build the copyright label */
		HTMLWidget copyrightLabel = new HTMLWidget(messages.loginFormCopyrightLabel());
		copyrightLabel.addStyleName(STYLE_CONTENT_LABEL);
		copyrightLabel.addStyleName(STYLE_CONTENT_LABEL_COPYRIGHT);
		loginContainer.add(copyrightLabel);
		container.add(loginContainer);

		this.initWidget(mainContainer);
	}

	/* Build the box containing the login form */
	private Widget buildLoginFormBox() {
		FlexTableContainer table = new FlexTableContainer();

		/* Add User Name */
		table.setWidget(0, 0, createFormLabel(messages.loginFormUserNameLabel()));
		userTextBox = new TextBoxWidget();
		userTextBox.addStyleName(STYLE_LOGINBOX_TEXTBOX);
		userTextBox.addStyleName(STYLE_LOGINBOX_TEXTBOX_USER);
		table.setWidget(0, 1, userTextBox);

		/* Add Password */
		table.setWidget(1, 0, createFormLabel(messages.loginFormPasswordLabel()));
		passwordTextBox = new PasswordTextBoxWidget();
		passwordTextBox.addStyleName(STYLE_LOGINBOX_TEXTBOX);
		passwordTextBox.addStyleName(STYLE_LOGINBOX_TEXTBOX_PASSWORD);
		passwordTextBox.addKeyPressHandler(new KeyPressHandler() {
			@Override
			public void onKeyPress(KeyPressEvent event) {
				handlePasswordBoxKeyPress(event);
			}
		});

		table.setWidget(1, 1, passwordTextBox);

		/* Add Login Button */
		loginButton = new ButtonWidget(messages.loginFormLoginButtonLabel());
		loginButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				clearErrorMessages();
				handleLoginButtonClick(event);
			}
		});
		table.setWidget(2, 1, loginButton);

		return table;
	}

	/* Builds the labels used in the box section of login form */
	private Widget createFormLabel(String text) {
		LabelWidget label = new LabelWidget(text);
		label.addStyleName(STYLE_LOGINBOX_LABEL);
		return label;
	}

	// ========================================================================
	// ===================== FOCUS HANDLING
	// ========================================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		setFocus();
	}

	// ========================================================================
	// ===================== API
	// ========================================================================

	/**
	 * Set the focus on the first widget in the login form - the user name box
	 */
	public void setFocus() {
		userTextBox.setCursorPos(0);
		userTextBox.setFocus(true);
	}

	/**
	 * Set the application title to display
	 * 
	 * @param title The application title
	 */
	public void setApplicationTitle(String title) {
		if (title != null && title.trim().length() > 0)
			title = messages.loginFormTitleLabel(title);

		titleLabel.setHTML(title);
	}
	
	public void setApplicationLicensed(boolean applicationLicensed) {
		this.applicationLicensed = applicationLicensed;
	}

	/**
	 * Set the application version to display
	 * 
	 * @param version The application version
	 */
	public void setApplicationVersion(String version) {
		if (version != null && version.trim().length() > 0)
			version = messages.loginFormVersionLabel(version);
		versionLabel.setHTML(version);
	}

	/**
	 * Reset the login form, clearing any previous messages, and data entered in the form
	 */
	public void reset() {
		clearErrorMessages();
		setUserName("");
		setPassword("");
		validateLicense();
	}

	private void validateLicense() {
		if (!applicationLicensed) {
			// Add error message
			List<String> errorMessages = new ArrayList<String>(1);
			errorMessages.add(messages.licenseExpired());
			showErrorMessages(errorMessages);
			
			// Disable login form fields 
			userTextBox.setEnabled(false);
			passwordTextBox.setEnabled(false);
			loginButton.setEnabled(false);
		}		
	}

	/**
	 * Set the user name text box with the given value
	 * 
	 * @param userName The user name value
	 */
	public void setUserName(String userName) {
		this.userTextBox.setText(userName);
	}

	/**
	 * Set the password text box with the given value
	 * 
	 * @param password The password value
	 */
	public void setPassword(String password) {
		this.passwordTextBox.setText(password);
	}

	/**
	 * Display the list of error messages to the user
	 * 
	 * @param messages The messages to display
	 */
	public void showErrorMessages(List<String> errorMessages) {
		/* Show an alert with the messages */
		StringBuilder buf = new StringBuilder();
		boolean addLine = false;

		/* Build a list of the messages separated by new lines */
		for (String message : errorMessages) {
			if (addLine)
				buf.append("\n");
			else
				addLine = false;

			buf.append(message);
		}

		messageArea.setVisible(true);
		messageArea.setErrorMessage(buf.toString());
	}

	/**
	 * Clear the previously shown error messages
	 */
	public void clearErrorMessages() {
		messageArea.setVisible(false);
		messageArea.setErrorMessage(null);
	}

	/**
	 * Register a new event handler
	 * 
	 * @param handler The event handler
	 */
	public void addLoginHandler(LoginHandler handler) {
		if (handler != null && !handlers.contains(handler)) {
			handlers.add(handler);
		}
	}

	/**
	 * Remove the event handler registered
	 * 
	 * @param handler the event handler
	 */
	public void removeLoginHandler(LoginHandler handler) {
		handlers.remove(handler);
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the click event for the login button
	 */
	private void handleLoginButtonClick(ClickEvent event) {
		String userName = userTextBox.getText();
		String password = passwordTextBox.getText();

		LoginEvent loginEvent = new LoginEvent(this, userName, password);
		for (LoginHandler handler : handlers) {
			handler.onLogin(loginEvent);
		}
	}

	/*
	 * Handle the 'enter' key press when focus is in password
	 */
	private void handlePasswordBoxKeyPress(KeyPressEvent event) {
		/* Pressing "Enter" key on password field should trigger login button "Click" */
		if (event.getNativeEvent().getKeyCode() == KeyCodes.KEY_ENTER) {
			loginButton.click();
		}
	}

	
}
