/*
 * Created on Jul 2, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.datatype;

import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.event.dom.client.HasKeyDownHandlers;
import com.google.gwt.event.dom.client.HasKeyPressHandlers;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.event.dom.client.KeyPressEvent;
import com.google.gwt.event.dom.client.KeyPressHandler;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.logical.shared.HasSelectionHandlers;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.HasAnimation;
import com.google.gwt.user.client.ui.HasText;
import com.google.gwt.user.client.ui.MultiWordSuggestOracle;
import com.google.gwt.user.client.ui.SuggestOracle;
import com.google.gwt.user.client.ui.SuggestOracle.Suggestion;
import com.google.gwt.user.client.ui.TextBoxBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.SuggestBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IValueConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.ValueFormatException;

/**
 * A Suggestion Box that displays the list of available file URI values
 * 
 * @author Vijay Silva
 */
public class TypedSuggestBoxWidget<DataType> extends DataTypeWidget<DataType> implements HasText,
        HasAnimation, HasKeyDownHandlers, HasKeyPressHandlers, HasKeyUpHandlers,
        HasSelectionHandlers<Suggestion> {

	private SuggestBoxWidget widget = null;
	private IValueConverter<DataType> valueConverter;

	/**
	 * Constructor
	 * 
	 * @param converter The value converter for data type conversion
	 */
	public TypedSuggestBoxWidget(IValueConverter<DataType> converter) {
		this(converter, new MultiWordSuggestOracle());
	}

	/**
	 * Constructor
	 * 
	 * @param converter The value converter for data type conversion
	 * @param oracle The suggestion oracle that provides the suggestion values
	 */
	public TypedSuggestBoxWidget(IValueConverter<DataType> converter, SuggestOracle oracle) {
		super();

		this.valueConverter = converter;

		/* Build the widget */
		if (oracle == null)
			widget = new SuggestBoxWidget();
		else
			widget = new SuggestBoxWidget(oracle);

		/* Add event handler / delegation for widget */
		EventHandler eventHandler = new EventHandler();
		this.widget.addValueChangeHandler(eventHandler);
		this.widget.addSelectionHandler(eventHandler);
		this.initWidget(widget);
	}

	// ========================================================================
	// ===================== WIDGET ACCESS
	// ========================================================================

	/**
	 * Get the suggest box
	 * 
	 * @return The suggest box
	 */
	protected SuggestBoxWidget getSuggestBox() {
		return this.widget;
	}

	// ========================================================================
	// ===================== HAS TEXT
	// ========================================================================

	/**
	 * @see SuggestBoxWidget#getText()
	 */
	@Override
	public String getText() {
		return widget.getText();
	}

	/**
	 * @see SuggestBoxWidget#setText(String)
	 */
	@Override
	public void setText(String text) {
		widget.setText(text);
	}

	// ========================================================================
	// ===================== HAS ANIMATION
	// ========================================================================

	/**
	 * @see SuggestBoxWidget#isAnimationEnabled()
	 */
	@Deprecated
	@Override
	public boolean isAnimationEnabled() {
		return widget.isAnimationEnabled();
	}

	/**
	 * @see SuggestBoxWidget#setAnimationEnabled(boolean)
	 */
	@Deprecated
	@Override
	public void setAnimationEnabled(boolean enable) {
		widget.setAnimationEnabled(enable);
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addKeyDownHandler(KeyDownHandler handler) {
		return addDomHandler(handler, KeyDownEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyPressHandler(KeyPressHandler handler) {
		return addDomHandler(handler, KeyPressEvent.getType());
	}

	@Override
	public HandlerRegistration addKeyUpHandler(KeyUpHandler handler) {
		return addDomHandler(handler, KeyUpEvent.getType());
	}

	@Override
	public HandlerRegistration addSelectionHandler(SelectionHandler<Suggestion> handler) {
		return addHandler(handler, SelectionEvent.getType());
	}

	// ========================================================================
	// ===================== HAS VALUE
	// ========================================================================

	@Override
	public DataType getValue() {
		DataType value = null;

		try {
			value = valueConverter.getTypedValue(widget.getValue());
		} catch (ValueFormatException vfe) {
		}

		return value;
	}

	@Override
	public void setValue(DataType newValue) {
		setValue(newValue, false);
	}

	@Override
	public void setValue(DataType value, boolean fireEvents) {
		widget.setValue(valueConverter.getStringValue(value), fireEvents);
		this.updateWidgetState();
	}

	@Override
	public HandlerRegistration addValueChangeHandler(ValueChangeHandler<DataType> handler) {
		return this.addHandler(handler, ValueChangeEvent.getType());
	}

	// ========================================================================
	// ===================== FOCUSABLE
	// ========================================================================

	/**
	 * @see SuggestBoxWidget#getTabIndex()
	 */
	@Override
	public int getTabIndex() {
		return widget.getTabIndex();
	}

	/**
	 * @see SuggestBoxWidget#setTabIndex(int)
	 */
	@Override
	public void setTabIndex(int index) {
		widget.setTabIndex(index);
	}

	/**
	 * @see SuggestBoxWidget#setAccessKey(char)
	 */
	@Override
	public void setAccessKey(char key) {
		widget.setAccessKey(key);
	}

	/**
	 * @see SuggestBoxWidget#setFocus(boolean)
	 */
	@Override
	public void setFocus(boolean focused) {
		widget.setFocus(focused);
	}

	// ========================================================================
	// ===================== FOCUS HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addFocusHandler(FocusHandler handler) {
		return this.addDomHandler(handler, FocusEvent.getType());
	}

	@Override
	public HandlerRegistration addBlurHandler(BlurHandler handler) {
		return this.addDomHandler(handler, BlurEvent.getType());
	}

	// ========================================================================
	// ===================== DATA TYPE WIDGET ABSTRACT METHODS
	// ========================================================================

	@Override
	public boolean isValidValue() {
		boolean valid = false;
		try {
			valueConverter.getTypedValue(widget.getText());
			valid = true;
		} catch (ValueFormatException vfe) {
		}

		return valid;
	}

	@Override
	public void setTextValue(String textValue) {
		widget.setText(textValue);
		this.updateWidgetState();
	}

	@Override
	public String getTextValue() {
		return this.getText();
	}

	@Override
	public boolean isReadOnly() {
		return widget.getTextBox().isReadOnly();
	}

	/**
	 * @see TextBoxWidget#setReadOnly(boolean)
	 */
	@Override
	public void setReadOnly(boolean readOnly) {
		widget.getTextBox().setReadOnly(readOnly);
	}

	/**
	 * @see TextBoxWidget#isEnabled()
	 */
	@Override
	public boolean isEnabled() {
		return widget.getTextBox().isEnabled();
	}

	/**
	 * @see TextBoxWidget#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		widget.getTextBox().setEnabled(enabled);
	}

	/**
	 * Add or remove the error style based on the error state
	 */
	protected void updateStyle() {
		if (this.isInErrorState())
			widget.addStyleDependentName(ERROR_STYLE);
		else
			widget.removeStyleDependentName(ERROR_STYLE);
	}

	@Override
	public String getCurrentToolTip() {
		return widget.getTitle();
	}

	@Override
	protected void setCurrentToolTip(String toolTip) {
		widget.setTitle(toolTip);
	}

	// ========================================================================
	// ===================== SUGGEST BOX METHODS
	// ========================================================================

	/**
	 * @see SuggestBoxWidget#getLimit()
	 */
	public int getLimit() {
		return widget.getLimit();
	}

	/**
	 * @see SuggestBoxWidget#getSuggestOracle()
	 */
	public SuggestOracle getSuggestOracle() {
		return widget.getSuggestOracle();
	}

	/**
	 * @see SuggestBoxWidget#getTextBox()
	 */
	public TextBoxBase getTextBox() {
		return widget.getTextBox();
	}

	/**
	 * @see SuggestBoxWidget#hideSuggestionList()
	 */
	@Deprecated
	public void hideSuggestionList() {
		widget.hideSuggestionList();
	}

	/**
	 * @see SuggestBoxWidget#isAutoSelectEnabled()
	 */
	public boolean isAutoSelectEnabled() {
		return widget.isAutoSelectEnabled();
	}

	/**
	 * @see SuggestBoxWidget#isSuggestionListShowing()
	 */
	@Deprecated
	public boolean isSuggestionListShowing() {
		return widget.isSuggestionListShowing();
	}

	/**
	 * @see SuggestBoxWidget#setAutoSelectEnabled(boolean)
	 */
	public void setAutoSelectEnabled(boolean selectsFirstItem) {
		widget.setAutoSelectEnabled(selectsFirstItem);
	}

	/**
	 * @see SuggestBoxWidget#setLimit(int)
	 */
	public void setLimit(int limit) {
		widget.setLimit(limit);
	}

	/**
	 * @see SuggestBoxWidget#setPopupStyleName(String)
	 */
	@Deprecated
	public void setPopupStyleName(String style) {
		widget.setPopupStyleName(style);
	}

	/**
	 * @see SuggestBoxWidget#showSuggestionList()
	 */
	public void showSuggestionList() {
		widget.showSuggestionList();
	}

	// ========================================================================
	// ===================== EVENT DELEGATOR
	// ========================================================================

	/*
	 * Inner class that delegates the value change events to this class
	 */
	private class EventHandler implements ValueChangeHandler<String>, SelectionHandler<Suggestion> {

		@Override
		public void onValueChange(ValueChangeEvent<String> event) {
			updateWidgetState();
			DataType value = null;

			try {
				/* Try getting the converted value */
				value = valueConverter.getTypedValue(event.getValue());
				String formattedValue = valueConverter.getStringValue(value);
				widget.setValue(formattedValue);
			} catch (ValueFormatException vfe) {
			}

			ValueChangeEvent.fire(TypedSuggestBoxWidget.this, value);
		}

		@Override
		public void onSelection(SelectionEvent<Suggestion> event) {
			/* Delegate the event */
			delegateEvent(TypedSuggestBoxWidget.this, event);

			/* Also fire a new value change event */
			updateWidgetState();
			DataType value = null;

			try {
				/* Try getting the converted value */
				value = valueConverter.getTypedValue(event.getSelectedItem().getReplacementString());
				String formattedValue = valueConverter.getStringValue(value);
				widget.setValue(formattedValue);
			} catch (ValueFormatException vfe) {
			}

			ValueChangeEvent.fire(TypedSuggestBoxWidget.this, value);
		}
	}
}
