/*
 * Created on Dec 30, 2011
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view.external;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUser;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserList;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiService;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;

/**
 * The data provider for the external user entity
 * 
 * @author Vijay Silva
 */
final class ExternalUserDataProvider implements DataProvider<String, ExternalUserRecord> {

	private final ExternalUserTableView view;
	private UserUiServiceAsync service;
	private List<Column<?, ExternalUserRecord>> columns = null;

	/* The column names */
	public static final String USERNAME_COLUMN = "userName";
	public static final String FIRSTNAME_COLUMN = "firstName";
	public static final String LASTNAME_COLUMN = "lastName";

	/**
	 * Constructor
	 */
	public ExternalUserDataProvider(ExternalUserTableView view) {
		this.view = view;
	}

	/*
	 * No initialization required
	 */
	@Override
	public void initialize(AsyncCallback<Void> callback) {
		callback.onSuccess(null);
	}

	/*
	 * Get the columns
	 */
	@Override
	public List<Column<?, ExternalUserRecord>> getColumns() {
		if (this.columns == null) {
			/* Build the columns if not previously built */
			this.columns = new ArrayList<Column<?, ExternalUserRecord>>();
			ExternalUserTableColumn column = null;
			String displayName = null;

			/* The user name column */
			displayName = view.getRealmConstants().externalUserNameColumn();
			column = new ExternalUserTableColumn(this, USERNAME_COLUMN, displayName);
			column.setHandlingViewClick(true);
			column.setColumnWidth("34%");
			this.columns.add(column);

			/* The first name column */
			displayName = view.getRealmConstants().externalFirstNameColumn();
			column = new ExternalUserTableColumn(this, FIRSTNAME_COLUMN, displayName);
			column.setColumnWidth("33%");
			this.columns.add(column);

			/* The last name column */
			displayName = view.getRealmConstants().externalLastNameColumn();
			column = new ExternalUserTableColumn(this, LASTNAME_COLUMN, displayName);
			column.setColumnWidth("33%");
			this.columns.add(column);
		}

		return this.columns;
	}

	/**
	 * Get a column given the column name
	 * 
	 * @param columnName The column name
	 * @return The matching column, or null if no column matches
	 */
	public Column<?, ExternalUserRecord> getColumn(String columnName) {
		for (Column<?, ExternalUserRecord> column : getColumns()) {
			if (columnName.equals(column.getName()))
				return column;
		}

		return null;
	}

	@Override
	public void getRecords(final AsyncCallback<List<ExternalUserRecord>> callback) {
		PageFeature pageFeature = view.getTable().getPageFeature();
		int pageSize = pageFeature.getPageSize();
		int startIndex = (pageFeature.getPageNumber() - 1) * pageSize;

		SortFeature<String, ExternalUserRecord> sortFeature = view.getTable().getSortFeature();
		String sortByProperty = null;
		boolean isAscending = true;
		if (sortFeature != null) {
			sortByProperty = sortFeature.getSortColumn().getName();
			isAscending = SortOrder.ASCENDING.equals(sortFeature.getSortOrder());
		}

		/* Get the external users */
		view.getBusyIndicator().center();
		getService().listExternalUsers(view.getSearchCriteria(), startIndex, pageSize,
		        sortByProperty, isAscending, new NeptuneAsyncCallback<UiUserList>() {

			        @Override
			        public void onNeptuneFailure(Throwable caught) {
				        view.getBusyIndicator().hide();
				        view.getTable().getPageFeature().setRecordCount(-1);
				        callback.onFailure(caught);
			        }

			        @Override
			        public void onNeptuneSuccess(UiUserList result) {
				        view.getBusyIndicator().hide();
				        List<ExternalUserRecord> records = new ArrayList<ExternalUserRecord>();
				        for (UiUser user : result.getUsers())
					        records.add(new ExternalUserRecord(user));
				        int recordCount = result.getTotal();
				        boolean hasRecordCount = recordCount >= 0;
				        PageFeature feature = view.getTable().getPageFeature();
				        feature.setRecordCount(recordCount);
				        feature.setShowTotalCount(hasRecordCount);
				        if (!hasRecordCount) {
					        boolean hasMorePages = (records.size() == feature.getPageSize());
					        view.getTable().getPageFeature().setHasMorePages(hasMorePages);
				        }
				        callback.onSuccess(records);
			        }
		        });
	}

	@Override
	public void getRecord(String key, final AsyncCallback<ExternalUserRecord> callback) {
		view.getBusyIndicator().center();
		getService().getExternalUser(key, new NeptuneAsyncCallback<UiUser>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.getBusyIndicator().hide();
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UiUser result) {
				view.getBusyIndicator().hide();
				callback.onSuccess(new ExternalUserRecord(result));
			}
		});
	}

	@Override
	public boolean isRecordCountEnabled() {
		/* Show the record count */
		return true;
	}

	@Override
	public boolean isCheckboxEnabled() {
		/* No batch operations are supported */
		return false;
	}

	/*
	 * Handle notification from the column that a record was clicked
	 */
	void handleRecordClick(ExternalUserRecord record) {
		/* Notify the view that the record was clicked */
		view.handleTableRecordClick(record);
	}

	/*
	 * Get the user service
	 */
	private UserUiServiceAsync getService() {
		if (service == null)
			service = GWT.create(UserUiService.class);

		return service;
	}
}
