/*
 * Created on Jan 19, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.neptune.ui.framework.client.application;

import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_MIDDLE;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions;
import com.tandbergtv.neptune.ui.realm.client.RealmUIEvents;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.ui.realm.client.networkelements.service.INetworkElementsUIService;
import com.tandbergtv.neptune.ui.realm.client.networkelements.service.INetworkElementsUIServiceAsync;
import com.tandbergtv.neptune.ui.realm.client.networkelements.view.RedundancyStatusButton;
import com.tandbergtv.neptune.ui.realm.client.networkelements.view.RedundancyStatusStateHolder;
import com.tandbergtv.neptune.ui.realm.client.settings.service.ISettingsUIService;
import com.tandbergtv.neptune.ui.realm.client.settings.service.ISettingsUIServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.servertime.ServerTimeLabel;

/**
 * The header widget used by the application
 * 
 * @author Vijay Silva
 */
public class HeaderWidget extends Composite {

	/* Styles */
	private static final String STYLE_NAME = "nfw-HeaderWidget";
	private static final String STYLE_DESCRIPTION_CONTAINER = "nfw-HeaderWidget-description";
	private static final String STYLE_LOGO = "nfw-HeaderWidget-logo";
	private static final String STYLE_TITLE_CONTAINER = "nfw-HeaderWidget-titleSection";
	private static final String STYLE_TITLE_CONTAINER_LABEL = "nfw-HeaderWidget-titleSectionLabel";
	private static final String STYLE_TITLE_LABEL = "nfw-HeaderWidget-titleLabel";
	private static final String STYLE_SUBTITLE_LABEL = "nfw-HeaderWidget-subTitleLabel";
	private static final String STYLE_USER_CONTAINER = "nfw-HeaderWidget-userSection";
	private static final String STYLE_DETAIL_CONTAINER = "nfw-HeaderWidget-detailSection";
	private static final String STYLE_DETAIL_SEPARATOR_LABEL = "nfw-HeaderWidget-detailSeparatorLabel";
	private static final String STYLE_DETAIL_SECTION_LABEL = "nfw-HeaderWidget-detailSectionLabel";
	private static final String STYLE_BUTTON_CONTAINER = "nfw-HeaderWidget-buttonPanel";

	private static final String STYLE_IMAGE_ABOUT = "nfw-HeaderWidget-aboutImage";
	private static final String STYLE_BUTTON_ABOUT = "nfw-HeaderWidget-aboutButton";
	private static final String STYLE_IMAGE_GUIDE = "nfw-HeaderWidget-guideImage";
	private static final String STYLE_BUTTON_GUIDE = "nfw-HeaderWidget-guideButton";
	private static final String STYLE_IMAGE_USER = "nfw-HeaderWidget-userImage";
	private static final String STYLE_BUTTON_USER = "nfw-HeaderWidget-userButton";
	private static final String STYLE_IMAGE_LOGOUT = "nfw-HeaderWidget-logoutImage";
	private static final String STYLE_BUTTON_LOGOUT = "nfw-HeaderWidget-logoutButton";
	private static final String STYLE_BUTTON_SEPARATOR = "nfw-HeaderWidget-buttonSeparator";

	/* Constants */
	//private static final String IMAGE_LOGO_PATH = "neptune_framework/images/header-logo.png";
	private static final String IMAGE_LOGO_PATH = "neptune_framework/images/econ01.svg";
	private static final String IMAGE_INFO_PATH = "neptune_framework/images/information_black_16px.svg";
	private static final String IMAGE_GUIDE_PATH = "neptune_framework/images/help_black_16px.svg";
	private static final String IMAGE_USER_PATH = "neptune_framework/images/user_black_16px.svg";
	private static final String IMAGE_LOGOUT_PATH = "neptune_framework/images/logout_black_16px.svg";

	private static final int TIMER_REFRESH_PERIOD = 60*1000;
	
	// Added Page Label Const
	private static final String TITLE_LABEL = "Ericsson Content Management System";
	

	/* Widgets */
	private final ApplicationMessages messages;
	private final RealmConstants realmConstants;
	
	private ImageWidget logoImage;
	private HTMLWidget titleLabel, subTitleLabel;

	
	private HorizontalContainer buttonContainer;
	private ServerTimeLabel serverTimeLabel;
	private ButtonWidget	aboutButton, guideButton, logoutButton;
	private ImageWidget		aboutImage, guideImage, logoutImage, userImage;
	private HTMLWidget		aboutSeparator, guideSeparator, userSeparator, userLabel;

	/* Event Handlers */
	private List<HeaderWidget.EventHandler> handlers = new ArrayList<HeaderWidget.EventHandler>();

	/* The timer to trigger updates of the header widgets. */
	private Timer timer;
	/* Handler for the redundancy status widget event. */
	private HandlerRegistration redundancyStatusEventHandler;
	
	/* The NetworkElements service to check if the module is licensed */
	private static final INetworkElementsUIServiceAsync networkElementsService = GWT.create(INetworkElementsUIService.class);
	private static final ISettingsUIServiceAsync settingsService = GWT.create(ISettingsUIService.class);
	
	/**
	 * The header widget used by the application
	 */
	public HeaderWidget() {
		messages = GWT.create(ApplicationMessages.class);
		realmConstants = GWT.create(RealmConstants.class);
		initializeWidget();
	}

	/*
	 * Initialize the widget contents / layout
	 */
	private void initializeWidget() {
		/* Main container */
		SimpleContainer mainContainer = new SimpleContainer();
		mainContainer.setStyleName(STYLE_NAME);

		/* The main table container */
		HorizontalContainer container = new HorizontalContainer();
		mainContainer.setWidget(container);

		/* start Description Container */
		
		/* Container for logo and application title */
		HorizontalContainer descriptionContainer = new HorizontalContainer();
		descriptionContainer.setStylePrimaryName(STYLE_DESCRIPTION_CONTAINER);
		container.add(descriptionContainer);

		/* Add the logo */
		logoImage = new ImageWidget(IMAGE_LOGO_PATH);
		logoImage.addStyleName(STYLE_LOGO);
		descriptionContainer.add(logoImage);
		descriptionContainer.setCellVerticalAlignment(logoImage, ALIGN_MIDDLE);

		/* add the title */
		titleLabel = new HTMLWidget(TITLE_LABEL);
		titleLabel.addStyleName(STYLE_TITLE_CONTAINER_LABEL);
		titleLabel.addStyleName(STYLE_TITLE_LABEL);
		descriptionContainer.add(titleLabel);
		descriptionContainer.setCellVerticalAlignment(titleLabel, ALIGN_MIDDLE);

		/*
		*  remove subtitle as not needed 
		*/
		/*
		subTitleLabel = new HTMLWidget();
		subTitleLabel.addStyleName(STYLE_TITLE_CONTAINER_LABEL);
		subTitleLabel.addStyleName(STYLE_SUBTITLE_LABEL);
		titleContainer.add(subTitleLabel);
		*/

		/* end Description Container */



		/* start the button container */
		
		buttonContainer = new HorizontalContainer();
		buttonContainer.setStylePrimaryName(STYLE_BUTTON_CONTAINER);
		container.add(buttonContainer);
		
		/* Add the server time label */
		serverTimeLabel = new ServerTimeLabel();
		serverTimeLabel.addStyleName(STYLE_DETAIL_SECTION_LABEL);
		buttonContainer.add(serverTimeLabel);
		buttonContainer.setCellVerticalAlignment(serverTimeLabel, ALIGN_MIDDLE);
	
		/* separator */
		aboutSeparator = new HTMLWidget("&nbsp;");
		aboutSeparator.addStyleName(STYLE_BUTTON_SEPARATOR);
		buttonContainer.add(aboutSeparator);
		
		/* About Image */
		aboutImage = new ImageWidget(IMAGE_INFO_PATH);
		aboutImage.addStyleName(STYLE_IMAGE_ABOUT);
		buttonContainer.add(aboutImage);
		buttonContainer.setCellVerticalAlignment(aboutImage, ALIGN_MIDDLE);
		
		/* About Button */
		aboutButton = new ButtonWidget();
		aboutButton.addStyleName(STYLE_BUTTON_ABOUT);
		aboutButton.setText(messages.headerWidgetAboutButton());
		aboutButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleAboutButtonClick(event);
			}
		});
		buttonContainer.add(aboutButton);
		buttonContainer.setCellVerticalAlignment(aboutButton, ALIGN_MIDDLE);

		/* separator */
		guideSeparator = new HTMLWidget("&nbsp;");
		guideSeparator.addStyleName(STYLE_BUTTON_SEPARATOR);
		buttonContainer.add(guideSeparator);
		
		/* guide image */
		guideImage = new ImageWidget(IMAGE_GUIDE_PATH);
		guideImage.addStyleName(STYLE_IMAGE_GUIDE);
		buttonContainer.add(guideImage);
		buttonContainer.setCellVerticalAlignment(guideImage, ALIGN_MIDDLE);
		
		/* guide button */
		guideButton = new ButtonWidget();
		guideButton.addStyleName(STYLE_BUTTON_GUIDE);
		guideButton.setText(messages.headerWidgetGuideButton());
		guideButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleGuideButtonClick(event);
			}
		});
		buttonContainer.add(guideButton);
		buttonContainer.setCellVerticalAlignment(guideButton, ALIGN_MIDDLE);

		/* separator */
		userSeparator = new HTMLWidget("&nbsp;");
		userSeparator.addStyleName(STYLE_BUTTON_SEPARATOR);
		buttonContainer.add(userSeparator);

		/* user image */
		userImage = new ImageWidget(IMAGE_USER_PATH);
		userImage.addStyleName(STYLE_IMAGE_USER);
		buttonContainer.add(userImage);
		buttonContainer.setCellVerticalAlignment(userImage, ALIGN_MIDDLE);
		
		/* user name */
		userLabel = new HTMLWidget();
		userLabel.addStyleName(STYLE_BUTTON_USER);
		userLabel.setHTML("Admin");
		buttonContainer.add(userLabel);
		buttonContainer.setCellVerticalAlignment(userLabel, ALIGN_MIDDLE);
		
		/* logout button */
		logoutButton = new ButtonWidget();
		logoutButton.addStyleName(STYLE_BUTTON_LOGOUT);
		logoutButton.setText(messages.headerWidgetLogoutButton());
		logoutButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleLogoutButtonClick(event);
			}
		});
		buttonContainer.add(logoutButton);
		buttonContainer.setCellVerticalAlignment(logoutButton, ALIGN_MIDDLE);
		
		/* logout image */
		logoutImage = new ImageWidget(IMAGE_LOGOUT_PATH);
		logoutImage.addStyleName(STYLE_IMAGE_LOGOUT);
		buttonContainer.add(logoutImage);
		buttonContainer.setCellVerticalAlignment(logoutImage, ALIGN_MIDDLE);
		

		/* end the button container */
		
		
		/* Initialize the timer */
		this.timer = new Timer() {
			@Override
			public void run() {
				fireTimeElapsedEvent();
			}
		};
		initWidget(mainContainer);
	}

	/**
	 * 		Notifies the timer listeners
	 */
	protected void fireTimeElapsedEvent() {
		// publishes the time elapsed event to the event bus
		RealmUIEvents.EVENT_BUS.fireEvent(new TimedEvent());
	}

	/*
	 * Start the timer when the widget is attached to the DOM to make server calls
	 */
	@Override
	protected void onLoad() {
		super.onLoad();

		networkElementsService.isHighAvailabilityModuleLicensed(new NeptuneAsyncCallback<Boolean>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				// do nothing, keep last state
			}

			@Override
			public void onNeptuneSuccess(Boolean isLicensed) {
				if (isLicensed) {
					/*Subscribes the redundancy status event handler to the timer event. 
					  so that every 60 seconds the redundancy status handler will be called */
					redundancyStatusEventHandler = RealmUIEvents.EVENT_BUS.addHandler(TimedEvent.TYPE, new RedundancyStatusStateHolder());
					RedundancyStatusStateHolder.refreshCurrentStatus(true);					
				}
			}

		});
		
		if (ClientAuthorizationManager.isAuthorized(RealmComponentPermissions.SETTINGS_MODIFY)) {
			// msg only makes sense to admin users that has settings_modify permission
			settingsService.hasInconsistentGroups(new NeptuneAsyncCallback<Boolean>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					// do nothing, keep last state
				}
	
				@Override
				public void onNeptuneSuccess(Boolean hasSettingsErrors) {
					if (hasSettingsErrors) {
						Window.alert(realmConstants.inconsistentSettingsLoginAlert());					
					}
				}
	
			});
		}
		fireTimeElapsedEvent();
		this.timer.scheduleRepeating(TIMER_REFRESH_PERIOD);
	}

	/*
	 * Stop the timer (ensuring that no server calls are made)
	 */
	@Override
	protected void onUnload() {
		/* Stop the timer. */
		this.timer.cancel();
		super.onUnload();
		if (redundancyStatusEventHandler != null) {
			redundancyStatusEventHandler.removeHandler();
			redundancyStatusEventHandler = null;
		}
	}
	
	// ========================================================================
	// ===================== API
	// ========================================================================

	/**
	 * Set the pattern to use for the server time display
	 * 
	 * @param pattern The date time format pattern to use when displaying the server time
	 */
	public void setServerTimeDisplayPattern(String pattern) {
		serverTimeLabel.setDisplayPattern(pattern);
	}

	/**
	 * Set the Application Title to display in the header widget
	 * 
	 * @param title The application title
	 */
	public void setApplicationTitle(String title) {
		if (title != null && title.trim().length() > 0)
			title = messages.headerWidgetTitleLabel(title);

		//titleLabel.setHTML(title);
	}

	/**
	 * Set the Application sub-title to display in the header widget
	 * 
	 * @param subtitle The application sub-title
	 */
	public void setApplicationSubtitle(String subtitle) {
		if (subtitle != null && subtitle.trim().length() > 0)
			subtitle = messages.headerWidgetSubtitleLabel(subtitle);

		/*
		 *  Disabled due to removing the subTitleLabel
		 */
		//subTitleLabel.setHTML(subtitle);
	}

	/**
	 * Set the currently logged-in user in the header widget
	 * 
	 * @param userName the user name for the currently logged in user
	 */
	public void setCurrentUser(String userName) {
		if (userName != null && userName.trim().length() > 0)
			userName = messages.headerWidgetCurrentUserLabel(userName);

		//userLabel.setHTML(userName);
	}

	/**
	 * Remove the currently logged-in user.
	 */
	public void clearCurrentUser() {
		//userLabel.setHTML("");
	}

	/**
	 * Determine if the logout button is visible
	 * 
	 * @return true if visible, false otherwise
	 */
	public boolean isLogoutButtonVisible() {
		return logoutButton.isVisible();
	}

	/**
	 * Set the logout button visible / invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setLogoutButtonVisible(boolean visible) {
		logoutButton.setVisible(visible);
	}

	/**
	 * Determine if the user guide button is visible
	 * 
	 * @return true if visible, false otherwise
	 */
	public boolean isUserGuideButtonVisible() {
		return guideButton.isVisible();
	}

	/**
	 * Set the user guide button visible / invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setUserGuideButtonVisible(boolean visible) {
		guideButton.setVisible(visible);
	}

	/**
	 * Determine if the user guide button is visible
	 * 
	 * @return true if visible, false otherwise
	 */
	public boolean isAboutButtonVisible() {
		return guideButton.isVisible();
	}

	/**
	 * Set the about button visible / invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setAboutButtonVisible(boolean visible) {
		aboutButton.setVisible(visible);
	}

	/**
	 * Set all the buttons to be visible or invisible
	 * 
	 * @param visible true to make visible, false to make invisible
	 */
	public void setButtonsVisible(boolean visible) {
		setLogoutButtonVisible(visible);
		setUserGuideButtonVisible(visible);
		setAboutButtonVisible(visible);
	}

	/**
	 * Add an event handler
	 * 
	 * @param handler The event handler
	 */
	public void addEventHandler(HeaderWidget.EventHandler handler) {
		if (handler != null && !handlers.contains(handler))
			this.handlers.add(handler);
	}

	/**
	 * Remove a previously registered event handler
	 * 
	 * @param handler The event handler
	 */
	public void removeEventHandler(HeaderWidget.EventHandler handler) {
		this.handlers.remove(handler);
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the 'About' button click
	 */
	private void handleAboutButtonClick(ClickEvent event) {
		for (HeaderWidget.EventHandler handler : handlers)
			handler.onAboutButtonClick(new HeaderWidget.Event(this));
	}

	/*
	 * Handle the 'Guide' button click
	 */
	private void handleGuideButtonClick(ClickEvent event) {
		for (HeaderWidget.EventHandler handler : handlers)
			handler.onUserGuideButtonClick(new HeaderWidget.Event(this));
	}

	/*
	 * Handle the 'Log Out' button click
	 */
	private void handleLogoutButtonClick(ClickEvent event) {
		for (HeaderWidget.EventHandler handler : handlers)
			handler.onLogoutButtonClick(new HeaderWidget.Event(this));
	}

	// ========================================================================
	// ===================== WIDGET EVENT
	// ========================================================================

	/**
	 * The event fired by this widget
	 */
	public static class Event {

		private HeaderWidget source;

		/**
		 * Constructor
		 * 
		 * @param source The event source
		 */
		public Event(HeaderWidget source) {
			this.source = source;
		}

		/**
		 * Get the event source
		 * 
		 * @return The source
		 */
		public HeaderWidget getSource() {
			return source;
		}
	}

	// ========================================================================
	// ===================== WIDGET EVENT HANDLER
	// ========================================================================

	/**
	 * Event handler for the events fired by the Header Widget
	 * 
	 * @author Vijay Silva
	 */
	public interface EventHandler {

		/**
		 * Triggered by the 'About' button is clicked on the header widget
		 * 
		 * @param event The event
		 */
		void onAboutButtonClick(HeaderWidget.Event event);

		/**
		 * Triggered by the 'User Guide' button is clicked on the header widget
		 * 
		 * @param event The event
		 */
		void onUserGuideButtonClick(HeaderWidget.Event event);

		/**
		 * Triggered by the 'Log Out' button is clicked on the header widget
		 * 
		 * @param event The event
		 */
		void onLogoutButtonClick(HeaderWidget.Event event);
	}
}
