package com.tandbergtv.neptune.ui.realm.client.alerts.view;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.TextArea;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.realm.client.alerts.service.UiAlertItem;
import com.tandbergtv.neptune.ui.realm.client.alerts.service.UiRecoveryItem;
import com.tandbergtv.neptune.ui.realm.client.alerts.service.UiRecoveryItemType;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.AnchorWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.InlineHTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.SimpleListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.CheckBox;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

public class AlertItemsTableDataProvider implements DataProvider<String, AlertItemRecord> {

	AlertTableView view;
	
	private List<Column<?, AlertItemRecord>> columns = null;

	private RealmConstants constants = GWT.create(RealmConstants.class);

	/* The column names */
	public static final String TIMESTAMP_COLUMN = "timestamp";
	public static final String NAME_COLUMN = "name";
	public static final String SEVERITY_COLUMN = "severity";
	public static final String MESSAGE_COLUMN = "message";
	// private static final String TYPE_COLUMN = "type";
	public static final String RECOVERY_OPTIONS_COLUMN = "recovery options";

	/* Column widths */
	// private static final String TYPE_COLUMN_WIDTH = "10%";
	private static final String NAME_COLUMN_WIDTH = "20%";
	private static final String SEVERITY_COLUMN_WIDTH = "10%";
	private static final String TIMESTAMP_COLUMN_WIDTH = "9%";
	private static final String MESSAGE_COLUMN_WIDTH = "47%";
	private static final String RECOVERY_OPTIONS_WIDTH = "13%";

	private static final String DETAILED_MSG_ICON = "realm/images/info-icon.png";

	AlertItemsTableDataProvider(AlertTableView view) {
		this.view = view;
	}

	@Override
	public void initialize(AsyncCallback<Void> callback) {
		callback.onSuccess(null);
	}

	@Override
	public List<Column<?, AlertItemRecord>> getColumns() {
		if (this.columns == null) {
			/* Build the columns if not previously built */
			this.columns = new ArrayList<Column<?, AlertItemRecord>>();
			
			/* The item name column */
			String nameDisplayName = constants.itemNameColumn();
			ColumnBase<String, AlertItemRecord> nameColumn = new ColumnBase<String, AlertItemRecord>(
					NAME_COLUMN, nameDisplayName) {
				public View<String> getView(AlertItemRecord record) {
					return new LabelStringView(record.getUiAlertItem().getItemName());
				}

				@Override
				public String getColumnWidth() {
					return NAME_COLUMN_WIDTH;
				}
			};
			nameColumn.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
			this.columns.add(nameColumn);
			
			/* The severity column */
			String severityDisplayName = constants.severityColumn();
			Column<String, AlertItemRecord> severityColumn = prepareSeverityColumn(SEVERITY_COLUMN,
					severityDisplayName);
			this.columns.add(severityColumn);

			/* The timestamp column */
			String timestampDisplayName = constants.timestampColumn();
			ColumnBase<String, AlertItemRecord> timestampColumn = new ColumnBase<String, AlertItemRecord>(
					TIMESTAMP_COLUMN, timestampDisplayName) {
				public View<String> getView(AlertItemRecord record) {
					return new LabelStringView(record.getUiAlertItem().getDisplayDate());
				}

				@Override
				public String getColumnWidth() {
					return TIMESTAMP_COLUMN_WIDTH;
				}
			};
			timestampColumn.setCellStyle(TableConstants.STYLE_DATACELL_DATE);
			this.columns.add(timestampColumn);

			/* The message name column */
			String messageDisplayName = constants.messageColumn();
			Column<String, AlertItemRecord> messageColumn = prepareMessageColumn(MESSAGE_COLUMN,
					messageDisplayName);
			this.columns.add(messageColumn);

			/* Recovery Option column */
			Column<String, AlertItemRecord> recoveryOptions = prepareRecoveryOptionsColumn(
					RECOVERY_OPTIONS_COLUMN, constants.recoveryOptionColumn());
			this.columns.add(recoveryOptions);
		}

		return this.columns;
	}
	
	/**
	 * Get a column given the column name
	 * 
	 * @param columnName The column name
	 * @return The matching column, or null if no column matches
	 */
	public Column<?, AlertItemRecord> getColumn(String columnName) {
		for (Column<?, AlertItemRecord> column : getColumns()) {
			if (columnName.equals(column.getName()))
				return column;
		}

		return null;
	}

	private Column<String, AlertItemRecord> prepareSeverityColumn(String columnName,
			String displayName) {
		ColumnBase<String, AlertItemRecord> column = null;

		column = new ColumnBase<String, AlertItemRecord>(columnName, displayName) {
			@Override
			public View<String> getView(final AlertItemRecord record) {

				return new View<String>() {
					@Override
					public String getStyleName() {
						return TableConstants.STYLE_DATACELL_ICON_TEXT;
					}

					@Override
					public Widget getWidget() {
						String severity = record.getUiAlertItem().getSeverity().toString().toUpperCase();
						ImageWidget severityImage = AlertMessagesPortlet.getImage(severity);
						
						String severityFormatted;
						
						if(severity.equals("ERROR"))
							severityFormatted = "Error";
						else if(severity.equals("WARNING"))
							severityFormatted = "Warning";
						else if(severity.equals("INFORMATIONAL"))
							severityFormatted = "Informational";
						else
							severityFormatted = "";

						InlineHTMLWidget severityMessage = new InlineHTMLWidget(severityFormatted);
						
						HorizontalPanel severityPanel = new HorizontalPanel();
						severityPanel.setWidth("100%");
						severityPanel.setVerticalAlignment(HasVerticalAlignment.ALIGN_MIDDLE);
						severityPanel.add(severityImage);
						severityPanel.add(severityMessage);
						severityPanel.setCellWidth(severityImage, "35px");
						severityPanel.setCellHorizontalAlignment(severityPanel.getWidget(0), HasHorizontalAlignment.ALIGN_CENTER);
						severityPanel.setCellHorizontalAlignment(severityPanel.getWidget(1), HasHorizontalAlignment.ALIGN_LEFT);
						return severityPanel;
					}

					@Override
					public void release() {
					}
				};
			}

			@Override
			public String getColumnWidth() {
				return SEVERITY_COLUMN_WIDTH;
			}
		};
		column.setCellStyle(TableConstants.STYLE_DATACELL_ICON_TEXT);

		return column;
	}

	private Column<String, AlertItemRecord> prepareMessageColumn(String columnName,
			String displayName) {
		ColumnBase<String, AlertItemRecord> column = null;
		column = new ColumnBase<String, AlertItemRecord>(columnName, displayName) {
			@Override
			public View<String> getView(final AlertItemRecord record) {

				return new View<String>() {
					@Override
					public String getStyleName() {
						return TableConstants.STYLE_DATA_TEXT;
					}

					@Override
					public Widget getWidget() {
						String message = record.getUiAlertItem().getMessage();
						if (message == null || message.isEmpty()) {
							return new LabelWidget("");
						}
						final String detMessage = record.getUiAlertItem().getDetailedMessage();
						if (detMessage == null || detMessage.isEmpty()) {
							return new InlineHTMLWidget(message);
						}
						InlineHTMLWidget messageLink = new InlineHTMLWidget(message);

						ImageWidget detIcon = new ImageWidget(DETAILED_MSG_ICON);
						detIcon.setTitle(detMessage);
						detIcon.setStyleName("nwt-actionItemsInfoIcon");
						detIcon.addClickHandler(new ClickHandler() {
							@Override
							public void onClick(ClickEvent event) {
								final ExtMsgPopup popUp = new ExtMsgPopup(detMessage);
								popUp.setPopupPositionAndShow(new PopupPanel.PositionCallback() {
									public void setPosition(int offsetWidth, int offsetHeight) {
										int left = (Window.getClientWidth() - offsetWidth) / 3;
										int top = (Window.getClientHeight() - offsetHeight) / 3;
										popUp.setPopupPosition(left, top);
									}
								});
							}
						});
						FlowPanel descriptionMessagePanel = new FlowPanel();
						descriptionMessagePanel.add(messageLink);
						descriptionMessagePanel.add(detIcon);
						return descriptionMessagePanel;
					}

					@Override
					public void release() {
					}
				};

			}

			@Override
			public String getColumnWidth() {
				return MESSAGE_COLUMN_WIDTH;
			}
		};
		column.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);

		return column;
	}

	private Column<String, AlertItemRecord> prepareRecoveryOptionsColumn(String columnName,
			String displayName) {
		ColumnBase<String, AlertItemRecord> column = new ColumnBase<String, AlertItemRecord>(
				columnName, displayName) {

			@Override
			public View<String> getView(final AlertItemRecord record) {

				return new View<String>() {
					@Override
					public String getStyleName() {
						return "nwt-TableColumnPanel-dataCell-actionItemFixOptions";
					}

					@Override
					public Widget getWidget() {
						final VerticalPanel recoveryOptionsPanel = new VerticalPanel();
						recoveryOptionsPanel.setWidth("100%");
						recoveryOptionsPanel.setVerticalAlignment(HasVerticalAlignment.ALIGN_MIDDLE);

						if (record.getUiAlertItem().getRecoveryOptions() != null && !record.getUiAlertItem().getRecoveryOptions().isEmpty()) {
							record.setRecoveryOptionsPanel(recoveryOptionsPanel);

							final SimpleListBoxWidget recoveryListBox = new SimpleListBoxWidget(false);
							recoveryListBox.setWidth("100%");
							recoveryListBox.addItem("");

							for (int i = 0; i < record.getUiAlertItem().getRecoveryOptions().size(); i++) {
								UiRecoveryItem recoveryItem = record.getUiAlertItem().getRecoveryOptions().get(i);

								if (recoveryItem.getType() == UiRecoveryItemType.Hyperlink) {
									recoveryOptionsPanel.add(new AnchorWidget(recoveryItem.getName(), recoveryItem.getUrl()));
								} 
								else if(recoveryItem.getType() == UiRecoveryItemType.Instruction) {
									recoveryOptionsPanel.add(createRecoveryInstructionsWidget(recoveryItem));
								}
								else if(recoveryItem.getType() == UiRecoveryItemType.Command) {
									//Set the name and the value (the index of the recovery option in the list) for the select box option
									recoveryListBox.addItem(recoveryItem.getName(), Integer.toString(i));
								}
							}

							//Only add the box if we have more items then the default empty item
							if(recoveryListBox.getItemCount() > 1) {
								recoveryListBox.addChangeHandler(new RecoveryOptionsChangeHandler(record, recoveryListBox));
								recoveryOptionsPanel.add(recoveryListBox);
							}

						} else {
							recoveryOptionsPanel.add(createNoRecoveryOptionWidget());
						}

						return recoveryOptionsPanel;
					}

					@Override
					public void release() {
					}
				};
			}

			@Override
			public String getColumnWidth() {
				return RECOVERY_OPTIONS_WIDTH;
			}
		};
		column.setCellStyle(TableConstants.STYLE_DATACELL_GENERAL);

		return column;
	}

	@Override
	public void getRecords(final AsyncCallback<List<AlertItemRecord>> callback) {
		PageFeature pageFeature = view.getTable().getPageFeature();
		int pageSize = pageFeature.getPageSize();
		int startIndex = (pageFeature.getPageNumber() - 1) * pageSize;
		
		SortFeature<?, AlertItemRecord> sortFeature = view.getTable().getSortFeature();
		Column<?, AlertItemRecord> sortColumn = sortFeature.getSortColumn();
		String sortColumnName = (sortColumn != null) ? sortColumn.getName() : null;

		List<UiAlertItem> alertItems = view.getPagedAlertItems(startIndex, pageSize, sortColumnName, sortFeature.getSortOrder());
		
		List<AlertItemRecord> records = new ArrayList<AlertItemRecord>();
		for (UiAlertItem alertItem : alertItems) {
			AlertItemRecord record = new AlertItemRecord(alertItem);
			records.add(record);
		}
		
		callback.onSuccess(records);
	}

	@Override
	public void getRecord(String key, AsyncCallback<AlertItemRecord> callback) {
		callback.onSuccess(null);
	}

	@Override
	public boolean isRecordCountEnabled() {
		return true;
	}

	@Override
	public boolean isCheckboxEnabled() {
		return true;
	}
	
	private Widget createRecoveryInstructionsWidget(UiRecoveryItem recoveryItem) {
		if(recoveryItem.getParameters() != null && !recoveryItem.getParameters().isEmpty()
				&& recoveryItem.getParameters().get("message") != null) {
			
			final String instructions = recoveryItem.getParameters().get("message");
			
			HorizontalPanel panel = new HorizontalPanel();
			panel.setVerticalAlignment(HasVerticalAlignment.ALIGN_MIDDLE);
			panel.add(new Label(recoveryItem.getName()));

			ImageWidget detIcon = new ImageWidget(DETAILED_MSG_ICON);
			detIcon.setStyleName("nwt-actionItemsInfoIcon");
			detIcon.setTitle(instructions);

			detIcon.addClickHandler(new ClickHandler() {											
				@Override
				public void onClick(ClickEvent event) {
					final RecoveryInstructionsPopup popUp = new RecoveryInstructionsPopup(instructions);
					popUp.setPopupPositionAndShow(new PopupPanel.PositionCallback() {
						public void setPosition(int offsetWidth, int offsetHeight) {
							int left = Window.getClientWidth() - offsetWidth - 100;
							int top = (Window.getClientHeight() - offsetHeight) / 2;
							popUp.setPopupPosition(left, top);
						}
					});
				}
			});	
			
			panel.add(detIcon);
			return panel;
		}
		
		return createNoRecoveryOptionWidget();
	}

	
	class RecoveryInstructionsPopup extends PopupPanel {
		public RecoveryInstructionsPopup(String message) {
			// PopupPanel's constructor takes 'auto-hide' as its boolean parameter.
			// If this is set, the panel closes itself automatically when the user
			// clicks outside of it.
			super(true);			
			
			// PopupPanel is a SimplePanel, set it's widget property to TextArea
			TextArea txtArea = new TextArea();
			txtArea.setText(message);
			txtArea.setWidth("350px");
			txtArea.setVisibleLines(15);
			txtArea.setReadOnly(true);
			setWidget(txtArea);
		}
	}	
	
	private Widget createNoRecoveryOptionWidget() {
		LabelWidget noFixAvailableWidget = new LabelWidget(constants.noFixAvailable());
		noFixAvailableWidget.setStyleName(TableConstants.STYLE_DATA_ITALICS);
		return noFixAvailableWidget;
	}	

	class ExtMsgPopup extends PopupPanel {
		public ExtMsgPopup(String detMsg) {
			// PopupPanel's constructor takes 'auto-hide' as its boolean parameter.
			// If this is set, the panel closes itself automatically when the user
			// clicks outside of it.
			super(true);

			// PopupPanel is a SimplePanel, set it's widget property to TextArea
			TextArea txtArea = new TextArea();
			txtArea.setText(detMsg);
			txtArea.setWidth("750px");
			txtArea.setVisibleLines(40);
			txtArea.setReadOnly(true);
			setWidget(txtArea);
		}
	}	
	
	/**
	 * Change handler for the recovery options dropdown.  Checks the selected index and if it's a non empty
	 * selection (i.e. has a name value, not the default option) sets that selected UiRecoverItem on the
	 * AlertItemRecord.  Also, if the selected recovery option is a hyperlink, it "disables" the selection
	 * checkbox.
	 */
	private class RecoveryOptionsChangeHandler implements ChangeHandler {
		
		private final AlertItemRecord record;
		
		private final SimpleListBoxWidget listBox;
		
		/**
		 * @param record The corresponding Alert for the row
		 * @param listBox The listbox that is the handler it attached to
		 */
		private RecoveryOptionsChangeHandler(final AlertItemRecord record, final SimpleListBoxWidget listBox) {
			this.record = record;
			this.listBox = listBox;
		}
		
		@Override
		public void onChange(ChangeEvent event) {
			//Offset for empty select box entry
			final int recoveryOptionIndex = listBox.getSelectedIndex();
			final String selectedValue = listBox.getValue(recoveryOptionIndex);

			UiRecoveryItem recoveryItem = null;
			
			if(selectedValue != null && !selectedValue.isEmpty()) {
				recoveryItem = record.getUiAlertItem().getRecoveryOptions().get(Integer.parseInt(selectedValue));
				record.setSelectedRecoveryOption(recoveryItem); 
			}
			else {				
				record.setSelectedRecoveryOption(null);
			}			
			
			//Disable the data row checkbox when a Hyperlink UiRecoveryItem type is selected.
			for(CheckBox<AlertItemRecord> checkbox : view.getTable().getDataRowsCheckBoxes()) {
				if(checkbox.getRecord().equals(record)) {
					if(record.getSelectedRecoveryOption() != null) {
						checkbox.setValue(Boolean.TRUE);
					}
					else {
						checkbox.setValue(Boolean.FALSE);
					}
					
					break;
				}
			}						
		}
	}
}
