/*
 * Created on Jan 3, 2012
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view.external;

import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_CREATE;
import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_VIEW;

import java.util.HashMap;
import java.util.Map;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmMessages;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUser;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserConfiguration;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiService;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiServiceAsync;
import com.tandbergtv.neptune.ui.realm.client.tab.user.view.UserForm;
import com.tandbergtv.neptune.ui.realm.client.tab.user.view.external.ExternalUserViewEvent.EventType;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.LazyView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCancelHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCommitHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitHandler;

/**
 * View for the external users that switches between the table view and the record details view
 * 
 * @author Vijay Silva
 */
public class ExternalUserView extends LazyView implements
        HasViewCommitHandlers<ExternalUserRecord>, HasViewCancelHandlers {

	private final UiUserConfiguration configuration;
	private VerticalPanel mainPanel;
	private Label errorLabel;
	private SimplePanel widgetPanel;
	private ExternalUserTableView tableView;
	private BusyIndicator busyIndicator = new BusyIndicator(this);
	private UserUiServiceAsync service;
	private RealmMessages messages;
	private UserForm form;
	
	/* Styles / constants */
	private static final String STYLE_NAME = "realm-ExternalUserView";
	private static final String STYLE_ERROR_LABEL = STYLE_NAME + "-errorLabel";
	private static final String STYLE_WIDGET_PANEL = STYLE_NAME + "-widgetPanel";
	private static final String RECORD_KEY_TOKEN = "key";
	
	/* view */
	ResizableContainer container;
	

	/**
	 * Constructor
	 */
	public ExternalUserView(UiUserConfiguration configuration,ResizableContainer container) {
		this.configuration = configuration;
		this.container = container;
		addStyleName(STYLE_NAME);
	}

	/**
	 * Reset the view to the initial state. Does not fire an anchor change event.
	 */
	public void reset() {
		/* Still lazy, do nothing */
		if (getWidget() == null)
			return;

		/* Clear the previously generated error */
		hideErrorMessage();

		/* Display the table and reset the table view */
		setWidgetInPanel(null);
		tableView.reset();
	}

	/**
	 * Show the error message in the view
	 * 
	 * @param message The error message to display
	 */
	public void showErrorMessage(String message) {
		if (message == null)
			message = "";

		errorLabel.setText(message);
		errorLabel.setVisible(true);
	}

	/**
	 * Clear the error message displayed in the view
	 */
	public void hideErrorMessage() {
		errorLabel.setText("");
		errorLabel.setVisible(false);
	}

	/**
	 * Determine if the view is showing the blank widget
	 */
	public boolean isShowingBlankWidget() {
		Widget widget = getWidgetInPanel();
		return (widget == null);
	}

	/**
	 * Show table view, keeping the current state of the features. The records are re-fetched from
	 * the server. Does not fire an anchor change event.
	 * 
	 * @param showFirstPage Flag indicating the table should change to show the first page or
	 *        continue to show the current page of records
	 */
	public void showTable(boolean showFirstPage) {
		/* No more lazy widget */
		ensureWidget();

		/* Clear the previously generated error */
		hideErrorMessage();

		/* Display the table and refresh the state */
		setWidgetInPanel(tableView);
		tableView.refresh(showFirstPage);
	}

	/**
	 * Check if this view is showing the table of external users. If the view is lazy and is not
	 * initialized, this method will return true
	 * 
	 * @return true if the table view is showing, false otherwise
	 */
	public boolean isShowingTable() {
		Widget widget = getWidgetInPanel();
		return (widget != null && widget == tableView);
	}

	/**
	 * Show the external user record given the entity, hiding the table if currently displayed. No
	 * anchor change event fired.
	 */
	public void showExternalUser(ExternalUserRecord record) {
		/* No more lazy widget */
		ensureWidget();

		/* Clear the error message */
		hideErrorMessage();

		/* Build a new user form */
		UiUser externalUser = record.getExternalUser();
		boolean existingUser = externalUser.getKey().isValueAssigned();
		boolean readOnly = (existingUser || !ClientAuthorizationManager.isAuthorized(USER_CREATE));
		form = new UserForm(configuration, externalUser, readOnly,container);
		if (existingUser) {
			form.showErrorMessage(getMessages().externalUserCreated(externalUser.getUserName()),
			        false);
		}

		/* Add event handler */
		FormEventHandler eventHandler = new FormEventHandler();
		form.addViewCancelHandler(eventHandler);
		form.addViewCommitHandler(eventHandler);

		/* Show the form */
		setWidgetInPanel(form);
		
		/* update Buttons */
		updateButtonsForExternalView();
	}

	/**
	 * Check if this view is showing an external user record.
	 * 
	 * @return true if the external user record is showing, false otherwise
	 */
	public boolean isShowingExternalUser() {
		Widget widget = getWidgetInPanel();
		return (widget != null && widget instanceof UserForm);
	}

	/**
	 * Get the external user record selected for display
	 * 
	 * @return The external user selected for display, or null if no record is selected
	 */
	public ExternalUserRecord getSelectedExternalUser() {
		Widget widget = getWidgetInPanel();
		if (widget != null && widget instanceof UserForm) {
			UiUser externalUser = ((UserForm) widget).getUser();
			return new ExternalUserRecord(externalUser);
		}

		return null;
	}

	/*
	 * Get the widget anchor
	 */
	@Override
	public String getAnchor() {
		Widget widget = getWidgetInPanel();
		if (widget == null) {
			return "";
		} else if (widget == tableView) {
			/* We have the table anchor, drop everything up to first '?' */
			String anchor = tableView.getAnchor();
			int index = anchor.indexOf('?');
			anchor = (index < 0) ? "" : anchor.substring(index + 1);
			return anchor;
		} else if (widget instanceof UserForm) {
			Map<String, String> tokens = new HashMap<String, String>();
			form = (UserForm) widget;
			String key = form.getUser().getExternalKey();
			if (key == null)
				key = "";
			tokens.put(RECORD_KEY_TOKEN, key);
			return new AnchorTokenizer().buildAnchor(tokens);
		}

		return "";
	}

	/*
	 * Set the widget anchor
	 */
	@Override
	protected void setWidgetAnchor(String anchor) {
		/* Reset the widget state */
		reset();

		/* Parse the anchor */
		Map<String, String> anchorTokens = new AnchorTokenizer().parseAnchor(anchor);
		if (anchorTokens.containsKey(RECORD_KEY_TOKEN)) {
			String key = anchorTokens.get(RECORD_KEY_TOKEN);
			if (key == null)
				key = "";

			showExternalUser(key, false);
		} else {
			anchor = "?" + anchor;

			/* Display the table and refresh the state */
			setWidgetInPanel(tableView);
			tableView.setAnchor(anchor);
		}
	}

	/*
	 * Create the widget
	 */
	@Override
	protected Widget createWidget() {
		mainPanel = new VerticalPanel();

		/* Add a blank error label */
		errorLabel = new LabelWidget();
		errorLabel.addStyleName(STYLE_ERROR_LABEL);
		errorLabel.setVisible(false);
		mainPanel.add(errorLabel);

		/* Add the widget panel */
		widgetPanel = new SimpleContainer();
		widgetPanel.addStyleName(STYLE_WIDGET_PANEL);
		mainPanel.add(widgetPanel);

		/* The widget panel starts with the table view */
		tableView = new ExternalUserTableView(configuration, busyIndicator,container);
		TableEventHandler eventHandler = new TableEventHandler();
		tableView.addRecordClickHandler(eventHandler);
		tableView.addViewCancelHandler(eventHandler);
		tableView.addViewAnchorChangeHandler(eventHandler);
		widgetPanel.setWidget(null);

		return mainPanel;
	}

	// ========================================================================
	// ===================== INTERNAL STUFF
	// ========================================================================

	/*
	 * If the widget is attached to the DOM, show the busy indicator
	 */
	private void showBusyIndicator() {
		busyIndicator.center();
	}

	/*
	 * Hide the busy indicator
	 */
	private void hideBusyIndicator() {
		busyIndicator.hide();
	}

	/*
	 * Get the user service
	 */
	private UserUiServiceAsync getService() {
		if (service == null)
			service = GWT.create(UserUiService.class);

		return service;
	}

	/*
	 * Get the messages
	 */
	private RealmMessages getMessages() {
		if (messages == null)
			messages = GWT.create(RealmMessages.class);

		return messages;
	}

	/*
	 * Show the external user record given the key, hiding the table if currently displayed.
	 */
	private void showExternalUser(final String key, final boolean fireAnchorChangeEvent) {
		/* Fetch the record and update the view */
		showBusyIndicator();
		getService().getExternalUser(key, new NeptuneAsyncCallback<UiUser>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				hideBusyIndicator();
				String msg = caught.getLocalizedMessage();
				showErrorMessage(getMessages().getExternalUserFailure(key, msg));
			}

			@Override
			public void onNeptuneSuccess(UiUser result) {
				hideBusyIndicator();

				/* Show the external user widget */
				showExternalUser(new ExternalUserRecord(result));

				/* Fire anchor change */
				if (fireAnchorChangeEvent)
					fireViewAnchorChange();
			}
		});
	}

	/*
	 * Get the widget being displayed in the widget panel
	 */
	private Widget getWidgetInPanel() {
		if (widgetPanel == null)
			return null;

		return widgetPanel.getWidget();
	}

	/*
	 * Set the panel widget
	 */
	private void setWidgetInPanel(Widget widget) {
		if (widgetPanel.getWidget() != widget) {
			widgetPanel.setWidget(widget);
			if (widget == null) {
				fireEvent(new ExternalUserViewEvent(EventType.SHOW_BLANK));
			} else if (widget == tableView) {
				if(form != null && form.getSaveButton() != null)
					form.getSaveButton().setVisible(false);
				fireEvent(new ExternalUserViewEvent(EventType.SHOW_TABLE));
			} else if (widget instanceof UserForm) {
				
				
				fireEvent(new ExternalUserViewEvent(EventType.SHOW_EXTERNAL_USER));
			}
		}
		
	}

	// ========================================================================
	// ===================== EVENT MANAGEMENT
	// ========================================================================

	@Override
	public HandlerRegistration addViewCommitHandler(ViewCommitHandler<ExternalUserRecord> handler) {
		return addHandler(handler, ViewCommitEvent.getType());
	}

	@Override
	public HandlerRegistration addViewCancelHandler(ViewCancelHandler handler) {
		return addHandler(handler, ViewCancelEvent.getType());
	}

	/**
	 * Add event handler for the external user view events
	 * 
	 * @param handler The handler
	 * @return The handler registration
	 */
	public HandlerRegistration addViewEventHandler(ExternalUserViewEventHandler handler) {
		return addHandler(handler, ExternalUserViewEvent.getType());
	}

	/*
	 * Fire the view anchor change event
	 */
	private void fireViewAnchorChange() {
		fireViewAnchorChangeEvent(getAnchor());
	}

	// ========================================================================
	// ===================== TABLE EVENT HANDLER
	// ========================================================================

	/*
	 * Internal event handler for the table
	 */
	private final class TableEventHandler implements RecordClickHandler<ExternalUserRecord>,
	        ViewCancelHandler, ViewAnchorChangeHandler {

		@Override
		public void onCancel(ViewCancelEvent event) {
			/* Clear the previously generated error */
			hideErrorMessage();

			/* Clear the previously generated error */
			fireEvent(new ViewCancelEvent());
		}

		@Override
		public void onRecordClick(RecordClickEvent<ExternalUserRecord> event) {
			/* Clear the previously generated error */
			hideErrorMessage();

			/* Verify that the user permissions authorize this action */
			if (!ClientAuthorizationManager.isAuthorized(USER_VIEW, USER_CREATE))
				return;

			/* If the record does not have a user name, do nothing */
			ExternalUserRecord record = event.getRecord();
			String userName = record.getExternalUser().getUserName();
			if (userName == null || userName.trim().isEmpty())
				return;

			/* Get the record from the server and show the user */
			showExternalUser(record.getKey(), true);
		}

		@Override
		public void onAnchorChange(ViewAnchorChangeEvent event) {
			/* Clear the previously generated error */
			hideErrorMessage();

			/* If the table is being displayed, fire the anchor change event */
			if (widgetPanel.getWidget() == tableView) {
				fireViewAnchorChange();
			}
		}
	}

	// ========================================================================
	// ===================== USER FORM EVENT HANDLER
	// ========================================================================

	/*
	 * Internal event handler for the user form
	 */
	private final class FormEventHandler implements ViewCommitHandler<UiUser>, ViewCancelHandler {

		@Override
		public void onCommit(ViewCommitEvent<UiUser> event) {
			/* Reset the view */
			reset();

			/* Fire event indicating that the view has been saved */
			UiUser savedUser = event.getData();
			fireEvent(new ViewCommitEvent<ExternalUserRecord>(new ExternalUserRecord(savedUser)));
			
			//update buttons
			tableView.getCancelButton().setVisible(false);
			form.getCancelButton().setVisible(false);
		}

		@Override
		public void onCancel(ViewCancelEvent event) {
			/* Reset the view */
			showTable(false);

			/* Fire anchor change */
			fireViewAnchorChange();
			
			//update Buttons
			tableView.getCancelButton().setVisible(true);
			form.getCancelButton().setVisible(false);
			container.reset();
		}
	}
	
	private void updateButtonsForExternalView(){
		if(tableView !=null && tableView.getCancelButton() != null)
			tableView.getCancelButton().setVisible(false);
	}
	
	public UserForm getExternalUserForm(){
		return form;
	}
	
	public ExternalUserTableView getExternalTableView(){
		return tableView;
	}
}
