/*
 * Created on Jun 3, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter;

import java.util.Date;

import com.google.gwt.i18n.client.DateTimeFormat;

/**
 * Converts a string from a specified input date format to a specified output date format, where
 * both values are strings.
 * 
 * @author Vijay Silva
 */
public class DateConverter implements IDateConverter {

	/* Properties */
	private String inputFormat;
	private String outputFormat;

	/**
	 * Constructor.
	 * 
	 * @param inputFormat The input format (format of string as fetched from server)
	 * @param outputFormat The output format (format to use in widget display)
	 */
	public DateConverter(String inputFormat, String outputFormat) {
		this.inputFormat = inputFormat;
		this.outputFormat = outputFormat;
	}

	/**
	 * Get the input format to use
	 * 
	 * @return The input format
	 */
	@Override
	public String getInputFormat() {
		return this.inputFormat;
	}

	/**
	 * Get the the output format to use
	 * 
	 * @return The output format
	 */
	@Override
	public String getDisplayFormat() {
		return this.outputFormat;
	}

	/**
	 * Get the String representation of the date string. Converts the string from the input format
	 * to the output format.
	 */
	@Override
	public String getStringValue(String typedValue) {
		String value = null;
		try {
			value = this.convert(typedValue, inputFormat, outputFormat);
		} catch (Exception e) {
			/* Should not fail when converting to string value */
			value = typedValue;
		}

		return value;
	}

	/**
	 * Convert the string from the output format to the input format
	 */
	@Override
	public String getTypedValue(String value) throws ValueFormatException {
		String convertedValue = null;
		try {
			convertedValue = this.convert(value, outputFormat, inputFormat);
		} catch (Exception e) {
			throw new ValueFormatException("Failed to convert value(" + value + ") with format("
			        + outputFormat + ") into date with format(" + inputFormat + ").");
		}

		return convertedValue;
	}

	/**
	 * @see com.tandbergtv.cms.portal.ui.title.client.view.datatype.converter.IDateConverter#getDateForInputValue(java.lang.String)
	 */
	@Override
	public Date getDateForInputValue(String value) {
		return getDateForValue(value, inputFormat);
	}

	/**
	 * @see com.tandbergtv.cms.portal.ui.title.client.view.datatype.converter.IDateConverter#getDateForDisplayValue(java.lang.String)
	 */
	@Override
	public Date getDateForDisplayValue(String value) {
		return getDateForValue(value, outputFormat);
	}

	/**
	 * @see com.tandbergtv.cms.portal.ui.title.client.view.datatype.converter.IDateConverter#getInputValue(java.util.Date)
	 */
	@Override
	public String getInputValue(Date date) {
		return this.convert(date, inputFormat);
	}

	/**
	 * @see com.tandbergtv.cms.portal.ui.title.client.view.datatype.converter.IDateConverter#getDisplayValue(java.util.Date)
	 */
	@Override
	public String getDisplayValue(Date date) {
		return this.convert(date, outputFormat);
	}

	/*
	 * Build a date for the value given the format of the value.
	 */
	private Date getDateForValue(String value, String format) {
		Date date = null;
		try {
			date = this.convert(value, format);
		} catch (Exception e) {
			throw new ValueFormatException("Failed to convert value(" + value + ") with format("
			        + format + ") into date.");
		}

		return date;
	}

	/*
	 * Convert the input value which obeys the inFormat to a string that obeys the outFormat
	 */
	private String convert(String value, String inFormat, String outFormat) {
		Date date = convert(value, inFormat);
		return this.convert(date, outFormat);
	}

	/*
	 * Convert the string value to a date given the format the string value obeys
	 */
	private Date convert(String value, String formatPattern) {
		if (value == null || value.trim().length() == 0)
			return null;

		value = value.trim();

		/* Convert from input format to date to output format */
		DateTimeFormat format = DateTimeFormat.getFormat(formatPattern);
		return format.parseStrict(value);
	}

	/*
	 * Convert the date value to a string given the output format
	 */
	private String convert(Date value, String formatPattern) {
		if (value == null)
			return null;

		DateTimeFormat format = DateTimeFormat.getFormat(formatPattern);
		return format.format(value);
	}

	public String getOutputFormat() {
		return outputFormat;
	}

	public void setOutputFormat(String outputFormat) {
		this.outputFormat = outputFormat;
	}

	public void setInputFormat(String inputFormat) {
		this.inputFormat = inputFormat;
	}
}
