# encoding: utf-8
require 'yaml'
require 'set'
require 'logstash/outputs/base'
require 'logstash/namespace'
require 'snmp'
require 'mail'

require "java"


class Filter

   # Constructor
   def initialize(str)
      @property, $values = str.split('=')
      @values = Set.new
      for $value in $values.split('|')
         @values.add($value)
      end
   end

   # Returns the property
   def property()
      @property
   end

   # Returns whether the string is one of the values matched
   def matches?(str)
      @values.include?(str)
   end

   # Returns the alert as a string
   def to_s
      "#@property=#{@values.to_a.join('|')}"
   end

end

class Alert

   # Constructor
   def initialize(attr)
      @name, @pattern, @filters, @trap, @emails, @th_count, @th_time, @freq_count, @freq_time = '', '', Array.new, 0, Set.new, 1, 0, 1, 0
      @name = attr['name'] if attr['name']
      @pattern = attr['pattern'] if attr['pattern']
      if attr['filter']
         for $filter in attr['filter'].split(',')
            @filters.push(Filter.new($filter))
         end
      end
      @trap = attr['trap'] if attr['trap']
      if attr['email']
         for $email in attr['email'].split()
            @emails.add($email)
         end
      end
      if attr['threshold']
         @th_count = attr['threshold']['count'] if attr['threshold']['count']
         @th_time = parseTime(attr['threshold']['time']) if attr['threshold']['time']
      end
      if attr['frequency']
         @freq_count = attr['frequency']['count'] if attr['frequency']['count']
         @freq_time = parseTime(attr['frequency']['time']) if attr['frequency']['time']
      end
      @regexp = Regexp.new(@pattern)
      @alert_times = Array.new
      @history_times = Array.new
      @squelch_time = 0
   end

   # Returns whether or not the given event matches the filter and pattern of the alert
   def matches?(event)

      return false if not event

      # Check the filters
      for $filter in @filters
         if event.get($filter.property) and not $filter.matches?(event.get($filter.property))
            return false
         end
      end

      # Check if it matches - if not, return
      if (event.get('message') !~ @regexp)
         return false
      end

      # Get the event time
      $event_time = event.get("@timestamp").to_i

      # Check if within the squelch time
      if ($event_time < @squelch_time)
         return false
      end

      # Add to the history
      @history_times.push($event_time)

      # Check if the event has passed the threshold
      if ((@history_times.length < @th_count) or (($event_time - @history_times.first) > @th_time))
         return false
      end

      # Calculate squelch time
      @squelch_time = $event_time + @th_time

      # Clear the history
      @history_times.clear

      # Clear alerts older than the frequency time
      $cutoff_time = $event_time - @freq_time

      while ((not @alert_times.empty?) and (@alert_times.first <= $cutoff_time))
         @alert_times.shift
      end

      # Has the frequency been exceeded?
      if (@alert_times.length >= @freq_count)
         return false
      end

      @alert_times.push($event_time)
      return true
   end

   # Returns the alert's name
   def name()
      @name
   end

   # Returns the alert's trap number
   def trap()
      @trap
   end

   # Returns whether the alert includes a trap
   def sendTrap?()
      (@trap != 0)
   end

   # Returns the alert's e-mail address list
   def emails()
      @emails
   end

   # Returns whether the alert include e-mail addresses
   def sendEmail?()
      (not @emails.empty?)
   end

   # Returns the alert as a string
   def to_s
      "- alert: \n\tname: #@name\n\tpattern: #@pattern\n\tfilter: #{@filters.join(',')}\n\ttrap: #@trap\n\temails: #{@emails.to_a.join(' ')}\n\tthreshold:\n\t\tcount: #@th_count\n\t\ttime: #@th_time\n\tfrequency:\n\t\tcount: #@freq_count\n\t\ttime: #@freq_time"
   end

   # Utility function to parse the given time string and return a number of seconds
   protected
   def parseTime(s)
      if (s !~ /^(\d+):(\d+):(\d+)^/)
         return s.to_i
      end
      return (((($1.to_i * 60) + $2.to_i) * 60) + $3.to_i)
   end

end

class AlertConfiguration

   # Constructor
   def initialize(filename)
      @alerts = Array.new
      @traps = Set.new
      @emails = false
      @matched = Array.new
      @configurationFile = filename
      @configurationTimestamp = Time.new
   end

   # Load the configuration from the given file
   public
   def load?()

      # Check that the configuration file exists
      if not File.readable?(@configurationFile)
         return false
      end

      # Load the configuration file
      $config = YAML.load_file(@configurationFile)

      # Add each alert
      @alerts.clear
      for $alert in $config
          @alerts.push(Alert.new($alert['alert'])) if $alert['alert']
      end

      # Save the last modified time of the configuration file
      @configurationTimestamp = File.mtime(@configurationFile)

      return true
   end

   # Returns whether or not the file has been modified
   public
   def modified?()
      return @configurationTimestamp != File.mtime(@configurationFile)
   end

   # Returns whether or not the given event matches any of the alerts - if it does, the traps and matched are populated
   public
   def matches?(event)

      # Clear containers
      @traps.clear
      @emails = false
      @matched.clear

      # Iterate through the alerts
      $isMatch = false
      for $alert in @alerts
         if $alert.matches?(event)
            $isMatch = true
            @traps.add($alert.trap.to_s) if $alert.sendTrap?
            @emails = true if $alert.sendEmail?
            @matched.push($alert)
         end
      end

      return $isMatch
   end

   # Returns the traps that were matched
   public
   def traps()
      return @traps
   end

   # Returns whether to send any traps
   public
   def sendTraps?()
      return (not @traps.empty?)
   end

   # Returns whether to send any e-mails
   public
   def sendEmails?()
      @emails
   end

   # Return the matched rows for logging
   public
   def matched()
      @matched
   end

   # Returns the alert list as a string
   public
   def to_s
      "---\n#{@alerts.join("\n")}\n...\n"
   end

end

class LogStash::Outputs::CmsAlerts < LogStash::Outputs::Base

config_name "cmsalerts"
milestone 1

#USAGE:
#output {
#  cmsalerts {
#    alertconfiguration => ... # string (optional), default "/usr/share/logstash/alert/alerts.yaml"
#    alertlog => # string (optional), default "/usr/share/logstash/cmsalertlog/cms-alerts.log"
#    snmpserver => ... # string (required)
#    snmpoid => ... # string (optional), default: "1.3.6.1.4.1.11021.20"
#    snmpport => ... # number (optional), default: "162"
#    snmpcommunity => ... # string (optional), default: "public"
#    mailserver => ... # string (optional), default: "mail"
#    mailfrom => ... # string (optional), default: "CMS Alerts <user@hostname.domain>"
#  }
#}

######################## Alerts #########################

# The location of the alert configuration file
config :alertconfiguration, :validate => :string, :default => "/usr/share/logstash/alert/alerts.yaml"

# The location of the file where alert history is logged
config :alertlog, :validate => :string, :default => "/usr/share/logstash/cmsalertlog/cms-alerts.log"

######################## SNMP #########################

# SNMP version (1, 2, or 3)
config :snmpversion, :validate => :number, :default => 1

# The destination server to which SNMP traps are sent
config :snmpserver, :validate => :string, :required => true

# The destination port to which SNMP traps are sent
config :snmpport, :validate => :number, :default => 162

# The OID to use for SNMP traps
config :snmpoid, :validate => :string, :default => "1.3.6.1.4.1.11021.20"

# The community string to use for SNMP trap (v1 and v2)
config :snmpcommunity, :validate => :string, :default => "public"

####################### SNMPv3 #########################

# SNMPv3: User Name
config :snmpuser, :validate => :string

# SNMPv3: Security level
config :snmpsecuritylevel, :validate => :string

# SNMPv3: Authentication algorithm, e.g., MD5
config :authalgo, :validate => :string

# SNMPv3: Authentication password
config :authpassword, :validate => :string

# SNMPv3: Privacy algorithm, e.g., DES, 3DES, AES
config :privalgo, :validate => :string

# SNMPv3: Privacy password
config :privpassword, :validate => :string

# SNMPv3: Trap sender local engine ID
config :localengineid, :validate => :string

######################## e-mail #########################

# The SMTP server to use for sending e-mail alerts
config :mailserver, :validate => :string, :default => "mail"

# The e-mail address from which e-mail alerts are sent
config :mailfrom, :validate => :string, :default => 'CMS Alerts <' + `whoami` + '@' + `hostname -f` + '>'


# Constructor
def initialize(*args)
   super(*args)
   @logger.debug("Initializing CMS Alert Server plug-in")
   @alert_list = AlertConfiguration.new(@alertconfiguration)
end


# Loads SNMP jars. Currently there are 2 jars: com.ericsson.snmp.jar and snmp4j.jar.
# Copy those jars to $LOGSTASH_HOME/vendor/jar/snmp.
def load_snmp_jars
   # This is used in v1.4.2
   # Usually LogStash::Environment::JAR_DIR = $LOGSTASH_HOME/vendor/jar
   # Load jars from $LOGSTASH_HOME/vendor/jar/snmp
   logstash_home = ::File.expand_path(::File.join(::File.dirname(__FILE__), "/../../.."))
   jar_dir = ::File.join(logstash_home, "/vendor/jar")
   snmp_jar_dir = ::File.join(jar_dir, "/snmp")
   puts "Loading SNMP jars from #{snmp_jar_dir}..."

   # Get a list of jars
   jars_path = ::File.join(snmp_jar_dir, "/*.jar")
   jar_files = Dir.glob(jars_path)

   # Throw an error if there are no jars in snmp folder
   if jar_files.empty?
      raise RuntimeError.new("Could not find SNMP jar files under #{snmp_jar_dir}")
   end

   # Iterate over jar list and load each jar
   jar_files.each do |jar|
     loaded = require jar
     puts("Loaded #{jar}") if loaded
   end
end


def register_snmp
   # Validate SNMP version
   unless @snmpversion == 1 || @snmpversion == 2 || @snmpversion == 3
      raise RuntimeError.new("Invalid SNMP version: #@snmpversion")
   end

   # SNMPv2
   if @snmpversion == 1 || @snmpversion == 2
      # Create the SNMP options
      @snmp_options_v1 = {:trap_port => @snmpport, :host => @snmpserver, :community => @snmpcommunity, :Version => :SNMPv1 }
      @snmp_options_v2 = {:trap_port => @snmpport, :host => @snmpserver, :community => @snmpcommunity, :Version => :SNMPv2c }
   end

   # Validate SNMPv3 parameters
   if @snmpversion == 3
      register_snmp_v3
   end

   # Remember the startup time (for uptime)
   @snmp_starttime = Time.now.to_i
end


def register_snmp_v3
   # Security level
   if @snmpsecuritylevel.nil? || @snmpsecuritylevel.empty?
      raise RuntimeError.new("'snmpsecuritylevel' parameter is required for SNMPv3")
   end
   # User name
   if @snmpuser.nil? || @snmpuser.empty?
      raise RuntimeError.new("'snmpuser' parameter is required for SNMPv3")
   end

   # Load SNMP jars. SNMP jars are required only for SNMPv3
   load_snmp_jars

   # Create SNMPv3 trap sender. Note: this is our Java code.
   @trap_sender = com.ericsson.snmp.V3TrapSender.new()

   # Local Engine ID
   unless @localengineid.nil? || @localengineid.empty?
      @trap_sender.setLocalEngineId(@localengineid)
   end

   if @snmpsecuritylevel == "authPriv"
      @trap_sender.addUser(@snmpuser, @authalgo, @authpassword, @privalgo, @privpassword)
   elsif @snmpsecuritylevel == "authNoPriv"
      @trap_sender.addUser(@snmpuser, @authalgo, @authpassword)
   elsif @snmpsecuritylevel == "noAuthNoPriv"
      @trap_sender.addUser(@snmpuser)
   end

   @trap_sender.setTarget(@snmpserver, @snmpport, @snmpuser, @snmpsecuritylevel)
end


# Register the plug-in
public
def register
   @logger.debug("Registering CMS Alerts plug-in")
   
   @logger.debug("The alert file config parameters: alertlog-[#@alertlog] alertconfiguration-[#@alertconfiguration]")
   @logger.debug("The snmp config parameters: snmpcommunity-[#@snmpcommunity] snmpoid-[#@snmpoid] snmpport-[#@snmpport] snmpserver-[#@snmpserver] snmpversion-[#@snmpversion]")
   @logger.debug("The snmpv3 config parameters: localengineid-[#@localengineid] privpassword-[#@privpassword] privalgo-[#@privalgo] authpassword-[#@authpassword] authalgo-[#@authalgo] snmpsecuritylevel-[#@snmpsecuritylevel] snmpuser-[#@snmpuser]")
   @logger.debug("The email config parameters: mailfrom-[#@mailfrom] mailserver-[#@mailserver]")

   # Init SNMP
   register_snmp

   # Load the alerts
   if not @alert_list.load?()
      raise RuntimeError.new("Unable to load CMS Alert Server configuration")
   end

   # Open the alert logging file
   if not File.exists?(@alertlog) or File.writable?(@alertlog)
      begin
         @alert_logfile = File.open(@alertlog, 'a')
         @alert_logfile.sync = true
      rescue
         raise RuntimeError.new("Unable to open CMS Alerts history file #@alertlog")
      end
   else
      raise RuntimeError.new("Unable to open CMS Alerts history file #@alertlog")
   end

   # Create the SMTP options
   Mail.defaults do
      delivery_method :smtp, {
         :address              => "mailhost",
         :port                 => 25,
         :user_name            => nil,
         :password             => nil,
         :authentication       => nil,
         :enable_starttls_auto => false,
         :debug                => false
      }
   end

end

# Cleanup resources
def teardown
   @snmp_sender.close
end


# Get the e-mail subject
private
def emailSubject(alertid, time, gmttime, message)
   "Alert #{alertid} occurred at #{time}"
end

# Get the e-mail body in plain text format
private
def emailPlainBody(alertid, time, gmttime, message)
   "Alert #{alertid} occurred at #{time}

Alert      #{alertid}
Local Time #{time}
GMT Time   #{gmttime}
Message    #{message}"
end

# Get the e-mail body in HTML format
private
def emailHTMLBody(alertid, time, gmttime, message)
   "<p><font size='+1'>Alert <em>#{alertid}</em> occurred at <b>#{time}</b></font></p>
<table border='0' width='100%'>
<tr><th>Alert</th><td>#{alertid}</td></tr>
<tr><th>Local Time</th><td>#{time}</td></tr>
<tr><th>GMT Time</th><td>#{gmttime}</td></tr>
<tr><th>Message</th><td>#{message}</td></tr>
</table>"
end


def send_traps(event)
   # Send SNMP traps?
   if (@alert_list.sendTraps?)
      case @snmpversion
         when 1
            send_trap_v1(event)
         when 2
             send_trap_v2(event)
         when 3
             send_trap_v3(event)
      end
   end
end


def send_trap_v1(event)
   # Initialize the SNMP Manager and send each trap
   SNMP::Manager.open(@snmp_options_v1) do |manager|
      for oid in @alert_list.traps
         # Create the varbind object containing the message
         varbind = SNMP::VarBind.new(@snmpoid + ".0", SNMP::OctetString.new(event.get('message')))
         uptime = 100 * (Time.now.to_i - @snmp_starttime)

         begin
            manager.trap_v1(@snmpoid, @snmpserver, :enterpriseSpecific, oid, uptime, [ varbind ])
            @logger.info("Sent v1 trap #{oid} on #@snmpcommunity to server #@snmpserver:#@snmpport")
         rescue
            @logger.error("Unable to send v1 trap #{oid} on #@snmpcommunity to server #@snmpserver:#@snmpport")
         end
      end
   end
end


def send_trap_v2(event)
   # Initialize the SNMP Manager and send each trap
   SNMP::Manager.open(@snmp_options_v2) do |manager|
      for oid in @alert_list.traps
         # Create the varbind object containing the message
         varbind = SNMP::VarBind.new(@snmpoid + ".0", SNMP::OctetString.new(event.get('message')))

         uptime = 100 * (Time.now.to_i - @snmp_starttime)
         trapoid = @snmpoid + ".0." + oid

         begin
            manager.trap_v2(uptime, trapoid, [ varbind ])
            @logger.info("Sent v2 trap #{trapoid} on #@snmpcommunity to server #@snmpserver:#@snmpport")
         rescue
            @logger.error("Unable to send v2 trap #{trapoid} on #@snmpcommunity to server #@snmpserver:#@snmpport")
         end
      end
   end
end


def send_trap_v3(event)
   for oid in @alert_list.traps
      uptime = 100 * (Time.now.to_i - @snmp_starttime)
      trapoid = @snmpoid + ".0." + oid
      varoid = @snmpoid + ".0"
      message = event.get('message')

      begin
         @trap_sender.send(uptime, trapoid, varoid, message)
         @logger.info("Sent v3 trap #{trapoid} to server #@snmpserver:#@snmpport")
      rescue
         @logger.error("Unable to send v3 trap #{trapoid} to server #@snmpserver:#@snmpport")
      end
   end
end


# Handle the given event
public
def receive(event)

   # Output the event
   if not output?(event)
      return
   end

   # Check if logstash is shutting down
   if event == LogStash::SHUTDOWN
      finished
      return
   end

   # Re-load the alerts if the configuration has changed
   if (@alert_list.modified?())
      if not @alert_list.load?()
         @logger.error("Unable to reload CMS Alert Server configuration")
      else
         @logger.info("Re-loaded CMS Alert Server configuration")
      end
   end

   # Does the event match configured alerts?
   if @alert_list.matches?(event)

      # Send SNMP traps
      send_traps(event)

      # Send e-mails?
      if (@alert_list.sendEmails?)

         # Get the event time
        tstamp = event.get('@timestamp').to_i()
        $localTime = Time.at(tstamp).getlocal.strftime('%a %b %d %H:%M:%S %Z %Y')
        $gmtTime = Time.at(tstamp).utc.strftime('%a %b %d %H:%M:%S %Z %Y')

         # Initialize the mail server and send each message
         for $alert in @alert_list.matched

            if $alert.sendEmail?
               @logger.info("From email is #@mailfrom")
               mail = Mail.new
               mail.delivery_method :smtp, :address => "#@mailserver"
               mail.from = @mailfrom
               mail.to = $alert.emails.to_a
               mail.subject = emailSubject($alert.name, $localTime, $gmtTime, event.get('message'))
               $body = emailPlainBody($alert.name, $localTime, $gmtTime, event.get('message'))
               mail.text_part = Mail::Part.new do
                  content_type "text/plain; charset=UTF-8"
                  body $body
               end
               $body = emailHTMLBody($alert.name, $localTime, $gmtTime, event.get('message'))
               mail.html_part = Mail::Part.new do
                  content_type "text/html; charset=UTF-8"
                  body $body
               end
               @logger.debug("Detailed mail content as below: #{mail}")
               
               begin
                  mail.deliver!
                  @logger.info("Sent e-mail to [#{$alert.emails.to_a.to_a.join(' ')}] using server #@mailserver")
               rescue StandardError, AnotherError => e
                  @logger.error("Unable to send e-mail to [#{$alert.emails.to_a.to_a.join(' ')}] using server #@mailserver with error #{e.inspect}")
               end

            end

         end

      end

      # Log audit messages
      $now = Time.now
      for $alert in @alert_list.matched
         @alert_logfile.write("#{$now.to_s}|#{$alert.name}|#{$alert.trap if ($alert.trap != 0)}|#{$alert.emails.to_a.join(' ')}|#{event.get('message')}\n")
      end

   end

end

end
