#!/bin/bash
# Copyright (c) 2012-2014, EnterpriseDB Corporation.  All rights reserved

# Define our version
EDB_BUG_REPORT_VERSION=20131009

# Get the current OS name
OS=`uname | awk '{print $1}'`

# Get the current BIN Directory path
BIN=`pwd`

# Ensure our path is sane to get the binaries we want

if [ $OS = Linux ] ;then 
	PATH="/sbin:/usr/sbin:${PATH}"
fi 

if [ $OS = Darwin ] ;then 
	PATH="/sbin:/usr/sbin:${PATH}"
fi 

if [ $OS = SunOS ] ;then 
	PATH="/usr/xpg4/bin:/usr/xpg6/bin:/sbin:/usr/sbin:/usr/bin:/usr/ucb:${PATH}"
fi 

# Where are we going to write to?
LOGFILE="edb-bug-report.log"

# Check for compression
BZIP_CMD=$(which bzip2 2>/dev/null | head -n 1)
GZIP_CMD=$(which gzip 2>/dev/null | head -n 1)
if [ -z "${BZIP_CMD}" ] ; then
    COMPRESS="${GZIP_CMD} -c"
    LOGFILE="${LOGFILE}.gz"
else
    COMPRESS="${BZIP_CMD} -c"
    LOGFILE="${LOGFILE}.bz2"
fi

#if [ -z "${BZIP_CMD}" ] ; then
#    COMPRESS="grep ."
#    LOGFILE="$LOGFILE"
#else
#	COMPRESS="grep ."
#    LOGFILE="$LOGFILE"
#fi

# Define our 'pls include this log' message
usage_bug_report_message() {
    echo "Please include the '${LOGFILE}' log file when reporting"
    echo "your issue via the ticketing system (see www.enterprisedb.com)"
    echo "or by sending email to 'support@enterprisedb.com'."
}

# Define our usage message
usage() {
        echo
        echo "$(basename ${0}): EnterpriseDB issue reporting shell script."
        echo
        echo "$(basename ${0}): [-h] [-d DataDirectory] [-p Port]"
        echo "     	-h	(Optional) Print this help output."
        echo "     	-d	(Mandatory) Data directory used by server, from this directory postgres.conf and other configuration files are appended to bugreport."
		echo "     	-p	(Mandatory) Database server port."
        echo ""         
}


# Check our arguments
SAVED_FLAGS=${@}

data_dir=
port=

# Check options passed in.
while getopts "h d:p:" OPTION
do
    case $OPTION in
        h)
            usage
            exit 1
            ;;
        d)
            data_dir="$OPTARG"
            ;;
		p)
            port="$OPTARG"
            ;;
		?)
            usage
            exit
            ;;
    esac
done

echo
echo "____________________________________________"
echo

if [[ ! "$data_dir" ]];
then
	echo "ERROR: -> Please provide the Data Directory"
	usage
	exit 
fi

if [[ ! -d "$data_dir" ]];
then
	echo "ERROR: -> Data Directory is not a valid path, please provide a valid path"
	usage
	exit
fi

if [[ ! "$port" ]];
then
	echo "ERROR: -> Please provide the database server port"
	usage
	exit 
fi

#if [[ ! "$port" > -1 ]];
if [[ ! $port =~ ^[\-0-9]+$ ]];
then
	echo "ERROR: -> Please provide a valid database server port"
	usage
	exit
fi

if [[ $port -lt 0 ]];
then
	echo "ERROR: -> Please provide a valid database server port in postive number"
	usage
	exit
fi


# Function for getting a file's metadata
echo_metadata() {
    printf "*** ls: "
    ls -l --full-time "${1}" 2>/dev/null

    if [ ${?} -ne 0 ] ; then
        # Run a dumb 'ls' and lose the full timestamp :(
        ls -l "${1}"
    fi
}

# Function to append a given file to the log
append() {
    (
    echo "____________________________________________"
    echo

    if [ ! -f "${1}" ] ; then
        echo "*** ${1} does not exist"
    elif [ ! -r "${1}" ] ; then
        echo "*** ${1} is not readable"
    else
        echo "*** ${1}"
        echo_metadata "${1}"
        cat "${1}"
    fi
    echo
    ) | ${COMPRESS} >> ${LOGFILE}
}

# Function to expand a list of files and call append() on each
append_glob() {
    for i in $(ls ${1} 2>/dev/null)
    do
        append "${i}"
    done
}

# Function to append a binary file by base64 encoding it to ascii
append_binary_file() {
    (
    base64=$(which base64 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${base64}" ] ; then
        if [ ! -f "${1}" ] ; then
            echo "*** ${1} does not exist"
        elif [ ! -r "${1}" ] ; then
            echo "*** ${1} is not readable"
        else
            echo "____________________________________________"
            echo
            echo "base64 \"${1}\""
            echo
            ${base64} "${1}" 2>/dev/null
            echo
        fi
    else
        echo "Skipping ${1} output (base64 not found or not executable"
        echo
    fi
    ) | ${COMPRESS} >> ${LOGFILE}
}

#
# main logic
#

# We need to be root, sorry
if [ $(id -u) -ne 0 ] ; then
    echo "ERROR: Please run $(basename ${0}) as root."
    exit 1
fi

# Move old log file out of the way
if [ -f ${LOGFILE} ] ; then
    mv ${LOGFILE} ${LOGFILE}.old
fi

# Ensure we can write to the log
touch ${LOGFILE} 2>/dev/null
if [ ${?} -ne 0 ] ; then
    echo
    echo "ERROR: Working directory is not writeable; please cd to a directory"
    echo "       where you have write permissions so that the ${LOGFILE}"
    echo "       file can be written."
    echo
    exit 1
fi

# Print out our startup message
echo
echo "$(basename ${0}) will now collect information about your"
echo "system and create the file '${LOGFILE}' in the current"
echo "directory.  It may take several seconds to run.  In some"
echo "cases, it may hang trying to capture data generated dynamically"
echo "by the Linux kernel and/or the PostgreSQL cluster.  While"
echo "the bug report log file will be incomplete if this happens, it"
echo "may still contain enough data to diagnose your issue"
echo
usage_bug_report_message
echo
echo -n "Running $(basename ${0})...";
echo

# Print our prologue to the log as well
(
    echo "____________________________________________"
    echo
    echo "Start of EnterpriseDB bug report log file.  Please include this file"
    echo "when reporting an issue via the ticketing system (see www.enterprisedb.com)"
    echo "or by sending email to support@enterprisedb.com"
    echo
    echo "$(basename ${0}) Version: $EDB_BUG_REPORT_VERSION"
    echo
    echo "Date: $(date)"
    echo "uname: $(uname -a)"
    echo "command line flags: ${SAVED_FLAGS}"
    echo
) | ${COMPRESS} >> ${LOGFILE}

# append useful files telling us about the distro
append "/etc/issue"
append "/etc/redhat-release"
append "/etc/redhat_version"
append "/etc/fedora-release"
append "/etc/slackware-release"
append "/etc/debian_release"
append "/etc/debian_version"
append "/etc/mandrake-release"
append "/etc/yellowdog-release"
append "/etc/sun-release"
append "/etc/release"
append "/etc/gentoo-release"
append "/etc/lsb"
append "/etc/lsb-release"

# sockets/cores info
(
    
	
	echo "____________________________________________"
    echo
	if [ -f "/proc/cpuinfo" ] ; then
		echo "Sockets: $(grep "physical id" /proc/cpuinfo | sort -u | wc -l)"
		echo "Cores per socket: $(grep "cpu cores" /proc/cpuinfo | uniq | cut -d: -f2)"
		echo "Virtual CPUs: $(grep '^processor' /proc/cpuinfo | wc -l)"
	else
		echo "/proc/cpuinfo  does not exist"
	fi 
) | ${COMPRESS} >> ${LOGFILE}


#iostat
#The iostat command iteratively reports terminal, drive, and tape I/O activity, as well as CPU utilization.
# iostat info
(
    echo "____________________________________________"
    echo

    iostat=$(which iostat 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${iostat}" ] ; then
        echo "${iostat}"
        echo
        ${iostat} 2>/dev/null
		echo
        echo "____________________________________________"
        echo
		echo "${iostat} -En"
        echo
        ${iostat} -En 2>/dev/null
        echo
        echo "____________________________________________"
        echo
        echo "${iostat} -Xe"
        echo
        ${iostat} -Xe 2>/dev/null
        echo
        echo "____________________________________________"
        echo
        echo "${iostat} -n"
        echo
		${iostat} -n 2>/dev/null
		echo	
		echo "____________________________________________"
        echo
        echo "${iostat} -c"
        echo
        ${iostat} -c 2>/dev/null
    else
        echo "Skipping iostat output (iostat not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

#prtdiag
#The prtdiag command displays configuration and diagnostic information for a system. The diagnostic information identifies any failed component in the system.

(
    echo "____________________________________________"
    echo

    prtdiag=$(which prtdiag 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${prtdiag}" ] ; then
        echo "${prtdiag} -v"
        echo
        ${prtdiag} -v 2>/dev/null
        echo
    else
        echo "Skipping prtdiag output (prtdiag not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

#prtconf
#The prtconf command identifies hardware that is recognized by the Solaris OS. If hardware is not suspected of being bad,
#yet software applications are having trouble with the hardware,
#the prtconf command can indicate if the Solaris software recognizes the hardware and if a driver for the hardware is loaded.
(
    echo "____________________________________________"
    echo

    prtconf=$(which prtconf 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${prtconf}" ] ; then
        echo "${prtconf}"
        echo
        ${prtconf} 2>/dev/null
		echo
        echo "____________________________________________"
        echo
		echo "${prtconf} -D"
        echo
        ${prtconf} -D 2>/dev/null
        echo
        echo "____________________________________________"
        echo
        echo "${prtconf} -p"
        echo
        ${prtconf} -p 2>/dev/null
        echo
        echo "____________________________________________"
        echo
        echo "${prtconf} -n"
        echo
		${prtconf} -n 2>/dev/null
		echo	
		echo "____________________________________________"
        echo
        echo "${prtconf} -V"
        echo
        ${prtconf} -v 2>/dev/null
    else
        echo "Skipping prtconf output (prtconf not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

#sysdef
#The prtdiag command displays configuration and diagnostic information for a system. The diagnostic information identifies any failed component in the system.

(
    echo "____________________________________________"
    echo

    sysdef=$(which sysdef 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${sysdef}" ] ; then
        echo "${sysdef} -v"
        echo
        ${sysdef} -v 2>/dev/null
        echo
    else
        echo "Skipping sysdef output (sysdef not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

#proceesor_type
(
	echo	
	echo "____________________________________________"
	echo

	if [ "$OS" = "Linux" ]
	then
        	# Get processor Type
        	procType=`awk '/model name/ {for(i=1;i<=NF;i=i+1) printf("%s ",$i);}' /proc/cpuinfo`
        	procType=`echo ${procType#model name : }`

	elif [ "$OS" = "SunOS" ]
	then
        	# Get the Processor type
        	procType=`/usr/sbin/psrinfo -v 0 | grep operates | awk '{printf("%s %s %s",$2,$6,$7)}' | awk -F',' '{printf ("%s",$1)}'`

	elif [ "$OS" = "Darwin" ]
	then
        	# Get Processor type
        	t=`uname -m`
        	if [ "$t" = "i386" ]
        	then
                	procType=`sysctl -n machdep.cpu.brand_string | awk '{printf("%s %s %s",$1,$2,$3)}'`
                	procSpd=`sysctl -n machdep.cpu.brand_string | awk -F'@' '{printf("%s",$2)}'`
                	procType="$procType$procSpd"
        	else

                	procType=`sysctl -n hw.machine`
                	procSpd=`sysctl -n hw.cpufrequency_max`
                	procSpd=`expr $procSpd / 1000000`
                	# Convert Proc spd from Hz to MHz
               	 	procType=$procType" "$procSpd" MHz"
        	fi
	fi
    
	echo PROCESSOR_TYPE=$procType 
) | ${COMPRESS} >> ${LOGFILE}

#number_of_cores
(
    if [ "$OS" = "Linux" ]
        then
		# Get number of processor cores
                nProc=`cat /proc/cpuinfo | grep ^processor | wc -l`
	elif [ "$OS" = "SunOS" ]
        then
		# Get the number of processor cores
                nProc=`/usr/bin/kstat -m cpu_info | grep core_id | uniq | wc -l | tr -d ' '`
	elif [ "$OS" = "Darwin" ]
        then
		# Get number of processor cores
                nProc=`sysctl -n hw.ncpu`
	fi
	echo NUMBER_OF_CORES=$nProc 
)  | ${COMPRESS} >> ${LOGFILE}

#processor_arch
(
	if [ -e /etc/debian_version ]
	then
		procArch=`uname -m`
	else
		procArch=`uname -p`
	fi
	echo PROCESSOR_ARCH=$procArch 
)  | ${COMPRESS} >> ${LOGFILE}

#memory_info
(
	if [ "$OS" = "Linux" ]
	then
	# Get ram
		totalMem=`cat /proc/meminfo | grep MemTotal | awk -F':' '{print $2}'| sed 's/^[ \t]*//' |sed 's/k[bB]//'`
	elif [ "$OS" = "SunOS" ]
	then
		# Get ram in MB
		tMem=`/usr/sbin/prtconf | head -3 | grep 'Memory size'`
		totalMem=`echo $tMem | awk '{print $3}'`
	elif [ "$OS" = "Darwin" ]
	then
	# Get ram
		totalMem=`sysctl -n hw.memsize`
	fi

	INSTALLED_BC=`which bc`
	# Make sure that ram is a numeric
	totalMemRound=`expr $totalMem`

	if [ "$OS" = "SunOS" ]
        then
		if [ "x"$INSTALLED_BC = "x" ]; then
			totalMeminKB=`expr $totalMemRound * 1024`
		else
			totalMeminKB=`echo "scale=2;$totalMemRound*1024"|bc`
		fi	
	else
		totalMeminKB=$totalMemRound
	fi
	echo TOTAL_MEMORY_IN_KB=$totalMeminKB KB 
)  | ${COMPRESS} >> ${LOGFILE}

#os_info
(
	if [ "$OS" = "Linux" ]
	then
	os_name=$OS
	# Get distribution name
			if [ -e /etc/debian_version ]
			then
					osFile=/etc/issue
			else
					osFile=`ls -al --time-style=full-iso /etc/*-release |grep -v lsb |grep -v ^l |awk '{print $9}' | head -n 1`
			fi
			osVersion=`cat $osFile | sed q | awk 'BEGIN { FS="@"
													} 
												{ 
													newlineIndex = index ($1,"\\\\n")
													if (newlineIndex == 0)
													  print $1
													else
													  print (substr($1,1,newlineIndex-1))
													}'`
	elif [ "$OS" = "SunOS" ]
	then
	os_name="Solaris"
	osVersion=`uname -r`
	elif [ "$OS" = "Darwin" ]
	then
	os_name="Macintosh"
	osVersion=`sysctl -n kern.osrelease`
	fi

	echo OS=$os_name $osVersion 	
) | ${COMPRESS} >> ${LOGFILE}

#Extra Processor Info for Solaris
if [ "$OS" = "SunOS" ]; then 
	(
		echo "____________________________________________"
		echo "*** Processor Information" 
		echo 
	
	
		kstat -m cpu_info | egrep "chip_id|core_id|module: cpu_info" > /var/tmp/cpu_info.log
		
		nproc=`(grep chip_id /var/tmp/cpu_info.log | awk '{ print $2 }' | sort -u | wc -l | tr -d ' ')`
		ncore=`(grep core_id /var/tmp/cpu_info.log | awk '{ print $2 }' | sort -u | wc -l | tr -d ' ')`
		vproc=`(grep 'module: cpu_info' /var/tmp/cpu_info.log | awk '{ print $4 }' | sort -u | wc -l | tr -d ' ')`
		
		nstrandspercore=$(($vproc/$ncore))
		ncoresperproc=$(($ncore/$nproc))
		
		speedinmhz=`(/usr/bin/kstat -m cpu_info | grep clock_MHz | awk '{ print $2 }' | sort -u)`
		speedinghz=`echo "scale=2; $speedinmhz/1000" | bc`
		
		echo "Total number of physical processors: $nproc"
		echo "Number of virtual processors: $vproc"
		echo "Total number of cores: $ncore"
		echo "Number of cores per physical processor: $ncoresperproc"
		echo "Number of hardware threads (strands or vCPUs) per core: $nstrandspercore"
		echo "Processor speed: $speedinmhz MHz ($speedinghz GHz)"
		
		# now derive the vcpu-to-core mapping based on above information #
		
		echo -e "\n** Socket-Core-vCPU mapping **"
		let linenum=2
		
		for ((i = 1; i <= ${nproc}; ++i ))
		do
				chipid=`sed -n ${linenum}p /var/tmp/cpu_info.log | awk '{ print $2 }'`
				echo -e "\nPhysical Processor $i (chip id: $chipid):"
		
				for ((j = 1; j <= ${ncoresperproc}; ++j ))
				do
						let linenum=($linenum + 1)
						coreid=`sed -n ${linenum}p /var/tmp/cpu_info.log | awk '{ print $2 }'`
						echo -e "\tCore $j (core id: $coreid):"
		
						let linenum=($linenum - 2)
						vcpustart=`sed -n ${linenum}p /var/tmp/cpu_info.log | awk '{ print $4 }'`
		
						let linenum=(3 * $nstrandspercore + $linenum - 3)
						vcpuend=`sed -n ${linenum}p /var/tmp/cpu_info.log | awk '{ print $4 }'`
		
						echo -e "\t\tvCPU ids: $vcpustart - $vcpuend"
						let linenum=($linenum + 4)
				done
		done
		
		rm /var/tmp/cpu_info.log
	) | ${COMPRESS} >> ${LOGFILE}
fi

# psrinfo info
(
    echo "____________________________________________"
    echo

    psrinfo=$(which psrinfo 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${psrinfo}" ] ; then
        echo "${psrinfo}"
        echo
        ${psrinfo} 2>/dev/null
        echo	
		echo "____________________________________________"
        echo
		echo "${psrinfo} -v"
        echo
        ${psrinfo} -v 2>/dev/null
        echo	
    else
        echo "Skipping psrinfo output (psrinfo not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}


# isainfo info
(
    echo "____________________________________________"
    echo

    isainfo=$(which isainfo 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${isainfo}" ] ; then
        echo "${isainfo}"
        echo
        ${isainfo} 2>/dev/null
        echo
    else
        echo "Skipping isainfo output (isainfo not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# isalist info
(
    echo "____________________________________________"
    echo

    isalist=$(which isalist 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${isalist}" ] ; then
        echo "${isalist}"
        echo
        ${isalist} 2>/dev/null
        echo
    else
        echo "Skipping isalist output (isalist not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# zonename info
(
    echo "____________________________________________"
    echo

    zonename=$(which zonename 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${zonename}" ] ; then
        echo "${zonename}"
        echo
        ${zonename} 2>/dev/null
        echo
    else
        echo "Skipping zonename output (zonename not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}


# Filesysteminfo
if [ "$OS" = "Solaris" ]; then 
	(
		vfstab=$(cat /etc/vfstab)
		echo "____________________________________________"
		echo
		echo   ${vfstab} 2>/dev/null
	) | ${COMPRESS} >> ${LOGFILE}
fi 

# ps info
(
    echo "____________________________________________"
    echo

    ps=$(which ps 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${ps}" ] ; then
        echo "${ps} "
        echo
        ${ps} -ef 2>/dev/null
        echo
    else
        echo "Skipping ps output (ps not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# df info
(
    echo "____________________________________________"
    echo

    df=$(which df 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${df}" ] ; then
        echo "${df} -h"
        echo
        ${df} -h 2>/dev/null
		echo
        echo "____________________________________________"
        echo
		echo "${df} -k"
        echo
        ${df} -k 2>/dev/null
        echo
    else
        echo "Skipping df output (df not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# lspci info
(
    echo "____________________________________________"
    echo

    lspci=$(which lspci 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${lspci}" ] ; then
        echo "${lspci} -v -xxx"
        echo
        ${lspci} -v -xxx 2>/dev/null
        echo
        echo "____________________________________________"
        echo
        echo "${lspci} -t"
        echo
        ${lspci} -t 2>/dev/null
        echo
        echo "____________________________________________"
        echo
        echo "${lspci} -nn"
        echo
        ${lspci} -nn 2>/dev/null
    else
        echo "Skipping lspci output (lspci not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}


# lsusb info
(
    echo "____________________________________________"
    echo

    lsusb=$(which lsusb 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${lsusb}" ] ; then
        echo "${lsusb}"
        echo
        ${lsusb} 2>/dev/null
        echo
    else
        echo "Skipping lsusb output (lsusb not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# dmidecode info
(
    echo "____________________________________________"
    echo

    dmidecode=$(which dmidecode 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${dmidecode}" ] ; then
        echo "${dmidecode}"
        echo
        ${dmidecode} 2>/dev/null
        echo
    else
        echo "Skipping dmidecode output (dmidecode not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# sysctl output
(
    echo "____________________________________________"
    echo

    sysctl=$(which sysctl 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${sysctl}" ] ; then
        echo "${sysctl}"
        echo
        ${sysctl} -a
        echo
    else
        echo "Skipping sysctl output (sysctl not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# dmesg info
(
    echo "____________________________________________"
    echo

    dmesg=$(which dmesg 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${dmesg}" ] ; then
        echo "${dmesg}"
        echo
        dmesg 2>/dev/null
    else
        echo "Skipping dmesg output (dmesg not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# useful /proc info
append "/proc/version"
append "/proc/cmdline"

append "/proc/cpuinfo"
append "/proc/sys/kernel/sched_migration_cost_ns"
append "/proc/sys/kernel/sched_autogroup_enabled"

append "/proc/meminfo"
append "/proc/swaps"
append "/proc/sys/vm/swappiness"
append "/proc/sys/vm/zone_reclaim_mode"
append "/proc/sys/vm/dirty_background_ratio"
append "/proc/sys/vm/dirty_ratio"
append "/proc/vmstat"
append "/proc/zoneinfo"
append "/proc/slabinfo"

append "/proc/sys/kernel/shmall"
append "/proc/sys/kernel/shmmax"
append "/proc/sys/kernel/shmmni"

append "/proc/devices"
append "/proc/interrupts"
append "/proc/modules"
append "/proc/pci"
append "/proc/iomem"

append "/proc/mounts"

# useful /sys info
append "/sys/kernel/mm/transparent_hugepage/enabled"
append "/sys/kernel/mm/transparent_hugepage/defrag"
# stupid RedHat
append "/sys/kernel/mm/redhat_transparent_hugepage/enabled"
append "/sys/kernel/mm/redhat_transparent_hugepage/defrag"

append_glob "/sys/block/*/queue/scheduler"

# Networking info
(
    echo "____________________________________________"
    echo

    ifconfig=$(which ifconfig 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${ifconfig}" ] ; then
        echo "${ifconfig}"
        echo
        ${ifconfig} -a
        echo
    else
        echo "Skipping ifconfig output (ifconfig not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}


(
    echo "____________________________________________"
    echo

    netstat=$(which netstat 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${netstat}" ] ; then
        echo "${netstat}"
        echo
        ${netstat}
        echo
		echo
        echo "____________________________________________"
        echo
		echo "${netstat} -n"
        echo
        ${netstat} -n
		echo
        echo "____________________________________________"
        echo
		echo "${netstat} -nr"
        echo
        ${netstat} -nr
		echo
    else
        echo "Skipping netstat output (netstat not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

(
    echo "____________________________________________"
    echo

    ip=$(which ip 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${ip}" ] ; then
        echo "${ip} addr"
        echo
        ${ip} addr
        echo
		echo
        echo "____________________________________________"
        echo
        echo "${ip} link"
        echo
        ${ip} link
        echo
		echo
        echo "____________________________________________"
        echo
        echo "${ip} route"
        echo
        ${ip} route
        echo
    else
        echo "Skipping ip output (ip not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

(
    echo "____________________________________________"
    echo

    ss=$(which ss 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${ss}" ] ; then
        echo "${ss}"
        echo
        ${ss}
        echo
    else
        echo "Skipping ss output (ss not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# lsof output
(
    echo "____________________________________________"
    echo

    lsof=$(which lsof 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${lsof}" ] ; then
        echo "${lsof}"
        echo
        ${lsof}
        echo
    else
        echo "Skipping lsof output (lsof not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

# PostgreSQL info

# find the postmaster 
postmaster=$(ps -ef | grep -i postgres|grep -- '-D')
# extract the executable from the postmaster
#executable=$(echo \"${postmaster}\"|awk '{print $8}')


# IF Data Directory is not passed as command line parameter, extract the data dir from the postmaster
if [[ ! "$data_dir" ]];
then
	data_dir=$(echo "${postmaster}"|awk '{print $NF}'|cut -d= -f2)
fi


# who owns the data dir?
owner=$(ls -ld "${data_dir}"|awk '{print $3}')

(
    echo "____________________________________________"
    echo

    limit=$(which ulimit 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${limit}" ] ; then
        echo "${limit}"
        echo
        su - ${owner} -c "${limit} -a"
        echo
    else
        echo "Skipping ulimit output (ulimit not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

#output of 'set'		
(
    echo "____________________________________________"
    echo

    set=$(which set 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${set}" ] ; then
        echo "set"
        echo
        su - ${owner} -c "set"
        echo
    else
        echo "Skipping set output (set not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}


#output of 'env'	
(
    echo "____________________________________________"
    echo

    env=$(which env 2>/dev/null | head -n 1)

    if [ ${?} -eq 0 -a -x "${env}" ] ; then
        echo "${env}"
        echo
        su - ${owner} -c "${env}"
        echo
    else
        echo "Skipping env output (env not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

append "${data_dir}/pg_hba.conf"
append "${data_dir}/postgresql.conf"
append "${data_dir}/PG_VERSION"
append "${data_dir}/pg_ident.conf"
append "${data_dir}/postmaster.opts"
append "${data_dir}/postmaster.pid"

(
    echo "____________________________________________"
    echo

    psql=${BIN}/psql
	createdb=${BIN}/createdb
	
	echo BIN is $BIN
	echo psql is $psql
	
    if [ -x "${psql}" ] ; then
        echo "select version()"
        echo
        if [ -f ${BIN}/../pgplus_env.sh ] ; then
            su - ${owner} -c "source ${BIN}/../pgplus_env.sh;${psql} -c 'show all'"
        else
			su - ${owner} -c "${createdb} -p ${port} edb 2>/dev/null "
            su - ${owner} -c "${psql} -p ${port} -c 'select version();'"
        fi
        echo
    else
        echo "Skipping 'select version()' output (psql not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

(
    echo "____________________________________________"
    echo

    psql=${BIN}/psql
	createdb=${BIN}/createdb

    if [ -x "${psql}" ] ; then
        echo "show all"
        echo
        if [ -f ${BIN}/../pgplus_env.sh ] ; then
            su - ${owner} -c "source ${BIN}/../pgplus_env.sh;${psql} -c 'show all'"
        else
            su - ${owner} -c "${createdb} -p ${port} edb 2>/dev/null "
            su - ${owner} -c "${psql} -p ${port} -c 'show all'"
        fi
        echo
    else
        echo "Skipping 'show all' output (psql not found or not executable)"
        echo
    fi
) | ${COMPRESS} >> ${LOGFILE}

#chmod 600 ${LOGFILE}

# Done
echo "Complete."
echo


