/*-------------------------------------------------------------------------
 *
 * ocipxy.h
 *	  Support for using OCI library through run-time linking.
 *
 * To avoid linking directly to the OCI library, to let the system start up
 * even if OCI is not set up properly, we use pg_dlopen() to open the OCI,
 * like we do for contrib modules etc. This header file contains definitions
 * needed to hide that run-time linking from users of the OCI API.
 *
 * Usage:
 *  1. Call init_oci() before any OCI calls to load the library if it's not
 *     loaded already.
 *  2. Use OCI calls as usual. The #defines in this header file will map
 *     them to function pointers that point to 
 *
 * If you need to use any new OCI functions that ocipxy.h/c doesn't know
 * about yet, feel free to add them. (but note which version of OCI the
 * function requires first!)
 *
 * Portions Copyright (c) 2004-2013, EnterpriseDB Corporation.
 *
 *-------------------------------------------------------------------------
 */
#ifndef OCIPXY_H
#define OCIPXY_H

#include "postgres.h"

/*
 * Unfortunately there's a conflict with the typedef of "text" between
 * the PostgreSQL c.h and OCI oratypes.h. To avoid compiler error,
 * #define the Postgres text type out of the way, while we include oci.h.
 *
 * This means that "text" will refer to the PostgreSQL text, not OCI one.
 * If you need to use the OCI "text" type, use "oratext" instead.
 */
#define text oci_text_hack

/*
 * Similarly, there's a definition of boolean in oratypes.h that conflicts
 * with the definition of boolean in one of the standard Windows header
 * files. It's not clear why we need this; AFAICS this isn't anything
 * specific to PostgreSQL...
 */
#ifdef WIN32
#ifndef boolean
#define boolean unsigned char
#endif
#endif

#include <oci.h>
#undef text

extern void init_oci(void);

/*
 * Function pointers defined in ocipxy. These are initialized by init_oci()
 * to point to the corresponding functions in the OCI library
 */
extern PGDLLIMPORT sword(*_OCIErrorGet) (dvoid *, ub4, unsigned char *, sb4 *, unsigned char *, ub4, ub4);
extern PGDLLIMPORT sword(*_OCIInitialize) (ub4, dvoid *, dvoid * (*) (dvoid *, size_t), dvoid * (*) (dvoid *, dvoid *, size_t),
									void (*) (dvoid *, dvoid *));
extern PGDLLIMPORT sword(*_OCIEnvInit) (OCIEnv **, ub4, size_t, dvoid **);
extern PGDLLIMPORT sword(*_OCIHandleAlloc) (CONST dvoid *, dvoid **, ub4, size_t, dvoid **);
extern PGDLLIMPORT sword(*_OCIServerAttach) (OCIServer *, OCIError *, CONST unsigned char *, sb4, ub4);
extern PGDLLIMPORT sword(*_OCIAttrSet) (dvoid *, ub4, dvoid *, ub4, ub4, OCIError *);
extern PGDLLIMPORT sword(*_OCISessionBegin) (OCISvcCtx *, OCIError *, OCISession *, ub4, ub4);
extern PGDLLIMPORT sword(*_OCISessionEnd) (OCISvcCtx *, OCIError *, OCISession *, ub4);
extern PGDLLIMPORT sword(*_OCIServerDetach) (OCIServer *, OCIError *, ub4);
extern PGDLLIMPORT sword(*_OCIHandleFree) (dvoid *, ub4);
extern PGDLLIMPORT sword(*_OCIAttrGet) (CONST dvoid *, ub4, dvoid *, ub4 *, ub4, OCIError *);
extern PGDLLIMPORT sword(*_OCIParamGet) (CONST dvoid *, ub4, OCIError *, dvoid **, ub4);
extern PGDLLIMPORT sword(*_OCIStmtPrepare) (OCIStmt *, OCIError *, CONST unsigned char *, ub4, ub4, ub4);
extern PGDLLIMPORT sword(*_OCIDefineByPos) (OCIStmt *, OCIDefine **, OCIError *, ub4, dvoid *, sb4, ub2, dvoid *, ub2 *, ub2 *, ub4);
extern PGDLLIMPORT sword(*_OCIStmtExecute) (OCISvcCtx *, OCIStmt *, OCIError *, ub4, ub4, CONST OCISnapshot *, OCISnapshot *, ub4);

extern PGDLLIMPORT sword(*_OCIStmtFetch2) (OCIStmt *, OCIError *, ub4, ub2, sb4, ub4);
extern PGDLLIMPORT sword(*_OCIBindByPos) (OCIStmt *, OCIBind **, OCIError *, ub4, dvoid *, sb4, ub2, dvoid *, ub2 *, ub2 *, ub4, ub4 *, ub4);
extern PGDLLIMPORT sword(*_OCIStmtGetPieceInfo) (OCIStmt *, OCIError *, dvoid **, ub4 *, ub1 *, ub4 *, ub4 *, ub1 *);
extern PGDLLIMPORT sword(*_OCIStmtSetPieceInfo) (dvoid *, ub4, OCIError *, dvoid *, ub4 *, ub1, dvoid *, ub2 *); 
extern PGDLLIMPORT sword(*_OCITransCommit) (OCISvcCtx *, OCIError *, ub4);
extern PGDLLIMPORT sword(*_OCITransRollback) (OCISvcCtx *, OCIError *, ub4);

extern PGDLLIMPORT sword(*_OCIDescriptorAlloc) (CONST dvoid *, dvoid **, CONST ub4, CONST size_t, dvoid **);
extern PGDLLIMPORT sword(*_OCIDescriptorFree) (dvoid *, CONST ub4);
extern PGDLLIMPORT sword(*_OCILobOpen) (OCISvcCtx *, OCIError *, OCILobLocator *, ub1);
extern PGDLLIMPORT sword(*_OCILobRead) (OCISvcCtx *, OCIError *, OCILobLocator *, ub4 *, ub4, dvoid *, ub4, dvoid *, OCICallbackLobRead, ub2, ub1);
extern PGDLLIMPORT sword(*_OCILobGetLength) (OCISvcCtx *, OCIError *, OCILobLocator *, ub4 *);
extern PGDLLIMPORT sword(*_OCILobGetChunkSize) (OCISvcCtx *, OCIError *, OCILobLocator *, ub4 *);
extern PGDLLIMPORT sword(*_OCILobClose) (OCISvcCtx *, OCIError *, OCILobLocator *);
extern PGDLLIMPORT sword(*_OCIDateTimeConstruct) (dvoid *, OCIError *, OCIDateTime *, sb2, ub1, ub1, ub1, ub1, ub1, ub4, char *, size_t);

/* Divert calls to OCI functions to ocipxy function pointers */
#define OCIInitialize (*_OCIInitialize)
#define OCIAttrGet (*_OCIAttrGet)
#define OCIAttrSet (*_OCIAttrSet)
#define OCIStmtFetch2 (*_OCIStmtFetch2)
#define OCISessionBegin (*_OCISessionBegin)
#define OCISessionEnd (*_OCISessionEnd)
#define OCIHandleAlloc (*_OCIHandleAlloc)
#define OCIHandleFree (*_OCIHandleFree)
#define OCIEnvInit (*_OCIEnvInit)
#define OCIStmtPrepare (*_OCIStmtPrepare)
#define OCIStmtExecute (*_OCIStmtExecute)
#define OCIErrorGet (*_OCIErrorGet)
#define OCIParamGet (*_OCIParamGet)
#define OCIServerAttach (*_OCIServerAttach)
#define OCIServerDetach (*_OCIServerDetach)
#define OCIDefineByPos (*_OCIDefineByPos)
#define OCIDescriptorAlloc (*_OCIDescriptorAlloc)
#define OCIDescriptorFree (*_OCIDescriptorFree)
#define OCITransCommit (*_OCITransCommit)
#define OCITransRollback (*_OCITransRollback)
#define OCIBindByPos (*_OCIBindByPos)
#define OCIStmtGetPieceInfo (*_OCIStmtGetPieceInfo)
#define OCIStmtSetPieceInfo (*_OCIStmtSetPieceInfo)
#define OCILobOpen (*_OCILobOpen)
#define OCILobRead (*_OCILobRead)
#define OCILobGetLength (*_OCILobGetLength)
#define OCILobGetChunkSize (*_OCILobGetChunkSize)
#define OCILobClose (*_OCILobClose)
#define OCIDateTimeConstruct (*_OCIDateTimeConstruct)

/*
 * Compatibilitiy macros to emulate old ocipxy_* functions. Do not use 
 * in new code. We should eventually replace all references to these
 * with calls to the proper OCI* functions (that are diverted to _OCI*
 * function pointers).
 */
#define ocipxy_HandleFree	OCIHandleFree
#define ocipxy_HandleAlloc(ptr, dest, type) OCIHandleAlloc(ptr, dest, type, 0, 0)
#define ocipxy_StmtPrepare(stmthp, errhp, sql, len) OCIStmtPrepare(stmthp, errhp, sql, len, OCI_NTV_SYNTAX, OCI_DEFAULT)
#define ocipxy_StmtExecute(svchp, stmthp, errhp, rows, type) OCIStmtExecute(svchp, stmthp, errhp, rows, 0, NULL, NULL, type)
#define ocipxy_StmtFetch2(stmthp, errhp, rows, ncur) OCIStmtFetch2(stmthp, errhp, rows, OCI_FETCH_ABSOLUTE, ncur, OCI_DEFAULT)
#define ocipxy_BindByPos(stmthp, p_Bind, errhp, i, val) \
	OCIBindByPos(stmthp, p_Bind, errhp, i, val, (((val) == NULL) ? 0 : (strlen(val) + 1)), SQLT_STR, 0,0,0,0,0, OCI_DEFAULT)
#define ocipxy_DefineByPos(stmthp, define, errhp, i, val, col_len, nulls) \
	OCIDefineByPos(stmthp, define, errhp, i, val, col_len, SQLT_STR, nulls, 0, 0, OCI_DEFAULT)
#define ocipxy_TransCommit(svchp, errhp) OCITransCommit(svchp, errhp, 0)
#define ocipxy_TransRollback(svchp, errhp) OCITransRollback(svchp, errhp, 0)
#define ocipxy_SessionBegin(svchp, errhp, authp) OCISessionBegin(svchp, errhp, authp, OCI_CRED_RDBMS, OCI_DEFAULT)
#define ocipxy_SessionEnd(svchp, errhp, authp) OCISessionEnd(svchp, errhp, authp, 0)
#define ocipxy_ServerAttach(srvhp, errhp, server, len) OCIServerAttach(srvhp, errhp, server, len, OCI_DEFAULT)
#define ocipxy_ServerDetach(svchp, errhp) OCIServerDetach(svchp, errhp, OCI_DEFAULT)
#define ocipxy_EnvInit(envp) OCIEnvInit(envp, OCI_DEFAULT, 0, 0)
#define ocipxy_ErrorGet(errhp, errCode, buff, bufflen) OCIErrorGet(errhp, 1, NULL, errCode, buff, bufflen, OCI_HTYPE_ERROR)
#define ocipxy_ParamGet OCIParamGet
#define ocipxy_AttrGet OCIAttrGet
#define ocipxy_AttrSet OCIAttrSet

#define HTYPE_STMT OCI_HTYPE_STMT
#define HTYPE_SVCCTX OCI_HTYPE_SVCCTX
#define HTYPE_ERROR OCI_HTYPE_ERROR
#define HTYPE_SESSION OCI_HTYPE_SESSION
#define HTYPE_SERVER OCI_HTYPE_SERVER
#define DESCRIBE_ONLY OCI_DESCRIBE_ONLY
#define ATTR_PARAM_COUNT OCI_ATTR_PARAM_COUNT
#define ATTR_NAME OCI_ATTR_NAME
#define ATTR_DATA_TYPE OCI_ATTR_DATA_TYPE
#define ATTR_CHAR_SIZE OCI_ATTR_CHAR_SIZE
#define ATTR_ROW_COUNT OCI_ATTR_ROW_COUNT
#define ATTR_PREFETCH_MEMORY OCI_ATTR_PREFETCH_MEMORY
#define ATTR_SERVER OCI_ATTR_SERVER
#define ATTR_USERNAME OCI_ATTR_USERNAME
#define ATTR_PASSWORD OCI_ATTR_PASSWORD
#define ATTR_SESSION OCI_ATTR_SESSION
#define STMT_SCROLLABLE_READONLY OCI_STMT_SCROLLABLE_READONLY
#define DTYPE_PARAM OCI_DTYPE_PARAM
#define EXEC_DEFAULT OCI_DEFAULT

#define ocipxy_SUCCESS() OCI_SUCCESS
#define ocipxy_SUCCESS_WITH_INFO() OCI_SUCCESS_WITH_INFO
#define ocipxy_ERROR() OCI_ERROR
#define ocipxy_NODATA() OCI_NO_DATA
#define ocipxy_RESERVED_FOR_INT_USE() OCI_RESERVED_FOR_INT_USE
#define ocipxy_SERVER_NOT_CONNECTED() OCI_SERVER_NOT_CONNECTED
#define ocipxy_INVALID_HANDLE() OCI_INVALID_HANDLE
#define ocipxy_NEED_DATA() OCI_NEED_DATA
#define ocipxy_STILL_EXECUTING() OCI_STILL_EXECUTING

extern bool ocipxy_Initialize(void);

/* end of deprecated compatibility stuff */

#endif   /* OCIPXY_H */
