/*-------------------------------------------------------------------------
 *
 * pg_synonym.h
 *	  definition of the system "synonym" relation (pg_synonym)
 *	  along with the relation's initial contents (if any).
 *
 * Copyright (c) 2004-2013, EnterpriseDB Corporation. All Rights Reserved.
 *
 * src/include/catalog/pg_synonym.h
 *
 * NOTES
 *	  the genbki.pl script reads this file and generates .bki
 *	  information from the DATA() statements.
 *
 *-------------------------------------------------------------------------
 */
#ifndef PG_SYNONYM_H
#define PG_SYNONYM_H

/* ----------------
 *	postgres.h contains the system type definitions and the
 *	CATALOG(), BKI_BOOTSTRAP and DATA() sugar words so this file
 *	can be read by both genbki.pl and the C compiler.
 * ----------------
 */

/* ----------------------------------------------------------------------------
 * Definition of pg_synonym
 *
 *	cpp turns this into typedef struct FormData_pg_synonym
 *
 *	STRUCTURE CONTENTS:
 *		synname				name of the synonym
 *		synnamespace		namespace where the synonym resides
 *		synowner			owner (creator) of the synonym
 *		synobjschema		name of the schema where base object resides
 *		synobjname			name of the base object referenced by the synonym
 * ----------------------------------------------------------------------------
 */
#define SynonymRelationId	5000

CATALOG(pg_synonym,5000)
{
	NameData	synname;				/* Name of the synonym */
	Oid			synnamespace;			/* synonym namespace */
	Oid			synowner;				/* OID of the synonym owner */

	/*
	 * As synonyms can reference objects which don't exist, we can only store
	 * the names of the base object and its schema as specified by the user
	 * during creation.  Later, at the time of lookup, the textual schema
	 * and object names are used for performing a catalog search to retrieve
	 * the actual OIDs representing the base object (if it exists).
	 */
	NameData	synobjschema;			/* Name of schema where base object resides */
	NameData	synobjname;				/* Name of the base object */
	NameData	synlink;				/* Name of database link or empty string */
} FormData_pg_synonym;

/* ----------------
 *	Form_pg_synonym corresponds to a pointer to a tuple with
 *	the format of the pg_synonym relation.
 * ----------------
 */
typedef FormData_pg_synonym *Form_pg_synonym;

/* ----------------
 *	compiler constants for pg_synonym
 * ----------------
 */

#define Natts_pg_synonym 			 6

#define Anum_pg_synonym_synname		 1
#define Anum_pg_synonym_synnamespace 2
#define Anum_pg_synonym_synowner	 3
#define Anum_pg_synonym_synobjschema 4
#define Anum_pg_synonym_synobjname	 5
#define Anum_pg_synonym_synlink		 6

/*
 * prototypes for functions in pg_synonym.c
 */
extern void	SynonymCreate(Oid namespaceId, const char *synName, Oid ownerId,
						  const char *linkName, const char *schemaName,
						  const char *objectName, bool replace);

extern Oid LookupSynonym(List *synname, bool ispublic, bool missing_ok);

#endif   /* PG_SYNONYM_H */
