/*-------------------------------------------------------------------------
 *
 * fdwapi.h
 *	  API for foreign-data wrappers
 *
 * Copyright (c) 2010-2013, PostgreSQL Global Development Group
 *
 * src/include/foreign/fdwapi.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef FDWAPI_H
#define FDWAPI_H

#include "nodes/execnodes.h"
#include "nodes/relation.h"

/* To avoid including explain.h here, reference ExplainState thus: */
struct ExplainState;


/*
 * Callback function signatures --- see fdwhandler.sgml for more info.
 */

typedef void (*GetForeignRelSize_function) (PlannerInfo *root,
														RelOptInfo *baserel,
														Oid foreigntableid);

typedef void (*GetForeignPaths_function) (PlannerInfo *root,
													  RelOptInfo *baserel,
													  Oid foreigntableid);

typedef ForeignScan *(*GetForeignPlan_function) (PlannerInfo *root,
														 RelOptInfo *baserel,
														  Oid foreigntableid,
													  ForeignPath *best_path,
															 List *tlist,
														 List *scan_clauses);

typedef void (*BeginForeignScan_function) (ForeignScanState *node,
													   int eflags);

typedef TupleTableSlot *(*IterateForeignScan_function) (ForeignScanState *node);

typedef void (*ReScanForeignScan_function) (ForeignScanState *node);

typedef void (*EndForeignScan_function) (ForeignScanState *node);

typedef void (*AddForeignUpdateTargets_function) (Query *parsetree,
												   RangeTblEntry *target_rte,
												   Relation target_relation);

typedef List *(*PlanForeignModify_function) (PlannerInfo *root,
														 ModifyTable *plan,
														 Index resultRelation,
														 int subplan_index);

typedef void (*BeginForeignModify_function) (ModifyTableState *mtstate,
														 ResultRelInfo *rinfo,
														 List *fdw_private,
														 int subplan_index,
														 int eflags);

typedef TupleTableSlot *(*ExecForeignInsert_function) (EState *estate,
														ResultRelInfo *rinfo,
														TupleTableSlot *slot,
												   TupleTableSlot *planSlot);

typedef TupleTableSlot *(*ExecForeignUpdate_function) (EState *estate,
														ResultRelInfo *rinfo,
														TupleTableSlot *slot,
												   TupleTableSlot *planSlot);

typedef TupleTableSlot *(*ExecForeignDelete_function) (EState *estate,
														ResultRelInfo *rinfo,
														TupleTableSlot *slot,
												   TupleTableSlot *planSlot);

typedef void (*EndForeignModify_function) (EState *estate,
													   ResultRelInfo *rinfo);

typedef int (*IsForeignRelUpdatable_function) (Relation rel);

typedef void (*ExplainForeignScan_function) (ForeignScanState *node,
													struct ExplainState *es);

typedef void (*ExplainForeignModify_function) (ModifyTableState *mtstate,
														ResultRelInfo *rinfo,
														   List *fdw_private,
														   int subplan_index,
													struct ExplainState *es);

typedef int (*AcquireSampleRowsFunc) (Relation relation, int elevel,
											   HeapTuple *rows, int targrows,
												  double *totalrows,
												  double *totaldeadrows);

typedef bool (*AnalyzeForeignTable_function) (Relation relation,
												 AcquireSampleRowsFunc *func,
													BlockNumber *totalpages);

/* EDB extensions */

typedef List *(*GetTables_function) (Oid serverid,
									 const char *remoteSearchPath);
typedef TupleDesc (*DescRel_function) (Oid serverid,
										const char *schema, const char *table);

/*
 * Inserting rows into a remote table.
 *
 * First, call BeginInsert.
 *
 * If there is a RETURNING clause, a tuple descriptor describing the
 * remote relation is passed in 'returning_desc'. Note that it is
 * supposed to be a descriptor describing the remote relation as is,
 * including all columns and no expressions. Any projections are done
 * locally. When 'returning_desc' is given, Insert() opens a
 * result set, call IterateReturning() to get the returned tuples.
 *
 * Then, call Insert for every row, passing the opaque state-struct returned
 * by BeginInsert. The row to be inserted is in 'slot', and the number of
 * rows actually inserted is returned (usually 1, but rules in the
 * remote relation can mess with that).
 *
 * When you're done inserting, call EndInsert.
 */
typedef void *(*BeginInsert_function) (Oid foreigntableid, TupleDesc returning_desc, int n_attr_types, RemoteAttributeType *attr_type_list);
typedef int (*Insert_function) (void *opaque, TupleTableSlot *slot);
typedef void (*EndInsert_function) (void *opaque);
typedef HeapTuple (*IterateReturning_function) (void *opaque);

typedef void *(*BeginUpdate_function) (Oid foreigntableid, RemoteAttributeType *attr_type_list);
typedef int (*Update_function) (void *opaque, char *rowid, TupleTableSlot *slot);
typedef void (*EndUpdate_function) (void *opaque);

typedef void *(*BeginDelete_function) (Oid foreigntableid);
typedef int (*Delete_function) (void *opaque, char *rowid);
typedef void (*EndDelete_function) (void *opaque);

typedef void (*CleanupCache_function) (Oid serverid);

/*
 * Common FDW callback routines to be used by dblink connectors.
 */
extern void dblinkExplainForeignScan(ForeignScanState *node, struct ExplainState *es);
extern void dblinkBeginForeignScan (ForeignScanState *node, int eflags);
extern TupleTableSlot *dblinkIterateForeignScan(ForeignScanState *node);
extern void dblinkReScanForeignScan(ForeignScanState *node);
extern void dblinkEndForeignScan(ForeignScanState *node);

/*
 * FdwRoutine is the struct returned by a foreign-data wrapper's handler
 * function.  It provides pointers to the callback functions needed by the
 * planner and executor.
 *
 * More function pointers are likely to be added in the future.  Therefore
 * it's recommended that the handler initialize the struct with
 * makeNode(FdwRoutine) so that all fields are set to NULL.  This will
 * ensure that no fields are accidentally left undefined.
 */
typedef struct FdwRoutine
{
	NodeTag		type;

	/* Functions for scanning foreign tables */
	GetForeignRelSize_function GetForeignRelSize;
	GetForeignPaths_function GetForeignPaths;
	GetForeignPlan_function GetForeignPlan;
	BeginForeignScan_function BeginForeignScan;
	IterateForeignScan_function IterateForeignScan;
	ReScanForeignScan_function ReScanForeignScan;
	EndForeignScan_function EndForeignScan;

	/*
	 * Remaining functions are optional.  Set the pointer to NULL for any that
	 * are not provided.
	 */

	/* Functions for updating foreign tables */
	AddForeignUpdateTargets_function AddForeignUpdateTargets;
	PlanForeignModify_function PlanForeignModify;
	BeginForeignModify_function BeginForeignModify;
	ExecForeignInsert_function ExecForeignInsert;
	ExecForeignUpdate_function ExecForeignUpdate;
	ExecForeignDelete_function ExecForeignDelete;
	EndForeignModify_function EndForeignModify;
	IsForeignRelUpdatable_function IsForeignRelUpdatable;

	/* Support functions for EXPLAIN */
	ExplainForeignScan_function ExplainForeignScan;
	ExplainForeignModify_function ExplainForeignModify;

	/* Support functions for ANALYZE */
	AnalyzeForeignTable_function AnalyzeForeignTable;

	/*** EDB extensions ***/
	GetTables_function GetTables;
	DescRel_function DescRel;

	BeginInsert_function BeginInsert;
	Insert_function Insert;
	EndInsert_function EndInsert;
	IterateReturning_function IterateReturning;

	BeginUpdate_function BeginUpdate;
	Update_function Update;
	EndUpdate_function EndUpdate;

	BeginDelete_function BeginDelete;
	Delete_function Delete;
	EndDelete_function EndDelete;

	CleanupCache_function CleanupCache;

	int dbmsType;
} FdwRoutine;


/* Functions in foreign/foreign.c */
extern FdwRoutine *GetFdwRoutine(Oid fdwhandler);
extern FdwRoutine *GetFdwRoutineByRelId(Oid relid);
extern FdwRoutine *GetFdwRoutineForRelation(Relation relation, bool makecopy);
extern FdwRoutine *GetFdwRoutineByServerId(Oid serverid);

#endif   /* FDWAPI_H */
