/*-------------------------------------------------------------------------
 *
 * spl.h		- Definitions for the edb-spl
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2013, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 * Portions Copyright (c) 2004-2013, EnterpriseDB Corporation.
 *
 *
 * IDENTIFICATION
 *	  src/pl/edb-spl/src/spl.h
 *
 *-------------------------------------------------------------------------
 */

#ifndef SPL_H
#define SPL_H

#include "postgres.h"

#include "access/xact.h"
#include "commands/event_trigger.h"
#include "commands/trigger.h"
#include "executor/spi.h"

/* EDB: Additional includes for SPL. */
#include "parser/parse_node.h"
#include "utils/seqtree.h"

/**********************************************************************
 * Definitions
 **********************************************************************/

/* define our text domain for translations */
#undef TEXTDOMAIN
#define TEXTDOMAIN PG_TEXTDOMAIN("spl")

#undef _
#define _(x) dgettext(TEXTDOMAIN, x)

/* ----------
 * ROWNO value for cursor params
 * ----------
 */
#define IS_CUR_PARAM -2

/* ----------
 * Compiler's namespace item types
 * ----------
 */
typedef enum
{
	PLPGSQL_NSTYPE_LABEL,
	PLPGSQL_NSTYPE_VAR,
	PLPGSQL_NSTYPE_ROW,
	PLPGSQL_NSTYPE_REC,
	PLPGSQL_NSTYPE_TAB,
	PLPGSQL_NSTYPE_EXCEPTION,
	PLPGSQL_NSTYPE_TYPE
} spl_nstype;

/* ----------
 * Datum array node types
 * ----------
 */
typedef enum
{
	PLPGSQL_DTYPE_VAR,
	PLPGSQL_DTYPE_ROW,
	PLPGSQL_DTYPE_REC,
	PLPGSQL_DTYPE_RECFIELD,
	PLPGSQL_DTYPE_ARRAYELEM,
	PLPGSQL_DTYPE_EXPR,
	PLPGSQL_DTYPE_COLLECTION,
	PLPGSQL_DTYPE_COLL_ELEM,
	PLPGSQL_DTYPE_EXCEPTION
} spl_dtype;

/* ----------
 * Variants distinguished in PLpgSQL_type structs
 * ----------
 */
enum
{
	PLPGSQL_TTYPE_SCALAR,		/* scalar types and domains */
	PLPGSQL_TTYPE_ROW,			/* composite types */
	PLPGSQL_TTYPE_REC,			/* RECORD pseudotype */
	PLPGSQL_TTYPE_PSEUDO,		/* other pseudotypes */
	SPL_TTYPE_NESTED,			/* nested table */
	SPL_TTYPE_ASSOCIATIVE,		/* associative array */
	SPL_TTYPE_CURROW,			/* cursor row type */
	SPL_TTYPE_VAR_COLLECTION,	/* single column collection */
	SPL_TTYPE_ROW_COLLECTION,	/* multi column collection */
};

/* ----------
 * Execution tree node types
 * ----------
 */
enum PLpgSQL_stmt_types
{
	PLPGSQL_STMT_BLOCK,
	PLPGSQL_STMT_ASSIGN,
	PLPGSQL_STMT_IF,
	PLPGSQL_STMT_CASE,
	PLPGSQL_STMT_LOOP,
	PLPGSQL_STMT_WHILE,
	PLPGSQL_STMT_FORI,
	PLPGSQL_STMT_FORS,
	PLPGSQL_STMT_FORC,
	PLPGSQL_STMT_FOREACH_A,
	PLPGSQL_STMT_EXIT,
	PLPGSQL_STMT_RETURN,
	PLPGSQL_STMT_RETURN_NEXT,
	PLPGSQL_STMT_RETURN_QUERY,
	PLPGSQL_STMT_RAISE,
	PLPGSQL_STMT_EXECSQL,
	PLPGSQL_STMT_DYNEXECUTE,
	PLPGSQL_STMT_DYNFORS,
	PLPGSQL_STMT_GETDIAG,
	PLPGSQL_STMT_OPEN,
	PLPGSQL_STMT_FETCH,
	PLPGSQL_STMT_CLOSE,
	PLPGSQL_STMT_PERFORM,
	/* statements added introduced by edb-spl */
	SPL_STMT_PROCALL,
	SPL_STMT_FORALL,
	SPL_STMT_COMMIT,
	SPL_STMT_ROLLBACK,
	SPL_STMT_GOTO,
	SPL_STMT_NULL
};


/* ----------
 * Execution node return codes
 * ----------
 */
enum
{
	PLPGSQL_RC_OK,
	PLPGSQL_RC_EXIT,
	PLPGSQL_RC_RETURN,
	PLPGSQL_RC_CONTINUE,
	PLPGSQL_RC_JUMP
};

/* ----------
 * GET DIAGNOSTICS information items
 * ----------
 */
enum
{
	PLPGSQL_GETDIAG_ROW_COUNT,
	PLPGSQL_GETDIAG_RESULT_OID,
	PLPGSQL_GETDIAG_ERROR_CONTEXT,
	PLPGSQL_GETDIAG_ERROR_DETAIL,
	PLPGSQL_GETDIAG_ERROR_HINT,
	PLPGSQL_GETDIAG_RETURNED_SQLSTATE,
	PLPGSQL_GETDIAG_COLUMN_NAME,
	PLPGSQL_GETDIAG_CONSTRAINT_NAME,
	PLPGSQL_GETDIAG_DATATYPE_NAME,
	PLPGSQL_GETDIAG_MESSAGE_TEXT,
	PLPGSQL_GETDIAG_TABLE_NAME,
	PLPGSQL_GETDIAG_SCHEMA_NAME
};

/* --------
 * RAISE statement options
 * --------
 */
enum
{
	PLPGSQL_RAISEOPTION_ERRCODE,
	PLPGSQL_RAISEOPTION_MESSAGE,
	PLPGSQL_RAISEOPTION_DETAIL,
	PLPGSQL_RAISEOPTION_HINT,
	PLPGSQL_RAISEOPTION_COLUMN,
	PLPGSQL_RAISEOPTION_CONSTRAINT,
	PLPGSQL_RAISEOPTION_DATATYPE,
	PLPGSQL_RAISEOPTION_TABLE,
	PLPGSQL_RAISEOPTION_SCHEMA,
	PLPGSQL_RAISEOPTION_SQLCODE
};


/* --------
 * Behavioral modes for plpgsql variable resolution
 * --------
 */
typedef enum
{
	PLPGSQL_RESOLVE_ERROR,		/* throw error if ambiguous */
	PLPGSQL_RESOLVE_VARIABLE,	/* prefer plpgsql var to table column */
	PLPGSQL_RESOLVE_COLUMN		/* prefer table column to plpgsql var */
} PLpgSQL_resolve_option;


/**********************************************************************
 * Node and structure definitions
 **********************************************************************/


typedef struct PLpgSQL_type
{								/* Postgres data type */
	char	   *typname;		/* (simple) name of the type */
	Oid			typoid;			/* OID of the data type */
	int			ttype;			/* PLPGSQL_TTYPE_ code */
	char		typtype;		/* typtype from the pg_type entry */
	int16		typlen;			/* stuff copied from its pg_type entry */
	bool		typbyval;		/* TRUE -> type is passed by value, otherwise, passed by reference */
	Oid			typrelid;		/* relation id of this type */
	Oid			typioparam;
	Oid			collation;		/* from pg_type, but can be overridden */
	Oid			typbasetype;	/* Base type */
	FmgrInfo	typinput;		/* lookup info for typinput function */
	int32		atttypmod;		/* typmod (taken from someplace else) */

	List	   *columns;		/* user defined record */
	TupleDesc	refcurtupdesc;	/* If this is a strongly typed refcursor, tuple descriptor of the cursor result */

	TupleDesc	collrowtupdesc;	/* If this is a SPL_TTYPE_ROW_COLLECTION, tuple descriptor of the rows stored in the collection */
	int			size_limit;		/* max number of elements, for VARRAY types */
	Oid			valuetypid;		/* OID of the value type */
	int32		valuetypmod;
	struct PLpgSQL_type	*indexbytype;	/* indexing type of a TABLE OF type*/
	bool		typnullconst;	/* SUBTYPEs can be declared NOT NULL */
} PLpgSQL_type;


/*
 * PLpgSQL_datum is the common supertype for PLpgSQL_expr, PLpgSQL_var,
 * PLpgSQL_row, PLpgSQL_rec, PLpgSQL_recfield, and PLpgSQL_arrayelem
 *
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_VAR, PLPGSQL_DTYPE_ROW, ...)
 * dno: Datum number (index into estate->datums[])
 */
typedef struct
{								/* Generic datum array item		*/
	int			dtype;
	int			dno;
} PLpgSQL_datum;

/*
 * In compilation phase, we create a dummy variable in the datums array
 * for each package variable referenced in the code. 'isdummy' field
 * is set to PACKVAR_DUMMY (or FIELDVAR_DUMMY for field variables of a
 * PLpgSQL_row), and they are replaced with pointers to the real, non-dummy
 * datums in the local datums array at the beginning of execution.
 *
 * The values are chosen so that you can do just "!isdummy" to check if
 * a variable is a local variable.
 */
typedef enum
{
	LOCAL_VARIABLE = 0,
	PACKVAR_DUMMY = 1,
	FIELDVAR_DUMMY = 2,
} spl_dummytype;

/*
 * The variants PLpgSQL_var, PLpgSQL_row, and PLpgSQL_rec share these
 * fields
 *
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_VAR)
 * dno: Datum number (index into estate->datums[])
 * refname: Variable name
 * lineno: Line number at which variable was declared
 */
typedef struct
{								/* Scalar or composite variable */
	int			dtype;
	int			dno;
	char	   *refname;
	Oid			packid;
	spl_dummytype isdummy;
	int			lineno;
} PLpgSQL_variable;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_EXPR)
 * dno: Expression number (obsolete)
 * query: Text of equivalent query
 * plan: Execution plan (null until executed first time)
 */
typedef struct PLpgSQL_expr
{								/* SQL Query to plan and execute	*/
	int			dtype;
	int			dno;
	char	   *query;
	SPIPlanPtr	plan;
	Bitmapset  *paramnos;		/* all dnos referenced by this query */
	Bitmapset  *inplace_modifiable;	/* dnos of params that can be modified in-place */

	/* function containing this expr (not set until we first parse query) */
	struct PLpgSQL_function *func;

	/* namespace chain visible to this expr */
	struct PLpgSQL_nsitem *ns;

	bool		isCursorQueryWithArgs;	/* is this an explicit cursor query, with args? */

	/*
	 * An expression for which we don't have a source query, only an Expr
	 */
	Expr	   *nosrc_expr;

	/* fields for "simple expression" fast-path execution: */
	Expr	   *expr_simple_expr;		/* NULL means not a simple expr */
	int			expr_simple_generation; /* plancache generation we checked */
	Oid			expr_simple_type;		/* result type Oid, if simple */

	/*
	 * if expr is simple AND prepared in current transaction,
	 * expr_simple_state and expr_simple_in_use are valid. Test validity by
	 * seeing if expr_simple_lxid matches current LXID.  (If not,
	 * expr_simple_state probably points at garbage!)
	 */
	ExprState  *expr_simple_state;		/* eval tree for expr_simple_expr */
	bool		expr_simple_in_use;		/* true if eval tree is active */
	LocalTransactionId expr_simple_lxid;
} PLpgSQL_expr;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_VAR)
 * dno:	Datum number (index into estate->datums[])
 * refname:	variable name
 * lineno: Line number at which variable was declared
 * datatype: Datatype for this variable
 * isconst: TRUE -> variable is CONST
 * notnull:	TRUE -> variable is NOT NULL
 * default_val: Initial value (expression) for this variable
 * cursor_explicit_expr: Cursor expression (query) for explicit cursor
 * cursor_explicit_argrow: Argument row number for explicit cursor
 * value: Current value of this variable
 * isnull: TRUE -> current value IS NULL
 * freeval: TRUE -> pfree() the value when we free this variable
 */
typedef struct
{								/* Scalar variable */
	int			dtype;
	int			dno;
	char	   *refname;
	Oid			packid;
	spl_dummytype isdummy;
	int			lineno;

	PLpgSQL_type *datatype;
	int			isconst;
	int			notnull;
	PLpgSQL_expr *default_val;
	PLpgSQL_expr *cursor_explicit_expr;
	int			cursor_explicit_argrow;
	int			cursor_options;

	Datum		value;
	bool		isnull;
	bool		freeval;
} PLpgSQL_var;


/*
 * EDB: This is an EDB extension, so we can include the comments inline
 * without causing merge difficulties.  In fact, it's better this way, as
 * it reduces the chance of an unrelated diff hunk accidentally managing
 * to apply to this code.
 */
typedef struct
{								/* User-defined exception */
	int			  dtype;		/* Datum type (PLPGSQL_DTYPE_UDEXCEPTION) */
	int 		  dno;			/* Datum number (index into estate->datums[]) */
	char		 *exceptionname;
	Oid			  packid;
	spl_dummytype isdummy;
	int64		  exceptionid;
	int			  sqlcode;		/* Redwood-style error code assigned to this exception using PRAGMA EXCEPTION_INIT */
	int			  sqlerrstate;	/* PG-style SQLSTATE which is specified in the EXCEPTION_INIT */
} PLpgSQL_udexception;

/*
 * a reduced form of a PLpgSQL_var
 * it does not contain varno, thus it cannot exist in plpgsql_datums
 */
typedef struct
{
	char	   	 *varname;			/* Variable name */
	PLpgSQL_type *datatype;			/* Datatype for this variable */
	bool		  notnull;			/* TRUE -> variable is NOT NULL */
	PLpgSQL_expr *defval;			/* Initial value (expression) for this variable */
} Type_varinfo;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_ROW)
 * dno: row number
 * refname:	name of this variable
 * lineno: Line number at which variable was declared
 * rowtupdesc: tuple descriptor for this row
 * nfields: count of fields in this row
 * fieldnames: field names
 * varnos: current values
 */
typedef struct
{								/* Row variable */
	int			dtype;
	int			dno;
	char	   *refname;
	Oid			packid;
	spl_dummytype	isdummy;
	int			lineno;

	PLpgSQL_type	*datatype;		/* EDB: structure of the row */
	TupleDesc	rowtupdesc;

	PLpgSQL_expr   *default_val; 	/* initial value (expression) for this row */
	/*
	 * Note: TupleDesc is only set up for named rowtypes, else it is NULL.
	 *
	 * Note: if the underlying rowtype contains a dropped column, the
	 * corresponding fieldnames[] entry will be NULL, and there is no
	 * corresponding var (varnos[] will be -1).
	 *
	 * EDB: the indexes in varnos array must be resolved in the right datums
	 * array - if packid is valid, that means in the package's datums array!
	 */
	int			nfields;
	char	  **fieldnames;
	int		   *varnos;
	int num_dropped_atts;			/* number of dropped attributes in a row*/
} PLpgSQL_row;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_REC)
 * dno: Datum number (index into estate->datums[])
 * refname: variable name
 * lineno: line number at which variable was declared
 * tup:	tuple that holds current value for this record
 * tupdesc:	tuple descriptor
 * freetup: TRUE -> free the tuple when we discard this variable
 * freetupdesc:	TRUE -> free the tuple descriptor when we discard this variable
 */
typedef struct
{								/* Record variable (non-fixed structure) */
	int			dtype;
	int			dno;
	char	   *refname;
	Oid			packid;
	spl_dummytype isdummy;
	int			lineno;

	HeapTuple	tup;
	TupleDesc	tupdesc;
	bool		freetup;
	bool		freetupdesc;
} PLpgSQL_rec;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_RECFIELD)
 * dno:	Datum number (index into estate->datums[])
 * fieldname: name of this field
 */
typedef struct
{								/* Field in record */
	int			dtype;
	int			dno;
	char	   *fieldname;
	int			recparentno;	/* dno of parent record */
} PLpgSQL_recfield;


/* Redwood-style collection */
typedef struct
{									/* Generic datum array item		*/
	int			   dtype;			/* Datum type (PLPGSQL_DTYPE_COLLECTION) */
	int		   	   dno;				/* Datum number (index into estate->datums[]) */
	char		  *refname;			/* variable name */
	Oid			  packid;
	spl_dummytype  isdummy;
	int			   lineno;			/* Line number at which variable was declared */
	bool		   isconst;			/* TRUE -> variable is CONST */
	PLpgSQL_type  *datatype;		/* defining structure of the table */

	/*
	 * coll_data is a SEQTREE for nested tables and associative arrays, and
	 * ArrayType * for VARRAYs.
	 */
	Datum		   coll_data;
	SEQTREE_CACHE  cache;
	PLpgSQL_row	  *tmprow;			/* temp variable to construct tuples for insertion */
	PLpgSQL_expr   *default_val; 	/* initial value (expression) for this table */
} Spl_collection;

typedef struct
{								/* collection element */
	int			  dtype;		/* Datum type (PLPGSQL_DTYPE_COLL_ELEM) */
	int 		  dno;			/* Datum number (index into estate->datums[]) */
	char	   	 *colname;		/* column name */
	PLpgSQL_expr *subscript;	/* Subscript expression */
	int			  ttype;		/* table type, SPL_TTYPE_ROW_COLLECTION or SPL_TTYPE_VAR_COLLECTION */
	int			  parentno;
} Spl_coll_elem;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * dtype: Datum type (PLPGSQL_DTYPE_ARRAYELEM)
 * dno: Datum number (index into estate->datums[])
 * subscript: subscript expression
 */
typedef struct
{								/* Element of array variable */
	int			dtype;
	int			dno;
	PLpgSQL_expr *subscript;
	int			arrayparentno;	/* dno of parent array variable */
	/* Remaining fields are cached info about the array variable's type */
	Oid			parenttypoid;	/* type of array variable; 0 if not yet set */
	int32		parenttypmod;	/* typmod of array variable */
	Oid			arraytypoid;	/* OID of actual array type */
	int32		arraytypmod;	/* typmod of array (and its elements too) */
	int16		arraytyplen;	/* typlen of array type */
	Oid			elemtypoid;		/* OID of array element type */
	int16		elemtyplen;		/* typlen of element type */
	bool		elemtypbyval;	/* element type is pass-by-value? */
	char		elemtypalign;	/* typalign of element type */
} PLpgSQL_arrayelem;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * itemtype: namestack item type (PLPGSQL_NSTYPE_LABEL, PLPGSQL_NSTYPE_VAR, ...)
 * itemno: item number (index into spl_Datums[])
 */
typedef struct PLpgSQL_nsitem
{								/* Item in the compilers namespace tree */
	int			itemtype;
	int			itemno;
	Oid			packid;			/* Package Oid  if this is a package variable */
	struct PLpgSQL_nsitem *prev;

	/* EDB: Only used for type items (PLPGSQL_NSTYPE_TYPE) */
	PLpgSQL_type			*type;

	char		name[1];		/* actually, as long as needed */
} PLpgSQL_nsitem;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_RETURN, PLPGSQL_STMT_BLOCK, ...)
 * lineno: line number at which statement appears (-1 for hidden statement)
 */
typedef struct
{								/* Generic execution node		*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
} PLpgSQL_stmt;


typedef struct PLpgSQL_condition
{								/* One EXCEPTION condition name */
	int			sqlerrstate;	/* SQLSTATE code */
	char	   *condname;		/* condition name (for debugging) */
	int64 		exceptionid;	/* Exception Id in case of User Defined Exceptions */
	int			sqlcode;		/* Redwood-style SQLCODE */
	struct PLpgSQL_condition *next;
} PLpgSQL_condition;

typedef struct
{
	/*
	 * EDB: sqlstate_varno and sqlerrm_varno appear in PLpgSQL_execstate,
	 * because they are available everywhere in SPL.
	 */
	List	   *exc_list;		/* List of WHEN clauses */
} PLpgSQL_exception_block;

typedef struct
{								/* One EXCEPTION ... WHEN clause */
	int			lineno;
	PLpgSQL_condition *conditions;
	List	   *action;			/* List of statements */
} PLpgSQL_exception;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_BLOCK)
 * lineno: line number in function source
 * label: optional label for this block of statements
 * n_initvars: number of entries in initvarnos array
 * initvarnos: array of variables requiring initialization upon entry into this
 *   block
 * exceptions: optional exception handlers for this block of statements
 */
typedef struct
{								/* Block of statements			*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	List	   *body;			/* List of statements */
	int			n_initvars;
	int		   *initvarnos;
	PLpgSQL_exception_block *exceptions;
} PLpgSQL_stmt_block;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_ASSIGN)
 * lineno: line number in function source
 * varno: pointer to target variable
 * expr: value to assign (in the form of an expression)
 */
typedef struct
{								/* Assign statement			*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	int			varno;
	PLpgSQL_expr *expr;
} PLpgSQL_stmt_assign;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_PERFORM)
 * lineno: line number in function source
 * expr: query to execute
 */
typedef struct
{								/* PERFORM statement		*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	PLpgSQL_expr *expr;
} PLpgSQL_stmt_perform;

typedef struct
{								/* Get Diagnostics item		*/
	int			kind;			/* id for diagnostic value desired */
	int			target;			/* where to assign it */
} PLpgSQL_diag_item;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_GETDIAG)
 * lineno: line number in function source
 */
typedef struct
{								/* Get Diagnostics statement		*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	bool		is_stacked;		/* STACKED or CURRENT diagnostics area? */
	List	   *diag_items;		/* List of PLpgSQL_diag_item */
} PLpgSQL_stmt_getdiag;

typedef enum PLpgSQL_trigtype
{
	PLPGSQL_DML_TRIGGER,
	PLPGSQL_EVENT_TRIGGER,
	PLPGSQL_NOT_TRIGGER
} PLpgSQL_trigtype;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_IF)
 * lineno: line number in function source
 */
typedef struct
{								/* IF statement				*/
	int			cmd_type;
	int			lineno;
	char		 *label;		/* EDB: optional label for this statement */
	PLpgSQL_expr *cond;			/* boolean expression for THEN */
	List	   *then_body;		/* List of statements */
	List	   *elsif_list;		/* List of PLpgSQL_if_elsif structs */
	List	   *else_body;		/* List of statements */
} PLpgSQL_stmt_if;

typedef struct					/* one ELSIF arm of IF statement */
{
	int			lineno;
	PLpgSQL_expr *cond;			/* boolean expression for this case */
	List	   *stmts;			/* List of statements */
} PLpgSQL_if_elsif;


typedef struct					/* CASE statement */
{
	int			cmd_type;
	int			lineno;
	char	   *label;			/* optional label for this statement */
	PLpgSQL_expr *t_expr;		/* test expression, or NULL if none */
	int			t_varno;		/* var to store test expression value into */
	List	   *case_when_list; /* List of PLpgSQL_case_when structs */
	bool		have_else;		/* flag needed because list could be empty */
	List	   *else_stmts;		/* List of statements */
} PLpgSQL_stmt_case;

typedef struct					/* one arm of CASE statement */
{
	int			lineno;
	PLpgSQL_expr *expr;			/* boolean expression for this case */
	List	   *stmts;			/* List of statements */
} PLpgSQL_case_when;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_LOOP)
 * lineno: line number in function source
 * label: optional label for this block of statements
 */
typedef struct
{								/* Unconditional LOOP statement		*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	List	   *body;			/* List of statements */
} PLpgSQL_stmt_loop;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_WHILE)
 * lineno: line number in function source
 * label: optional label for this block of statements
 * cond: loop condition (in the form of an expression)
 */
typedef struct
{								/* WHILE cond LOOP statement		*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	PLpgSQL_expr *cond;
	List	   *body;			/* List of statements */
} PLpgSQL_stmt_while;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_FORI)
 * lineno: line number in function source
 * label: optional label for this block of statements
 * var: loop control variable
 * lower: starting value expression
 * upper: ending value expression
 * reverse: 1 -> reverse iteration
 */
typedef struct
{								/* FOR statement with integer loopvar	*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	PLpgSQL_var *var;
	PLpgSQL_expr *lower;
	PLpgSQL_expr *upper;
	PLpgSQL_expr *step;			/* NULL means default (ie, BY 1) */
	int			reverse;
	List	   *body;			/* List of statements */
} PLpgSQL_stmt_fori;


/*
 * EDB: This is an EDB extension, so we can include the comments inline
 * without causing merge difficulties.  In fact, it's better this way, as
 * it reduces the chance of an unrelated diff hunk accidentally managing
 * to apply to this code.
 */
typedef struct
{								/* FORALL statement with integer loopvar */
	int			  cmd_type;		/* statement type (PLPGSQL_STMT_FORALL) */
	int			  lineno;		/* line number in function source */
	char		 *label;		/* optional label for this block of statements */
	PLpgSQL_var  *var;			/* loop control variable */
	PLpgSQL_expr *lower;		/* starting value expression */
	PLpgSQL_expr *upper;		/* ending value expression */
	PLpgSQL_stmt *body;			/* PLpgSQL_stmt_execsql */
} SPL_stmt_forall;


/*
 * PLpgSQL_stmt_forq represents a FOR statement running over a SQL query.
 * It is the common supertype of PLpgSQL_stmt_fors, PLpgSQL_stmt_forc
 * and PLpgSQL_dynfors.
 */
typedef struct
{
	int			cmd_type;
	int			lineno;
	char	   *label;
	PLpgSQL_rec *rec;
	PLpgSQL_row *row;
	Spl_coll_elem *coll_elem;
	List	   *body;			/* List of statements */
} PLpgSQL_stmt_forq;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_FORS)
 * lineno: line number in function source
 * label: optional label for this block of statements
 * rec: Control variable (if it's a RECORD)
 * row: Control variable (if it's a ROW)
 */
typedef struct
{								/* FOR statement running over SELECT	*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	PLpgSQL_rec *rec;
	PLpgSQL_row *row;
	Spl_coll_elem *coll_elem;	/* EDB: Control variable (if it's a TABROW) */
	List	   *body;			/* List of statements */
	/* end of fields that must match PLpgSQL_stmt_forq */
	PLpgSQL_expr *query;
} PLpgSQL_stmt_fors;

typedef struct
{								/* FOR statement running over cursor	*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	PLpgSQL_rec *rec;
	PLpgSQL_row *row;
	Spl_coll_elem *coll_elem;
	List	   *body;			/* List of statements */
	/* end of fields that must match PLpgSQL_stmt_forq */
	int			curvar;
	PLpgSQL_expr *argquery;		/* cursor arguments if any */
} PLpgSQL_stmt_forc;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_DYNFORS)
 * lineno: line number in function source
 * label: optional label for this block of statements
 * rec:	control variable (if it's a RECORD)
 * row: control variable (if it's a ROW)
 */
typedef struct
{								/* FOR statement running over EXECUTE	*/
	int			cmd_type;
	int			lineno;
	char	   *label;
	PLpgSQL_rec *rec;
	PLpgSQL_row *row;
	Spl_coll_elem *coll_elem;	/* EDB: control variable (if it's a TABROW) */
	List	   *body;			/* List of statements */
	/* end of fields that must match PLpgSQL_stmt_forq */
	PLpgSQL_expr *query;
	List	   *params;			/* USING expressions */
} PLpgSQL_stmt_dynfors;


typedef struct
{								/* FOREACH item in array loop */
	int			cmd_type;
	int			lineno;
	char	   *label;
	int			varno;			/* loop target variable */
	int			slice;			/* slice dimension, or 0 */
	PLpgSQL_expr *expr;			/* array expression */
	List	   *body;			/* List of statements */
} PLpgSQL_stmt_foreach_a;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_OPEN)
 * lineno: line number in function source
 * curvar: pointer to cursor variable
 * returntype: cursor row type
 * argquery: parameterized query
 * query: simple query (OPEN refcursor FOR SELECT)
 * dynquery: dynamic query (OPEN refcursor FOR EXECUTE)
 * params: USING expressions
 */
typedef struct
{								/* OPEN a curvar					*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	int			curvar;
	int			cursor_options;
	PLpgSQL_row *returntype;
	PLpgSQL_expr *argquery;
	PLpgSQL_expr *query;
	PLpgSQL_expr *dynquery;
	List	   *params;			/* USING expressions */
} PLpgSQL_stmt_open;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_FETCH)
 * lineno: line number in function source
 */
typedef struct
{								/* FETCH or MOVE statement */
	int			cmd_type;
	int			lineno;
	char		  *label;		/* EDB: optional label for this statement */
	PLpgSQL_rec *rec;			/* target, as record or row */
	PLpgSQL_row *row;
	Spl_coll_elem *coll_elem;	/* EDB: target tabrow */
	int			curvar;			/* cursor variable to fetch from */
	FetchDirection direction;	/* fetch direction */
	long		how_many;		/* count, if constant (expr is NULL) */
	PLpgSQL_expr *expr;			/* count, if expression */
	bool		is_move;		/* is this a fetch or move? */
	bool		returns_multiple_rows;	/* can return more than one row? */
	bool		   bulkcollect;	/* EDB: fetch using BULK COLLECT */
	PLpgSQL_expr  *limit;		/* EDB: BULK COLLECT LIMIT expression */
} PLpgSQL_stmt_fetch;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_CLOSE)
 * lineno: line number in function source
 * curvar: cursor to close
 */
typedef struct
{								/* CLOSE curvar						*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	int			curvar;
} PLpgSQL_stmt_close;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_EXIT)
 * lineno: line number in function source
 * cond: optional exit condition
 */
typedef struct
{								/* EXIT or CONTINUE statement			*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	bool		is_exit;		/* Is this an exit or a continue? */
	PLpgSQL_expr *cond;
} PLpgSQL_stmt_exit;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_RETURN)
 * lineno: line number in function source
 * expr: value to return (if an expression)
 * retvarno: value to return (if a variable)
 */
typedef struct
{								/* RETURN statement			*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	PLpgSQL_expr *expr;
	int			retvarno;
	int			retval_varno;	/* EDB: the variable to which return value
								 * should be in case we have added dummy
								 * _retval_ OUT parameter */
} PLpgSQL_stmt_return;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_RETURN_NEXT)
 * lineno: line number in function source
 * expr: value to add to result set (if an expression)
 * retvarno: value to add to result set (if a variable)
 */
typedef struct
{								/* RETURN NEXT statement */
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	PLpgSQL_expr *expr;
	int			retvarno;
} PLpgSQL_stmt_return_next;

typedef struct
{								/* RETURN QUERY statement */
	int			cmd_type;
	int			lineno;
	char	   *label;			/* optional label for this statement */
	PLpgSQL_expr *query;		/* if static query */
	PLpgSQL_expr *dynquery;		/* if dynamic query (RETURN QUERY EXECUTE) */
	List	   *params;			/* USING arguments for dynamic query */
} PLpgSQL_stmt_return_query;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_RAISE)
 * lineno: line number in function source
 * elog_level: severity
 */
typedef struct
{								/* RAISE statement			*/
	int			cmd_type;
	int			lineno;
	char	   *label;			/* EDB: optional label for this statement */
	int			elog_level;
	PLpgSQL_expr *ecode_expr;	/* EDB: value for error code (expression) */
	char	   *condname;		/* condition name, SQLSTATE, or NULL */
	char	   *message;		/* old-style message format literal, or NULL */
	int64	   exceptionid;		/* EDB: Variable Id for User Defined Exceptions */
	int		   sqlcode;			/* EDB: Redwood-style SQLCODE */
	List	   *params;			/* list of expressions for old-style message */
	List	   *options;		/* list of PLpgSQL_raise_option */
} PLpgSQL_stmt_raise;

typedef struct
{								/* RAISE statement option */
	int			opt_type;
	PLpgSQL_expr *expr;
} PLpgSQL_raise_option;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_EXECSQL)
 * lineno: line number in function source
 * sqlstmt: SQL statement to execute
 */
typedef struct
{								/* Generic SQL statement to execute */
	int			cmd_type;
	int			lineno;
	char	    *label;			/* EDB: optional label for this statement */
	PLpgSQL_expr *sqlstmt;
	bool		mod_stmt;		/* is the stmt INSERT/UPDATE/DELETE? */
	/* note: mod_stmt is set when we plan the query */
	bool		into;			/* INTO supplied? */
	bool		strict;			/* INTO STRICT flag */
	bool		bulkcollect;	/* EDB: BULK COLLECT INTO specified */
	PLpgSQL_rec *rec;			/* INTO target, if record */
	PLpgSQL_row *row;			/* INTO target, if row */
	Spl_coll_elem *coll_elem;	/* EDB: INTO target, if tabrow */
} PLpgSQL_stmt_execsql;


/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * cmd_type: statement type (PLPGSQL_STMT_DYNEXECUTE)
 * lineno: line number in function source
 */
typedef struct
{								/* Dynamic SQL string to execute */
	int			cmd_type;
	int			lineno;
	char	    *label;			/* EDB: optional label for this statement */
	PLpgSQL_expr *query;		/* string expression */
	bool		into;			/* INTO supplied? */
	bool		strict;			/* INTO STRICT flag */
	bool		  bulkcollect;	/* EDB: BULK COLLECT INTO specified */
	PLpgSQL_rec *rec;			/* INTO target, if record */
	PLpgSQL_row *row;			/* INTO target, if row */
	Spl_coll_elem *coll_elem;	/* EDB: INTO target, if tabrow */
	List	   *params;			/* USING expressions */
} PLpgSQL_stmt_dynexecute;

typedef struct
{								/* commit statement */
	int			   cmd_type;	/* statement type (SPL_STMT_COMMIT) */
	int			   lineno;		/* line number in function source */
	char		  *label;		/* optional label for this statement */
} SPL_stmt_commit;

typedef struct
{								/* rollback statement */
	int			   cmd_type;	/* statement type (SPL_STMT_ROLLBACK) */
	int			   lineno;		/* line number in function source */
	char		  *label;		/* optional label for this statement */
} SPL_stmt_rollback;

typedef struct					
{								/* procedure call stmt		*/
	int			 cmd_type;		/* statement type (SPL_STMT_PROCALL) */
	int			 lineno;		/* line number in function source */
	char		*label;			/* optional label for this statement */
	PLpgSQL_expr	*callexpr;
} SPL_stmt_procall;

typedef struct					/* goto statement */
{
	int			cmd_type;		/* statement type (SPL_STMT_GOTO) */
	int			lineno;			/* line number in function source */
	char	   *label;			/* optional label for this statement */
	char	   *dest_label;		/* label to which control will be jump */
} SPL_stmt_goto;

/* NULL stmt */
typedef PLpgSQL_stmt SPL_stmt_null;

typedef struct PLpgSQL_func_hashkey
{								/* Hash lookup key for functions */
	Oid			funcOid;

	bool		isTrigger;		/* true if called as a trigger */

	/* be careful that pad bytes in this struct get zeroed! */

	/*
	 * For a trigger function, the OID of the relation triggered on is part of
	 * the hash key --- we want to compile the trigger separately for each
	 * relation it is used with, in case the rowtype is different.	Zero if
	 * not called as a trigger.
	 */
	Oid			trigrelOid;

	/*
	 * We must include the input collation as part of the hash key too,
	 * because we have to generate different plans (with different Param
	 * collations) for different collation settings.
	 */
	Oid			inputCollation;

	/*
	 * We include actual argument types in the hash key to support polymorphic
	 * PLpgSQL functions.  Be careful that extra positions are zeroed!
	 */
	Oid			argtypes[FUNC_MAX_ARGS];
} PLpgSQL_func_hashkey;

typedef struct PLpgSQL_depelem
{										/* Package dependency element */
	Oid				objid;				/* OID of dependency */
	TransactionId	objxmin;			/* XMin (used to validate cache) */
	ItemPointerData objtid;				/* Item ID of object tuple (used with objxmin) */
} PLpgSQL_depelem;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * fn_oid: OID of pg_proc tuple for this function
 * fn_xmin:	XMin of pg_proc tuple (used to validate cache)
 * fn_tid: Item ID of pg_proc tuple (used with fn_xmin to validate cache)
 * fn_cxt: Memory context to use for this function (not for an invocation of
 *    this function)
 * fn_rettype: OID of pg_type that describes the type of the return value
 * fn_rettyplen: Length of returned-value  data type
 * fn_retbyval: TRUE -> return type is pass-by-value
 * fn_retinput: Return type information
 * fn_rettypioparam: Type OID for input functions
 * fn_retistuple: TRUE -> function returns a tuple (not a scalar) or a
 *    set of tuples
 * fn_retset: TRUE -> function returns a set of values
 * fn_readonly: TRUE -> function is stable/immutable
 * fn_nargs: number of arguments for this function
 * fn_argvarnos[FUNC_MAX_ARGS]:	estate->datums[] indexes for argument variables
 * ndatums: Count of entries in datums[] array
 * datums: Local variables for this function
 * action: First statement for this function (actually, the top-level block
 *    of statements)
 *
 * The various *_varno entries are estate->datums[] indexes for various
 * magical variables.  NEW, OLD, and everything beginning with TG are
 * applicable only to triggers.
 */
typedef struct PLpgSQL_function
{								/* Complete compiled function	  */
	char	   *fn_signature;
	Oid			fn_oid;
	Oid			fn_packageoid;	/* EDB: OID of package this proc belongs to */
	TransactionId fn_xmin;
	ItemPointerData fn_tid;
	PLpgSQL_trigtype fn_is_trigger;
	Oid			fn_input_collation;
	PLpgSQL_func_hashkey *fn_hashkey;	/* back-link to hashtable key */
	MemoryContext fn_cxt;

	Oid			fn_rettype;
	Oid			fn_spl_rettype;	/* EDB: OID of the pg_type that describes the
								 * type of _retval_ OUT parameter, which is
								 * the real return type of the SPL function */
	int			fn_rettyplen;
	bool		fn_retbyval;
	FmgrInfo	fn_retinput;
	Oid			fn_rettypioparam;
	bool		fn_retistuple;
	bool		fn_retset;
	bool		fn_readonly;

	int			fn_nargs;
	int			fn_argvarnos[FUNC_MAX_ARGS];
	int			out_param_varno;
	int			found_varno;
	int			notfound_varno;					/* EDB */
	int			rowcount_varno;					/* EDB */
	int			sqlstate_varno;	/* EDB (moved from PLpgSQL_exception_block) */
	int			sqlcode_varno;					/* EDB */
	int			sqlerrm_varno;	/* EDB (moved from PLpgSQL_exception_block) */
	int			new_varno;
	int			old_varno;
	int			tg_name_varno;
	int			tg_when_varno;
	int			tg_level_varno;
	int			tg_op_varno;
	int			tg_relid_varno;
	int			tg_relname_varno;
	int			tg_table_name_varno;
	int			tg_table_schema_varno;
	int			tg_nargs_varno;
	int			tg_argv_varno;

	/*
	 * EDB: estate->datums indees for INSERTING, UPDATING, and DELETING
	 * (triggers only)
	 */
	int		   tg_inserting_varno;
	int		   tg_updating_varno;
	int		   tg_deleting_varno;

	/* for event triggers */
	int			tg_event_varno;
	int			tg_tag_varno;

	PLpgSQL_resolve_option resolve_option;
	OutParamStyle		   out_param_style;

	List      *depelems;	/* EDB: List of package dependencies */

	int			ndatums;
	PLpgSQL_datum **datums;
	PLpgSQL_stmt_block *action;

	int			pkg_body_lineno;	/* EDB: lineno in package body, or -1 */
	bool		wrapped;			/* EDB: body is wrapped */
	List	   *saved_plans;		/* EDB: saved SPI plans for this function */
	char	   *argmodes;			/* EDB: argument modes */
	bool		isinitializer;		/* EDB: Is this a package variable initializer? */

	/* these fields change when the function is used */
	struct PLpgSQL_execstate *cur_estate;
	unsigned long use_count;
} PLpgSQL_function;

typedef struct PLpgSQL_package
{								/* Complete compiled package	  */
	char	   		 *pkg_name;		/* Package name */
	char			 *nsp_name;		/* Namespace name */
	Oid				  pkg_oid;		/* OID of pg_namespace tuple */
	int				  packno;	  	/* Index into ga_CompPkgs[] array */
	bool			  initialized;	/* has initializer been executed? */
	bool			  pkgsecdef;	/* Package Security definer 	 */ 
	Oid				  pkgowner;		/* Package Owner Oid			*/
	TransactionId 	  pkg_xmin;		/* XMin of pg_namespace tuple (used to validate cache) */
	ItemPointerData   pkg_tid;		/* TID  of pg_namespace tuple (used to validate cache) */
	PLpgSQL_function *initializer;	/* Compiled initializer for package vars */
	PLpgSQL_function *constructor;	/* Compiled constructor */
	MemoryContext 	  pkg_cxt;		/* Memory context for this package */
	int				  ndatums;		/* Number of datums currently in use (in datums[] array) */
	PLpgSQL_datum 	**datums;		/* Array of package variables */
	struct PLpgSQL_nsitem *ns;		/* namespace stack of package variables (only one level) */
} PLpgSQL_package;

typedef struct Package_CompState
{
	int				 pcurr_index;
	bool			 isinit;
	PLpgSQL_package	*currpack;
	PLpgSQL_nsitem	*ns;
} Package_CompState;

/*
 * EDB: Comments on structure members follow.  We don't include these
 * inline, to avoid merge conflicts.
 *
 * retval: Value to return to caller
 * retisnull: TRUE -> returning NULL to caller
 * retistuple: TRUE -> returning a tuple to caller
 * retisset: TRUE -> returning a SET OF values to caller
 * readonly_func: TRUE -> function is IMMUTABLE (i.e. not VOLATILE)
 * rettupdesc: Tuple descriptor for return value
 * tuple_store_cxt: Memory context for tuples in tuple_store
 * rsi: Result set context (for set-returning functions)
 * found_varno: index of FOUND (within datums[])
 * ndatums: number of entries in datums[] array
 * datums: runtime values for local variables
 * eval_tuptable: tuple table for evaluating expressions/queries
 * eval_processed: number of tuples processed when evaluating an
 *    expression/query
 * eval_lastoid: OID of most recently inserted/updated/deleted tuple
 * eval_econtext: expression context for evaluating an expression
 * cur_expr: current query/expr being evaluated
 */
typedef struct PLpgSQL_execstate
{								/* Runtime execution data	*/
	PLpgSQL_function *func;		/* function being executed */
	Oid					   pkg_oid;			/* EDB: package oid */	

	Datum		retval;
	bool		retisnull;
	Oid			rettype;		/* type of current retval */

	Oid			fn_rettype;		/* info about declared function rettype */
	bool		retistuple;
	bool		retisset;

	bool		readonly_func;

	TupleDesc	rettupdesc;
	char	   *exitlabel;		/* the "target" label of the current EXIT or
								 * CONTINUE stmt, if any */
	ErrorData  *cur_error;		/* current exception handler's error */

	Tuplestorestate *tuple_store;		/* SRFs accumulate results here */
	MemoryContext tuple_store_cxt;
	ResourceOwner tuple_store_owner;
	MemoryContext localVarCxt;		/* EDB: Memory context for local vars */
	ReturnSetInfo *rsi;

	int			found_varno;
	int			notfound_varno;	/* EDB: index of NOTFOUND (within datums[]) */
	int			rowcount_varno;	/* EDB: index of ROWCOUNT (within datums[]) */
	int			sqlstate_varno;	/* EDB: index of SQLSTATE (within datums[]) */
	int			sqlcode_varno;	/* EDB: index of SQLCODE (within datums[]) */
	int			sqlerrm_varno;	/* EDB: index of SQLERRM (within datums[]) */
	int			ndatums;
	PLpgSQL_datum **datums;

	/* temporary state for results from evaluation of query or expr */
	SPITupleTable *eval_tuptable;
	uint32		eval_processed;
	Oid			eval_lastoid;
	ExprContext *eval_econtext; /* for executing simple expressions */
	PLpgSQL_expr *cur_expr;		/* current query/expr being evaluated */

	/* status information for error context reporting */
	PLpgSQL_stmt *err_stmt;		/* current stmt */
	const char *err_text;		/* additional state info */

	void	   *plugin_info;	/* reserved for use by optional plugin */

	MemoryContext procall_cxt; 	/* EDB: Memory context for procedure call */ 

	/* For OUT-param handling */
	Datum	   *new_values;
	bool	   *new_isnulls;
	Oid		   *new_types;

	bool				   fn_wrapped;      /* function body is wrapped ? */
	char 				  *jump_to_label; 	/* destination for execution control */
} PLpgSQL_execstate;

typedef struct					/* OUT variable */
{
	bool			isinout;	/* TRUE -> INOUT, FALSE-> OUT */
	PLpgSQL_datum	*datum;		/* pointer to the datum */
	MemoryContext	valcontext;	/* context in which to allocate new value */
	PLpgSQL_execstate	*caller_estate;
} SPL_out_var;

/*
 * A PLpgSQL_plugin structure represents an instrumentation plugin.
 * To instrument PL/pgSQL, a plugin library must access the rendezvous
 * variable "PLpgSQL_plugin" and set it to point to a PLpgSQL_plugin struct.
 * Typically the struct could just be static data in the plugin library.
 * We expect that a plugin would do this at library load time (_PG_init()).
 * It must also be careful to set the rendezvous variable back to NULL
 * if it is unloaded (_PG_fini()).
 *
 * This structure is basically a collection of function pointers --- at
 * various interesting points in spl_exec.c, we call these functions
 * (if the pointers are non-NULL) to give the plugin a chance to watch
 * what we are doing.
 *
 *	func_setup is called when we start a function, before we've initialized
 *	the local variables defined by the function.
 *
 *	func_beg is called when we start a function, after we've initialized
 *	the local variables.
 *
 *	func_end is called at the end of a function.
 *
 *	stmt_beg and stmt_end are called before and after (respectively) each
 *	statement.
 *
 * Also, immediately before any call to func_setup, PL/pgSQL fills in the
 * error_callback, assign_expr, and get_package fields with pointers to its
 * own plpgsql_exec_error_callback, exec_assign_expr, and getPackage functions.	
 * This is a somewhat ad-hoc expedient to simplify life for debugger plugins.
 */

typedef struct
{
	/* Function pointers set up by the plugin */
	void		(*func_setup) (PLpgSQL_execstate *estate, PLpgSQL_function *func);
	void		(*func_beg) (PLpgSQL_execstate *estate, PLpgSQL_function *func);
	void		(*func_end) (PLpgSQL_execstate *estate, PLpgSQL_function *func);
	void		(*stmt_beg) (PLpgSQL_execstate *estate, PLpgSQL_stmt *stmt);
	void		(*stmt_end) (PLpgSQL_execstate *estate, PLpgSQL_stmt *stmt);

	/* Function pointers set by PL/pgSQL itself */
	void		(*error_callback) (void *arg);
	void		(*assign_expr) (PLpgSQL_execstate *estate, PLpgSQL_datum *target,
											PLpgSQL_expr *expr);
	PLpgSQL_package * (*get_package) (Oid namespaceOid);
} PLpgSQL_plugin;


/* Struct types used during parsing */

typedef struct
{
	char	   *ident;			/* palloc'd converted identifier */
	bool		quoted;			/* Was it double-quoted? */
} PLword;

typedef struct
{
	List	   *idents;			/* composite identifiers (list of String) */
} PLcword;

typedef struct
{
	PLpgSQL_datum *datum;		/* referenced variable */
	char	   *ident;			/* valid if simple name */
	bool		quoted;
	List	   *idents;			/* valid if composite name */
} PLwdatum;


typedef struct
{
	PLpgSQL_expr *expr;
	Oid packid;
} parser_setup_arg;

#include "spl_renames.h"
#define plpgsql_yylval			spl_yylval
#define plpgsql_yylloc			spl_yylloc
#define plpgsql_yyleng			spl_yyleng

/**********************************************************************
 * Global variable declarations
 **********************************************************************/

typedef enum
{
	IDENTIFIER_LOOKUP_NORMAL,	/* normal processing of var names */
	IDENTIFIER_LOOKUP_DECLARE,	/* In DECLARE --- don't look up names */
	IDENTIFIER_LOOKUP_EXPR		/* In SQL expression --- special case */
} IdentifierLookup;

extern IdentifierLookup plpgsql_IdentifierLookup;

extern int	plpgsql_variable_conflict;
extern int  spl_out_param_style;
extern int	spl_max_error_count;

extern bool plpgsql_check_syntax;
extern bool plpgsql_DumpExecTree;

extern PLpgSQL_stmt_block *plpgsql_parse_result;
/* list of argnos for INOUT params that should be treated like OUT params */
extern List	   *spl_nullify_list;

extern int	plpgsql_nDatums;
extern PLpgSQL_datum **plpgsql_Datums;

/* variables for packages */
extern int				   packages_alloc;
extern PLpgSQL_package**   ga_CompPkgs;
extern Package_CompState   cstate;

extern char *plpgsql_error_funcname;

extern PLpgSQL_function *plpgsql_curr_compile;
extern MemoryContext spl_comp_tmp_cxt;

extern PLpgSQL_plugin **plugin_ptr;

/**********************************************************************
 * Function declarations
 **********************************************************************/

/*
 * Like ereport(ERROR, ...), but if 'cancontinue' is true, doesn't immediately
 * stop execution. Instead, the error is added to a list of soft errors, which
 * will be reported to the client (or thrown to the caller, which might catch
 * it) at the end of SPL compilation. With cancontinue == false, this is just
 * like regular ereport(ERROR, ...).
 *
 * There is no convenient way to construct an error without actually throwing
 * it, so we first throw the error with ereport(), and immediately catch it.
 */
#define compile_ERROR(cancontinue, rest)								\
	do {																\
		MemoryContext oldcontext = CurrentMemoryContext;				\
		PG_TRY();														\
		{																\
			ereport(ERROR, rest);										\
		}																\
		PG_CATCH();														\
		{																\
			if (!cancontinue)											\
				PG_RE_THROW();											\
			spl_remember_compilation_error(true);						\
			MemoryContextSwitchTo(oldcontext);							\
		}																\
		PG_END_TRY();													\
	} while(0) \

/* ----------
 * Functions in spl_pack.c
 * ---------- 
 */
extern void init_package_array(void);
extern void initialize_package(PLpgSQL_package *newpack);

extern PLpgSQL_package *compile_package(Oid packOid);

extern PLpgSQL_package *getPackage(Oid namespaceOid);
extern bool check_packfunc_depend(List *depelems);

extern void mark_od_depend(HeapTuple packTup);

extern int get_package_index(Oid oid);
extern PLpgSQL_datum* package_datum_lookup(const char *name, const char *pack,
										   const char *schema, int *nnames);
extern PLpgSQL_datum* package_datum_od_lookup(const char *name,
											  const char *pack,
											  const char *schema);
extern PLpgSQL_udexception* spl_lookup_package_exception(const char *name,
														 const char *pack,
														 const char *schema);

extern void spl_init_compile_error_callback(void *arg);

/* ----------
 * Functions in pl_comp.c
 * ----------
 */

extern void package_cstate_setup(void);

extern PLpgSQL_function *plpgsql_compile(FunctionCallInfo fcinfo,
				bool forValidator);
extern PLpgSQL_function *spl_compile_inline(char *proc_source, List *args,
											List **compilation_errors);
extern void spl_throw_compilation_errors(List *errors, bool forPackage);

extern void plpgsql_parser_setup(struct ParseState *pstate,
					 PLpgSQL_expr *expr);
extern PLpgSQL_function *do_compile(FunctionCallInfo fcinfo,
						HeapTuple procTup,
						PLpgSQL_function *function,
						PLpgSQL_func_hashkey *hashkey,
						bool forValidator,
						List **compilation_errors);
extern bool plpgsql_parse_word(char *word1, const char *yytxt,
				   PLwdatum *wdatum, PLword *word);
extern bool plpgsql_parse_dblword(char *word1, char *word2,
					  PLwdatum *wdatum, PLcword *cword);
extern bool plpgsql_parse_tripword(char *word1, char *word2, char *word3,
					   PLwdatum *wdatum, PLcword *cword);
extern bool plpgsql_parse_fourword(char *word1, char *word2, char *word3, char *word4,
						PLwdatum *wdatum, PLcword *cword);
extern PLpgSQL_type *plpgsql_parse_wordtype(char *ident);
extern PLpgSQL_type *plpgsql_parse_cwordtype(List *idents);
extern PLpgSQL_type *plpgsql_parse_wordrowtype(char *ident);
extern PLpgSQL_type *plpgsql_parse_cwordrowtype(List *idents);
extern PLpgSQL_type *plpgsql_build_datatype(Oid typeOid, int32 typmod,
					   Oid collation);
extern PLpgSQL_variable *plpgsql_build_variable(const char *refname, int lineno,
					   PLpgSQL_type *dtype,
					   bool add2namespace);
extern PLpgSQL_udexception *spl_build_udexception(const char *refname,
					  int lineno, bool add2namespace, Oid ExpOid, int sqlcode, int errstate);
extern PLpgSQL_rec *plpgsql_build_record(const char *refname, int lineno,
										 bool add2namespace);
extern int	plpgsql_recognize_err_condition(const char *condname,
										bool allow_sqlstate);
extern PLpgSQL_condition *spl_parse_err_condition(char *condname);
extern void plpgsql_adddatum(PLpgSQL_datum *new);
extern int	plpgsql_add_initdatums(int **varnos);
extern void plpgsql_HashTableInit(void);

extern PLpgSQL_type * build_local_datatype(char *typname, List *cols);
extern PLpgSQL_type * build_local_refcursor_type(char *typname, PLpgSQL_type *rettype);
extern PLpgSQL_type * build_local_collection_type(char *typname,
							PLpgSQL_type *typ, PLpgSQL_type *indexbytype,
							char typtype, int size_limit);
extern int spl_var_getdno(PLpgSQL_datum *d);
extern int spl_nsitem_getdno(PLpgSQL_nsitem *nse);
extern PLpgSQL_datum *spl_build_dummy_variable(const char *refname,
											   PLpgSQL_type *dtype,
											   Oid packageoid, bool isconst);
extern PLpgSQL_row *spl_build_row_from_function_parameters(const char *relname,
														   List *args);
extern void spl_compile_error_callback(void *arg);
extern void spl_remember_compilation_error(bool checkerrcount);

/* ----------
 * Functions in pl_handler.c
 * ----------
 */
extern void _PG_init(void);
extern Datum spl_call_handler(PG_FUNCTION_ARGS);
extern Datum spl_inline_handler(PG_FUNCTION_ARGS);
extern Datum spl_validator(PG_FUNCTION_ARGS);

extern void push_exec_state(int *prev_ndatums, PLpgSQL_datum ***prev_datums);
extern void pop_exec_state(int prev_ndatums, PLpgSQL_datum **prev_datums);

/* ----------
 * Functions in pl_exec.c
 * ----------
 */
extern Oid exec_get_datum_type(PLpgSQL_execstate *estate,
					PLpgSQL_datum *datum);
extern int32 exec_get_datum_typmod(PLpgSQL_execstate *estate,
					PLpgSQL_datum *datum);
extern void exec_get_datum_type_info(PLpgSQL_execstate *estate,
						 PLpgSQL_datum *datum,
						 Oid *typeid, int32 *typmod, Oid *collation);
extern Datum spl_exec_function(PLpgSQL_function *func,
					  FunctionCallInfo fcinfo);
extern HeapTuple spl_exec_trigger(PLpgSQL_function *func,
					 TriggerData *trigdata);
extern void spl_exec_event_trigger(PLpgSQL_function *func,
					 EventTriggerData *trigdata);
extern void spl_xact_cb(XactEvent event, void *arg, bool spl_context);
extern void spl_subxact_cb(SubXactEvent event, SubTransactionId mySubid,
		                               SubTransactionId parentSubid, void *arg,
									   bool spl_context);
extern Datum spl_package_variable_eval(Oid packOid, char *varname,
						  Oid expectedtypeid, bool *isnull);
extern void spl_package_variable_store(Oid packOid, char *varname,
						   Datum value, bool isnull, Oid typeid);
extern PLpgSQL_variable *get_package_variable(Oid packOid, char *varname);

extern int tabindex_compare_string(const void *key1, 
						const void *key2, Size keysize);
extern int tabindex_compare_datum(const void *key1,
						const void *key2, Size keysize);

/* ----------
 * Functions for namespace handling in pl_funcs.c
 * ----------
 */
extern void plpgsql_ns_init(void);
extern void plpgsql_ns_push(const char *label);
extern void plpgsql_ns_pop(void);
extern PLpgSQL_nsitem *plpgsql_ns_top(void);
extern void plpgsql_ns_additem(int itemtype, int itemno, const char *name);
extern void spl_ns_additem_type(PLpgSQL_type *type, const char *name);
extern PLpgSQL_nsitem *plpgsql_ns_lookup(PLpgSQL_nsitem *ns_cur, bool localmode,
				  const char *name1, const char *name2,
				  const char *name3, int *names_used);
extern PLpgSQL_nsitem *spl_ns_lookup_type(PLpgSQL_nsitem *ns_cur, bool localmode,
										  const char *name1, const char *name2);
extern PLpgSQL_nsitem *spl_ns_lookup_exception(PLpgSQL_nsitem *ns_cur, bool localmode,
										  const char *name1, const char *name2);
extern PLpgSQL_nsitem *plpgsql_ns_lookup_label(PLpgSQL_nsitem *ns_cur,
						const char *name);

/* ----------
 * Other functions in pl_funcs.c
 * ----------
 */
extern const char *plpgsql_stmt_typename(PLpgSQL_stmt *stmt);
extern const char *plpgsql_getdiag_kindname(int kind);
extern void plpgsql_free_function_memory(PLpgSQL_function *func);
extern void plpgsql_dumptree(PLpgSQL_function *func);
extern bool	get_package_security_definer(Oid nspOid, Oid * ownerId);

/* ----------
 * Scanner functions in pl_scanner.c
 * ----------
 */
extern int	plpgsql_base_yylex(void);
extern int	plpgsql_yylex(void);
extern int	spl_raw_yylex(void);
extern void plpgsql_push_back_token(int token);
extern bool plpgsql_token_is_unreserved_keyword(int token);
extern void spl_save_token(void);
extern void spl_push_back_saved_token(int token);
extern void plpgsql_append_source_text(StringInfo buf,
						   int startlocation, int endlocation);
extern int	plpgsql_peek(void);
extern void plpgsql_peek2(int *tok1_p, int *tok2_p, int *tok1_loc,
			  int *tok2_loc);
extern int	plpgsql_scanner_errposition(int location);
extern void spl_yyerror(const char *message, bool continue_execution);
extern int	plpgsql_location_to_lineno(int location);
extern int	plpgsql_latest_lineno(void);
extern void plpgsql_scanner_init(const char *str, int *pkg_body_lineno);
extern void plpgsql_scanner_finish(void);

extern char *spl_parse_cursor_varstring(char *query, bool *hasargs);

/* ----------
 * Externs in spl_gram.y
 * ----------
 */
extern int	plpgsql_yyparse(void);
extern void label_stack_init(void);
extern void check_assignable(PLpgSQL_datum *datum, int location);
extern bool check_assignable_ext(PLpgSQL_datum *datum, int location, bool error);
extern	PLpgSQL_expr	*read_sql_expression(int until,
											 const char *expected);
extern char *spl_read_identifier(void);

#endif   /* SPL_H */
