/*-------------------------------------------------------------------------
 *
 * bulkload.h
 *	  Support routines for the edb_bulkload bulk loader
 *
 *
 * Portions Copyright (c) 1996-2010, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 * Copyright (c) 2010-2013, EnterpriseDB Corporation 
 *
 *-------------------------------------------------------------------------
 */
#ifndef BULKLOAD_H
#define BULKLOAD_H

#include "access/xlog.h"
#include "access/xlogdefs.h"
#include "lib/stringinfo.h"
#include "miscadmin.h"
#include "storage/block.h"
#include "storage/relfilenode.h"
#include "utils/relcache.h"

typedef struct BulkLoadRange
{
	TransactionId	xid;		/* bulk loading transaction */
	RelFileNode		rd_node;
	BlockNumber		start;
	BlockNumber		end;
} BulkLoadRange;

/*
 * Blocks allocated in each iteration. RELSEG_SIZE must be divisible with
 * BULKLOAD_EXTENT_SIZE.
 */
#define BULKLOAD_EXTENT_SIZE	4096

/*
 * We fsync the old segment when we switch to new one, so the max number of
 * chunks a backend needs at any point in time is
 * RELSEG_SIZE / BULKLOAD_EXTENT_SIZE.
 */
#define MAX_BULKLOAD_RANGES_PER_BACKEND	(RELSEG_SIZE / BULKLOAD_EXTENT_SIZE)
#define MAX_BULKLOAD_RANGES				(MAX_BULKLOAD_RANGES_PER_BACKEND * MaxBackends)

typedef struct BulkLoadControlData
{
	int				range_count;
	BulkLoadRange  	ranges[1];		/* MAX_BULKLOAD_RANGES entries */
} BulkLoadControlData;

typedef struct xl_bulkload_start
{
	TransactionId	xid;
	RelFileNode		rd_node;
	BlockNumber		start;
	BlockNumber		end;
} xl_bulkload_start;

#define SizeOfBulkloadStart		(offsetof(xl_bulkload_start, end) + sizeof (BlockNumber))

typedef struct xl_bulkload_stop
{
	TransactionId	xid;
	RelFileNode		rd_node;
} xl_bulkload_stop;

#define SizeOfBulkloadStop		(offsetof(xl_bulkload_stop, rd_node) + sizeof (RelFileNode))

#define XLOG_BULKLOAD_START		0x00
#define XLOG_BULKLOAD_STOP		0x10

extern Size BulkLoadShmemSize(void);
extern void BulkLoadShmemInit(void);
extern void BulkLoadAddRange(TransactionId xid, RelFileNode rd_node,
				 BlockNumber start, BlockNumber end);
extern void BulkLoadRemoveRanges(TransactionId xid, RelFileNode rd_node);
extern void CheckPointBulkLoad(void);

extern void bulkload_redo(XLogRecPtr lsn, XLogRecord *record);
extern void bulkload_desc(StringInfo buf, uint8 xl_info, char *rec);
extern void bulkload_xlog_cleanup(void);

#endif
