/*-------------------------------------------------------------------------
 * icache.h
 *	  memcached related code
 *
 * Copyright (c) 2008-2013, EnterpriseDB Corporation
 *
 * src/include/storage/icache.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef ICACHED_H 
#define ICACHED_H

#include "miscadmin.h"
#include "nodes/pg_list.h"
#include "storage/buf_internals.h"
#include "utils/builtins.h"
#include "utils/guc.h"

#ifdef ICACHE_STATS
#include <sys/time.h>
#include <time.h>
#endif

/* GUC variables */
extern PGDLLIMPORT bool edb_enable_icache;
extern PGDLLIMPORT char *edb_icache_servers;
extern PGDLLIMPORT int edb_icache_compression_level;

#define ICACHE_MAX_SERVERS			128
#define ICACHE_MAX_BUCKETS			(ICACHE_MAX_SERVERS * 8)
#define ICACHE_INVALID_SERVER		(-1)

#define ICACHE_FAILURE_THRESHOLD (NBuffers / 4)

typedef enum edb_icache_server_state_t
{
	ICSS_UNUSED,
	ICSS_OFFLINE,
	ICSS_MANUAL_OFFLINE,
	ICSS_UNHEALTHY,
	ICSS_ACTIVE,
} edb_icache_server_state_t;

/*
 * Information about one icache server.
 */
typedef struct edb_icache_server_t
{
	int			servernum;		/* index of this entry in icache_info->servers, for convenience */
	char		hostname[NI_MAXHOST];	/* server host name */
	uint32		port;					/* server port number */
	int			write_failures;
	unsigned int flush_counter;
	unsigned int fail_counter;
	edb_icache_server_state_t state;
} edb_icache_server_t;

/*
 * icache hash bucket
 */
typedef struct edb_icache_bucket_t
{
	int			servernum;		/* server to which this bucket is mapped to */
	int			reload_counter;	/* current reload counter for this server */
} edb_icache_bucket_t;

typedef struct edb_icache_reload_info_t
{
	int			servernum;
	int			target_buckets;
	int			current_buckets; /* number of buckets assigned to this server */
	bool		unused_slot;
} edb_icache_reload_info_t;

/*
 * InfiniteCache's main shared memory structure.
 */
typedef struct edb_icache_info_t
{
	int		total_write_failures;	/* total number of failed icache servers */
	uint64	system_id;				/* server system identifier */
	uint64	startup_id;				/* startup identification of this module */
	bool	init_done;				/* first-time initialization done yet? */
	bool	is_cacheempty;			/* are there no servers (after init and post reload) */
	int		reload_count;			/* number of reloads done */
	edb_icache_server_t servers[ICACHE_MAX_SERVERS]; /* status of each server */
	edb_icache_bucket_t buckets[ICACHE_MAX_BUCKETS]; /* hash buckets */
} edb_icache_info_t;

/* Structure to track parsed hostname/portnum combo from the config param */
typedef struct edb_icache_parse_server_t
{
	char 	*host;
	uint32	 port;
} edb_icache_parse_server_t;

extern PGDLLIMPORT edb_icache_info_t *icache_info;

/* Some macros to use with edb_enable_icache */

/*
 * Identify if Memcached based calls are allowed
 */
#define BufferUseInfCache() \
	(edb_enable_icache && IsUnderPostmaster)

extern void InfCacheFlushBuffer(volatile BufferDesc *buf);
extern bool InfCacheReadBuffer(volatile BufferDesc *buf);
extern bool InfCacheRefreshBufferFailedFlag(volatile BufferDesc *buf);
extern bool InfCacheShowInfo(int *NServers, float *icache_size);

/* 
 * memcached supports string type keys with a max of 250 characters 
 * +1 for the trailing '\0' char. So all keys upto 251 characters
 */
#define ICACHE_MAXKEYSZ 251
#define GET_BUFFER_KEY(info, tag, cntr, reload_cntr, key) \
		snprintf((key), ICACHE_MAXKEYSZ, UINT64_FORMAT "/" UINT64_FORMAT "/%u/%u/%u/%u/%u/%u/%u", 	\
			 (info)->system_id, (info)->startup_id, (tag).rnode.spcNode, (tag).rnode.dbNode,        \
				 (tag).rnode.relNode, (tag).forkNum, (tag).blockNum, (cntr), (reload_cntr))

extern Size InfCacheShmemSize(void);
extern void InfCacheShmemInit(void);
extern void BackendInfCacheShmemInit(void);
extern void RetryInfCacheServers(void);

extern bool check_edb_enable_icache(bool *newval, void **extra, GucSource source);
extern bool check_edb_icache_servers(char **newval, void **extra,
									 GucSource source);
extern void assign_edb_icache_servers(const char *newval, void *extra);
extern struct memcached_st *get_icache_server(int servernum);


/* in bufmgr.c */
extern void InfCacheWarmBuffer(Buffer buffer);

#ifdef ICACHE_STATS
extern struct timeval icache_pre, icache_post;
extern uint32 n_smgr_read, n_smgr_write, n_icache_get, n_icache_set;
extern double diff_smgr_read, diff_smgr_write, diff_icache_get, diff_icache_set;

#define ICACHE_PRE_OP_TIME() gettimeofday(&icache_pre, NULL)
#define ICACHE_PRE_OP_STATS()	ICACHE_PRE_OP_TIME()
#define ICACHE_POST_OP_TIME() gettimeofday(&icache_post, NULL)
#define ICACHE_OP_TIME_DIFF(diff_time) 											 \
do {																			 \
	(diff_time) += (double) (icache_post.tv_sec - icache_pre.tv_sec) * 1000000.0 \
		+ (double) (icache_post.tv_usec - icache_pre.tv_usec);					 \
} while (0)
#define ICACHE_OP_COUNTER(counter)	(counter)++
#define ICACHE_POST_OP_STATS(diff_time, counter)								 \
do {																			 \
	ICACHE_POST_OP_TIME();														 \
	ICACHE_OP_TIME_DIFF(diff_time);												 \
	ICACHE_OP_COUNTER(counter);													 \
} while (0)
/*
 * have to use printfs here, because elog is not available at this time of exit
 */
#define ICACHE_PRINT_STATS()													 \
do {																			 \
		printf("InfiniteCache Stats\n\tIcache get calls:  (%d)\t\tset calls: "   \
			   "  (%d)\n\tSmgr read calls: (%d)\t\twrite calls (%d)\n",			 \
			   n_icache_get, n_icache_set, n_smgr_read, n_smgr_write);			 \
		printf("Avg Icache get response time: %f\t\tset response time:   %f\n"	 \
			   "Avg Smgr read response time:  %f\t\twrite response time: %f\n",	 \
			   diff_icache_get/n_icache_get, diff_icache_set/n_icache_set,		 \
			   diff_smgr_read/n_smgr_read, diff_smgr_write/n_smgr_write);		 \
} while (0)

#else
#define ICACHE_PRE_OP_TIME()
#define ICACHE_PRE_OP_STATS()
#define ICACHE_POST_OP_TIME()
#define ICACHE_OP_TIME_DIFF(diff_time)
#define ICACHE_OP_COUNTER(counter)
#define ICACHE_POST_OP_STATS(diff_time, counter)
#define ICACHE_PRINT_STATS()
#endif /* ICACHE_STATS */

#endif /* ICACHED_H */
