/*-------------------------------------------------------------------------
 * seqtree.h
 *	sequential AA tree definitions 
 *
 * Copyright (c) 2009-2013, EnterpriseDB Corporation
 *
 * src/include/utils/seqtree.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef SEQTREE_H
#define SEQTREE_H

#include "fmgr.h"
#include "utils/sortsupport.h"

typedef struct SEQTREENODE SEQTREENODE;
typedef struct SEQTREE SEQTREE;

/* relabel SEQTREE as VarrayType to diffentiate its use for VARRAYs */
typedef SEQTREE VarrayType;

/*
 * This should be considered private, but needs to be here so that
 * callers know how much space to allocate
 */
typedef struct SEQTREE_STATUS
{
	SEQTREE	   		*treep;
	SEQTREENODE	    *curEntry;	/* current entry in the tree */
} SEQTREE_STATUS;

/*
 * This struct holds some supporting information about the key and value
 * types that's needed by most of the operatins. It is initialized with
 * seqtree_init_cache(), and should be treated as an opaque block by
 * code outside seqtree.c. It is only declared here instead of seqtree.c so
 * that callers know to allocate the right amount of space.
 */
typedef struct
{
	bool		val_byVal;
	int16		val_len;
	bool		key_byVal;
	int16		key_len;
	SortSupportData sortsupport;
} SEQTREE_CACHE;

extern SEQTREE *seqtree_init(long nelem, Oid keytypid, int32 keytypmod,
			 Oid valtypid, int32 valtypmod);
extern Datum seqtree_search(SEQTREE *treep, Datum key, bool *found, bool *isnull, SEQTREE_CACHE *cache);
extern void seqtree_insert(SEQTREE **treeptr, Datum key, Datum value, bool isnull, SEQTREE_CACHE *cache);

extern void seqtree_init_cache(SEQTREE_CACHE *cache, SEQTREE *tree);

extern void seqtree_seq_init(SEQTREE_STATUS *status, SEQTREE *treep);
extern bool seqtree_seq_search(SEQTREE_STATUS *status,
			       Datum *key, Datum *value, bool *isnull,
			       SEQTREE_CACHE *cache);
extern bool seqtree_delete(SEQTREE_CACHE *cache, SEQTREE *treep, Datum key);
extern void seqtree_trim(SEQTREE *treep, int ndeletes);
extern Oid seqtree_get_keytype(SEQTREE *treep, int32 *typmod);
extern Oid seqtree_get_valuetype(SEQTREE *treep, int32 *typmod);
extern int seqtree_get_nentries(SEQTREE *treep);
extern int seqtree_get_size(SEQTREE *treep);
extern void seqtree_set_size(SEQTREE *treep, int size);
extern Datum seqtree_first_key(SEQTREE *treep, bool *found);
extern Datum seqtree_last_key(SEQTREE *treep, bool *found);
extern Datum seqtree_next_prior(SEQTREE *treep, Datum key, bool prior,
								bool *found);
extern SEQTREE *seqtree_union(SEQTREE *atree, SEQTREE *btree);
extern SEQTREE *seqtree_union_distinct(SEQTREE *atree, SEQTREE *btree);

extern Datum udtable_eq(PG_FUNCTION_ARGS);
extern Datum udtable_isbalanced(PG_FUNCTION_ARGS);
extern Datum udtable_dumptree(PG_FUNCTION_ARGS);

extern Datum nestedtable_in(PG_FUNCTION_ARGS);
extern Datum nestedtable_out(PG_FUNCTION_ARGS);
extern Datum udtable_unnest(PG_FUNCTION_ARGS);

#define DatumGetSeqtreeP(X) ((SEQTREE *) PG_DETOAST_DATUM(X))
#define SeqtreePGetDatum(X)	PointerGetDatum(X)
#define PG_RETURN_SEQTREE_P(x)	PG_RETURN_POINTER((x))

/*
 * fmgr macros for VARRAY object, wrappers around SEQTREE once, to distinguish
 * the usage for VARRAY
 */
#define DatumGetVarrayTypeP	DatumGetSeqtreeP
#define PG_GETARG_VARRAYTYPE_P(X) ((VarrayType *)PG_GETARG_VARLENA_P(X))
#define PG_RETURN_VARRAYTYPE_P PG_RETURN_SEQTREE_P
#define VarrayTypePGetDatum	SeqtreePGetDatum

#endif   /* SEQTREE_H */
