/*
 * Copyright (c) 2004-2013, EnterpriseDB Corporation. All Rights Reserved.
 */

#ifndef WAIT_H
#define WAIT_H

#include "postgres.h"
#include "storage/lwlock.h"

/*
 * OVERVIEW
 *
 *  The following describes the Managed Array-based
 *  Reporting, Grading, and Aggregating Runtime
 *  Instrumentation and Tracing Architecture.
 *
 *  If you need to 
 *
 *  waited_function ()
 *  {
 *      DECLARE_WAIT_VARIABLES;
 *      WAIT_BEGIN_TRACE;
 *
 *          ...
 *
 *      WAIT_END_TRACE;
 *  }
 */
#define DECLARE_WAIT_VARIABLES                          \
    double e0 = 0.0, e1 = 0.0;

#define WAIT_BEGIN_TRACE                                \
    if (timed_statistics)                               \
    {                                                   \
        e0 = gettime();                                 \
    }

#define WAIT_END_TRACE(num,p1,p2,p3)                    \
    if (timed_statistics)                               \
    {                                                   \
        e1 = gettime();                                 \
        PostWaitEvent(num,                              \
            (uint64) ((e1-e0-gettime_overhead)*1E6),    \
            p1, p2, p3);                                \
    }                                                   \
    else                                                \
    {                                                   \
        PostWaitEvent(num, (uint64) 0, p1, p2, p3);     \
    }

extern double gettime(void);

extern bool   timed_statistics;
extern double gettime_overhead;

PGDLLIMPORT extern char *wait_event_text[];

typedef enum WAIT_EVENT_ENUM
{
#undef WAIT_EVENT
#define WAIT_EVENT(i, s) i,
#include "utils/wait_events.h"
} WAIT_EVENT_ENUM;

#define MAX_SESSION_WAIT_HISTORY    25

/*
 * This structure stores an actual wait event and it's parameters.
 */
typedef struct WaitEventDetail
{
    uint16          wait_id;
    uint64          elapsed;
    uint64          p1;
    uint64          p2;
    uint64          p3;
} WaitEventDetail;

/*
 * This structure is used to store statistics on each wait.
 */
typedef struct WaitEventStats
{
    LWLockId        myLock;
    uint64          max_wait;
    uint64          total_wait;
    uint64          num_waits;
} WaitEventStats;

/*
 * This structure keeps track of MAX_SESSION_WAIT_HISTORY number of
 * actual wait events including their parameters.  That way, if a session
 * becomes unresponsive or is noticably slow, one can query the last
 * MAX_SESSION_WAIT_HISTORY of wait events to identify possible
 * problem areas like contention on certain tables, indexes, or locks.
 */
typedef struct WaitEventHistory
{
    uint32 num_elem_in_hist;
    WaitEventDetail queue[MAX_SESSION_WAIT_HISTORY];
    int queue_rear;
} WaitEventHistory;

/*
 * This is the overall structure used to encapsulate wait event information
 * for the pgproc array.
 */
typedef struct ProcWaitEvents
{
    WaitEventStats      session_wait_stats[HIGHEST_WAIT_EVENT_NUM];
    WaitEventHistory    session_wait_history;
} ProcWaitEvents;

extern ProcWaitEvents *procWaits;	/* Space for tracking wait states */

extern void PostWaitEvent (WAIT_EVENT_ENUM wait_event_id, uint64 elapsed, uint64 p1, uint64 p2, uint64 p3);

extern void InitProcWaitArray(void);

#endif
