#!/bin/bash
# Copyright (c) 2012-2014, EnterpriseDB Corporation.  All rights reserved

# PostgreSQL cluster init script for Linux
# Check the command line
if [ $# -ne 11 ]; 
then
    echo "Usage: $0 <OSUsername> <SuperUsername> <TempDir> <Install dir> <Data dir> <xlog dir> <Port> <Locale> <Mode> <server_utilization> <workload_profile>"
    exit 127
fi

OSUSERNAME="$1"
SUPERNAME="$2"
SUPERPASSWORD="$PGPASSWORD"
TEMP_DIR="$3"
INSTALLDIR="$4"
DATADIR="$5"
XLOGDIR="$6"
PORT="$7"
LOCALE="$8"
DBMODE="$9"
shift
SERVER_UTILIZATION="$9"
shift
WORKLOAD_PROFILE="$9"

# Exit code
WARN=0

# Error handlers
_die() {
    if [ -f $TEMP_DIR/initdbpw.$$ ];
    then
        rm -rf $TEMP_DIR/initdbpw.$$
    fi
    echo $1
    exit 1
}

_warn() {
    echo $1
    WARN=2
}

validateDirPermissions ()
{
	USER1="$1"
	INST_DIR1="$2"
	CMD1="$3"
	CMD_RET1=`su - "${USER1}" -c "${INST_DIR1}/${CMD1}" 2>&1`
	if echo $CMD_RET1 | egrep "Permission denied" &> /dev/null ; then
                echo "User ${USER1} does not have required permissions on directory ${INST_DIR1}"
		return 2
	elif echo $CMD_RET1 | grep "su: No directory" &> /dev/null ; then
		echo "User ${USER1} failed to login : Check user home directory permissions"
		return 3
        fi
	return 0
}

validateDirPermissions "${OSUSERNAME}" "${INSTALLDIR}" "bin/initdb -V" || _die ""

# Create a password file
touch $TEMP_DIR/initdbpw.$$ || _die "Failed to create the initdb password file ($TEMP_DIR/initdbpw.$$)"
chmod 0600 $TEMP_DIR/initdbpw.$$ || _die "Failed to set the permissions on the initdb password file ($TEMP_DIR/initdbpw.$$)"
echo "$SUPERPASSWORD" > $TEMP_DIR/initdbpw.$$ || _die "Failed to write the initdb password file ($TEMP_DIR/initdbpw.$$)"
chown $OSUSERNAME:$OSUSERNAME $TEMP_DIR/initdbpw.$$ || _die "Failed to set the ownership of the initdb password file ($TEMP_DIR/initdbpw.$$)"

# Create the data directory, and set the appropriate permissions/owership
if [ ! -d "$DATADIR" ];
then
    mkdir -p "$DATADIR" || _die "Failed to create the data directory ($DATADIR)"
fi
chown $OSUSERNAME:$OSUSERNAME "$DATADIR" || _die "Failed to set the ownership of the data directory ($DATADIR)"

#Set custom pg_xlog directory
#Only continue if we have non-default pg_xlog location
xlog_default="$DATADIR/pg_xlog"
xlog_new="$XLOGDIR"
XLOGOPTION=" "
if [ "$xlog_default" != "$xlog_new" ];
then
  if [ ! -d "$XLOGDIR/pg_xlog" ];
  then
    mkdir -p "$XLOGDIR/pg_xlog" || _die "Failed to create the xlog directory ($XLOGDIR/pg_xlog)"
  fi
  chown -R $OSUSERNAME:$OSUSERNAME "$XLOGDIR/pg_xlog" || _die "Failed to set the ownership of the xlog directory ($XLOGDIR)"
 XLOGOPTION="-X \"$XLOGDIR/pg_xlog\""
fi

# Determine --no-redwood-compat 
if [ $DBMODE = "postgresql" ];
then
    DBMODE="--no-redwood-compat"
else
    DBMODE=" "
fi

# Initialise the database cluster
if [ $LOCALE = "DEFAULT" ];
then
    su - $OSUSERNAME -c "LD_LIBRARY_PATH=$INSTALLDIR/lib $INSTALLDIR/bin/initdb $DBMODE --pwfile $TEMP_DIR/initdbpw.$$ -A md5 -U \"$SUPERNAME\" -D \"$DATADIR\" $XLOGOPTION" || _die "Failed to initialise the database cluster with initdb"
else
    su - $OSUSERNAME -c "LD_LIBRARY_PATH=$INSTALLDIR/lib $INSTALLDIR/bin/initdb $DBMODE --pwfile $TEMP_DIR/initdbpw.$$ --locale=$LOCALE -A md5 -U \"$SUPERNAME\" -D \"$DATADIR\" $XLOGOPTION" || _die "Failed to initialise the database cluster with initdb"
fi

if [ ! -d "$DATADIR/pg_log" ];
then
    mkdir "$DATADIR/pg_log" || _die "Failed to create the log directory ($DATADIR/pg_log)"
fi
chown $OSUSERNAME:$OSUSERNAME "$DATADIR/pg_log" || _die "Failed to set the ownership of the log directory ($DATADIR/pg_log)"
rm $TEMP_DIR/initdbpw.$$ || _warn "Failed to remove the initdb password file ($TEMP_DIR/initdbpw.$$)"

# Edit the config files.
# Set the following in postgresql.conf:
#      listen_addresses = '*'
#      port = $PORT
#      log_destination = 'stderr'
#      logging_collector = on
#      log_line_prefix='%t'
#      edb_dynatune_profile=$WORKLOAD_PROFILE
su - $OSUSERNAME -c "sed -e \"s@\#listen_addresses = 'localhost'@listen_addresses = '*'@g\" \
                        -e \"s@\#port = 5432@port = $PORT@g\" \
                        -e \"s@\#port = 5444@port = $PORT@g\" \
                        -e \"s@\#log_destination = 'stderr'@log_destination = 'stderr'@g\" \
                        -e \"s@\#logging_collector = off@logging_collector = on@g\" \
                        -e \"s@\#log_line_prefix = ''@log_line_prefix = '%t '@g\" \
                        -e \"s@edb_dynatune = 66@edb_dynatune = $SERVER_UTILIZATION@g\" \
			-e \"s@\#edb_dynatune_profile = oltp@edb_dynatune_profile = $WORKLOAD_PROFILE@g\" \
                        $DATADIR/postgresql.conf > $DATADIR/postgresql.conf.$$" || _warn "Failed to modify the postgresql.conf file ($DATADIR/postgresql.conf)"
su - $OSUSERNAME -c "mv $DATADIR/postgresql.conf.$$ $DATADIR/postgresql.conf" || _warn "Failed to update the postgresql.conf file ($DATADIR/postgresql.conf)"
chmod 0600 $DATADIR/postgresql.conf || _warn "Failed to set the permissions on postgresql.conf file ($DATADIR/postgresql.conf)"

echo "$0 ran to completion"
exit $WARN
