# src/nls-global.mk

# Common rules for Native Language Support (NLS)
#
# If some subdirectory of the source tree wants to provide NLS, it
# needs to contain a file 'nls.mk' with the following make variable
# assignments:
#
# CATALOG_NAME          -- name of the message catalog (xxx.po); probably
#                          name of the program
# AVAIL_LANGUAGES       -- list of languages that are provided/supported
# GETTEXT_FILES         -- list of source files that contain message strings
# GETTEXT_TRIGGERS      -- (optional) list of functions that contain
#                          translatable strings
# GETTEXT_FLAGS         -- (optional) list of gettext --flag arguments to mark
#                          function arguments that contain C format strings
#                          (functions must be listed in TRIGGERS and FLAGS)
#
# That's all, the rest is done here, if --enable-nls was specified.
#
# The only user-visible targets here are 'init-po', to make an initial
# "blank" catalog from program sources, and 'update-po', which is to
# be called if the messages in the program source have changed, in
# order to merge the changes into the existing .po files.


# existence checked by Makefile.global; otherwise we won't get here
include $(srcdir)/nls.mk

# If user specified the languages he wants in --enable-nls=LANGUAGES,
# filter out the rest.  Else use all available ones.
ifdef WANTED_LANGUAGES
LANGUAGES = $(filter $(WANTED_LANGUAGES), $(AVAIL_LANGUAGES))
else
LANGUAGES = $(AVAIL_LANGUAGES)
endif

PO_FILES = $(addprefix po/, $(addsuffix .po, $(LANGUAGES)))
ALL_PO_FILES = $(addprefix po/, $(addsuffix .po, $(AVAIL_LANGUAGES)))
MO_FILES = $(addprefix po/, $(addsuffix .mo, $(LANGUAGES)))

ifdef XGETTEXT
XGETTEXT += -ctranslator --copyright-holder='PostgreSQL Global Development Group' --msgid-bugs-address=pgsql-bugs@postgresql.org --no-wrap --sort-by-file --package-name='$(CATALOG_NAME) (PostgreSQL)' --package-version='$(MAJORVERSION)'
endif

ifdef MSGMERGE
MSGMERGE += --no-wrap --previous --sort-by-file
endif

# _ is defined in c.h, so it's global
GETTEXT_TRIGGERS += _
GETTEXT_FLAGS    += _:1:pass-c-format


# common settings that apply to backend and all backend modules
BACKEND_COMMON_GETTEXT_TRIGGERS = \
    errmsg errmsg_plural:1,2 \
    errdetail errdetail_log errdetail_plural:1,2 \
    errhint \
    errcontext
BACKEND_COMMON_GETTEXT_FLAGS = \
    errmsg:1:c-format errmsg_plural:1:c-format errmsg_plural:2:c-format \
    errdetail:1:c-format errdetail_log:1:c-format errdetail_plural:1:c-format errdetail_plural:2:c-format \
    errhint:1:c-format \
    errcontext:1:c-format


all-po: $(MO_FILES)

# EDB:
# We combine the PostgreSQL .po file and EnterpriseDB-specific -edb.po
# file into one .mo file. In the PostgreSQL .po file, all references to
# "pgsql-bugs@postgresql.org" are first replaced with
# "support@enterprisedb.com". We used to do that with one search&replace
# throughout the source tree including .po files, but changing the PostgreSQL
# .po files cause trouble at merges. Upstream .po file updates almost always
# move translations around the .po file, and at least the source line numbers
# change, so if the .po file in our repository is not 100% identical to the
# upstream version, you get merge conflicts. So now, we keep the
# "pgsql-bugs@postgresql.org" strings in the .po files that are checked in
# the repository, and replace them on-the-fly when we build the .mo file.
#
# The PG and EDB .po files are passed to the edb_strip_po_header.pl macro,
# which will scrub off the header from the EDB one. It also checks that the
# encoding and Plurality settings match; otherwise the resulting combined .mo
# file will be nonsense. We then pass the PG file with the strings replaced,
# and the EDB file with the header stripped away to msgfmt, to produce the
# final .mo file. We could pass one of the files to msgfmt with a pipe, but
# with a temporary file you get more informative error messages from msgfmt
# if something goes wrong, as it can refer to the filename instead of just
# stdin.
%.mo: %.po %-edb.po
	perl -p -e 's/pgsql-bugs\@postgresql\.org/support\@enterprisedb\.com/g' $*.po > $*.po.tmp
	$(top_srcdir)/src/tools/edb_strip_po_header.pl $^ > $*-edb.po.tmp
	$(MSGFMT) -o $@ $*.po.tmp $*-edb.po.tmp
	rm $*.po.tmp $*-edb.po.tmp

# This is just to create an empty *-edb.po file for languages that don't
# have EDB translations.
%-edb.po:
	touch $@

ifeq ($(word 1,$(GETTEXT_FILES)),+)
po/$(CATALOG_NAME).pot: $(word 2, $(GETTEXT_FILES)) $(MAKEFILE_LIST)
ifdef XGETTEXT
	$(XGETTEXT) -D $(srcdir) -n $(addprefix -k, $(GETTEXT_TRIGGERS)) $(addprefix --flag=, $(GETTEXT_FLAGS)) -f $<
else
	@echo "You don't have 'xgettext'."; exit 1
endif
else # GETTEXT_FILES
po/$(CATALOG_NAME).pot: $(GETTEXT_FILES) $(MAKEFILE_LIST)
# Change to srcdir explicitly, don't rely on $^.  That way we get
# consistent #: file references in the po files.
ifdef XGETTEXT
	$(XGETTEXT) -D $(srcdir) -n $(addprefix -k, $(GETTEXT_TRIGGERS)) $(addprefix --flag=, $(GETTEXT_FLAGS)) $(GETTEXT_FILES)
else
	@echo "You don't have 'xgettext'."; exit 1
endif
endif # GETTEXT_FILES
	@$(MKDIR_P) $(dir $@)
	sed -e '1,18 { s/SOME DESCRIPTIVE TITLE./LANGUAGE message translation file for $(CATALOG_NAME)/;s/PACKAGE/PostgreSQL/g;s/VERSION/$(MAJORVERSION)/g;s/YEAR/'`date +%Y`'/g; }' messages.po >$@
	rm messages.po

# To create a template -edb.pot file, we need an unmodified .pot for vanilla
# PostgreSQL. This downloads it from babel.postgreqsl.org.
%-pg.pot:
	wget http://babel.postgresql.org/po-$(MAJORVERSION)-branch/$(CATALOG_NAME).pot -O po/$(CATALOG_NAME)-pg.pot

%-edb.pot: %-pg.pot %.pot
# Remove all messages from edb .pot file that are present in vanilla .pot file
	$(top_builddir)/src/tools/edb_scrub_po_file.pl po/$(CATALOG_NAME)-pg.pot po/$(CATALOG_NAME).pot > po/$(CATALOG_NAME)-edb.pot


# catalog name extensions must match behavior of PG_TEXTDOMAIN() in c.h
install-po: all-po installdirs-po
ifneq (,$(LANGUAGES))
	for lang in $(LANGUAGES); do \
	  $(INSTALL_DATA) po/$$lang.mo '$(DESTDIR)$(localedir)'/$$lang/LC_MESSAGES/$(CATALOG_NAME)$(SO_MAJOR_VERSION)-$(MAJORVERSION).mo || exit 1; \
	done
endif

installdirs-po:
	$(if $(LANGUAGES),$(MKDIR_P) $(foreach lang, $(LANGUAGES), '$(DESTDIR)$(localedir)'/$(lang)/LC_MESSAGES),:)

uninstall-po:
	$(if $(LANGUAGES),rm -f $(foreach lang, $(LANGUAGES), '$(DESTDIR)$(localedir)'/$(lang)/LC_MESSAGES/$(CATALOG_NAME)$(SO_MAJOR_VERSION)-$(MAJORVERSION).mo),:)


clean-po:
	$(if $(MO_FILES),rm -f $(MO_FILES))
	@$(if $(wildcard po/*.po.new),rm -f po/*.po.new)
	rm -f po/$(CATALOG_NAME).pot
	rm -f po/*.po.tmp


maintainer-check-po: $(ALL_PO_FILES)
	for file in $^; do \
	  $(MSGFMT) -c -v -o /dev/null $$file || exit 1; \
	done


init-po: po/$(CATALOG_NAME).pot po/$(CATALOG_NAME)-edb.pot

# For performance reasons, only calculate these when the user actually
# requested update-po or a specific file.
ifneq (,$(filter update-po %.po.new,$(MAKECMDGOALS)))
ALL_LANGUAGES := $(shell find $(top_srcdir) -name '*.po' -print | sed 's,^.*/\([^/]*\).po$$,\1,' | LC_ALL=C sort -u)
all_compendia := $(shell find $(top_srcdir) -name '*.po' -print | LC_ALL=C sort)
else
ALL_LANGUAGES = $(AVAIL_LANGUAGES)
all_compendia = FORCE
FORCE:
endif

ifdef WANTED_LANGUAGES
ALL_LANGUAGES := $(filter $(WANTED_LANGUAGES), $(ALL_LANGUAGES))
endif

update-po: $(ALL_LANGUAGES:%=po/%.po.new)

$(AVAIL_LANGUAGES:%=po/%.po.new): po/%.po.new: po/%.po po/$(CATALOG_NAME).pot $(all_compendia)
	$(MSGMERGE) --lang=$* $(word 1, $^) $(word 2,$^) -o $@ $(addprefix --compendium=,$(filter %/$*.po,$(wordlist 3,$(words $^),$^)))

update-edb-po: $(EDB_LANGUAGES:%=po/%-edb.po.new)

$(AVAIL_LANGUAGES:%=po/%-edb.po.new): po/%-edb.po.new: po/%-edb.po po/$(CATALOG_NAME)-edb.pot $(all_compendia)
	$(MSGMERGE) $(word 1, $^) $(word 2,$^) --force-po -o $@ $(addprefix --compendium=,$(filter %/$*-edb.po,$(wordlist 3,$(words $^),$^)))
	mv $@ $(word 1, $^) # Overwrite old .po file with the new one

# For languages not yet available, merge against oneself, to pick
# up translations from the compendia.  (Merging against /dev/null
# doesn't work so well; it inserts the headers from the first-named
# compendium.)
po/%.po.new: po/$(CATALOG_NAME).pot $(all_compendia)
	$(MSGMERGE) --lang=$* $(word 1,$^) $(word 1,$^) -o $@ $(addprefix --compendium=,$(filter %/$*.po,$(wordlist 2,$(words $^),$^)))


# EDB targets

copy-pots: po/$(CATALOG_NAME)-edb.pot po/$(CATALOG_NAME)-pg.pot
	cp po/$(CATALOG_NAME)-edb.pot $(top_builddir)/pots
	cp po/$(CATALOG_NAME)-pg.pot $(top_builddir)/pots


EDB_PO_FILES = $(addprefix po/, $(addsuffix .po, $(EDB_LANGUAGES)))

# Copy EDB-specific .po files to "pos" directory at the top level, with
# unique filenames that include the catalog and language names. These can
# then be zipped up and sent to the translator for translation.
copy-edb-po-files:
	for lang in $(EDB_LANGUAGES); do \
	  cp po/$$lang-edb.po $(top_builddir)/pos/$(CATALOG_NAME)-edb-$$lang.po; \
	done

# Same for community .po files, except that the .po file is fetched from
# babel.postgresql.org, instead of taking the version from the source tree.
copy-pg-po-files:
	for lang in $(EDB_LANGUAGES); do \
	  wget -q http://babel.postgresql.org/po-$(MAJORVERSION)-branch/$(CATALOG_NAME)-$$lang.po -O $(top_builddir)/pos/$(CATALOG_NAME)-pg-$$lang.po; \
	done

# Reverse of copy-edb-po-files. Copies files belonging to this catalog back
# from the "pos" directory, to the right place in the source tree.
copy-back-edb-po-files:
	for lang in $(EDB_LANGUAGES); do \
	  if [ -f $(top_builddir)/pos/$(CATALOG_NAME)-edb-$$lang.po ]; then \
	  cp $(top_builddir)/pos/$(CATALOG_NAME)-edb-$$lang.po po/$$lang-edb.po ; \
	  fi \
	done

all: all-po
install: install-po
installdirs: installdirs-po
uninstall: uninstall-po
clean distclean maintainer-clean: clean-po
maintainer-check: maintainer-check-po

.PHONY: all-po install-po installdirs-po uninstall-po clean-po \
        maintainer-check-po init-po update-po
