--------------------------------------------------------------------------------
-- accessHistory() 
--
--		This accessHistory() function returns a set of records representing 
--		the conection attempts (both successful and failed) found in the 
--		given server log file.
--
--		The caller supplies:
--			* the name of the directory that contains the log file (this name
--			  must correspond to the name of a UTL_FILE directory, created with
--			  the CREATE DIRECTORY command, not an OS directory name)
--			  
--			* the name of the log file within the directory
--			  
--			* the name of the role in which the caller is interested; this 
--			  function returns all connect/disconnect records for the given
--			  role (this parameter is optional; if no role is specified, or 
--			  if the role is specified as NULL, this function will return 
--			  the access history for the caller).
--
--		Please note that any user may read his own access history, but you must
--		be a database superuser to read another user's access history.
--
--		Limitations - 
--			  This function will only return the connection entries found in 
--			  the given log file; if the administrator has rotated the server 
--			  log, this function will not find entries in the old files; it is
--			  the administrator's responsibility to ensure that old log files
--			  are available if required.
--
--		We ensure that the following parameters are properly configured before
--		trying to read the log:
--				log_connections = on
--				log_line_prefix = '%t, %u:'
--
--		This function is required to conform to the Common Criteria, specifically
--
--			 FTA_TAH_EXP.1.1 Upon successful session establishment, the TSF 
--			 shall store and retrieve the date and time of the last successful
--			 session establishment to the user.  
--
--			 FTA_TAH_EXP.1.2 Upon successful session establishment, the TSF 
--			 shall store and retrieve the date and time of the last unsuccessful 
--			 attempt to session establishment and the number of unsuccessful 
--			 attempts since the last successful session establishment.  
--
--		Note: this requirement has been amended because it is impossible for 
--			  a server to force a client application to display the access 
--			  history; instead, the server must make access history accessible
--			  to a client and the client may choose to display that information, 
--			  but is not required to.  See: http://www.niap-ccevs.org/PD/0142.html			  

CREATE TYPE accessHistoryRow AS (roleName TEXT, connectTime TEXT, success BOOLEAN);

CREATE OR REPLACE FUNCTION accessHistory(directory TEXT, logFileName TEXT, targetRole TEXT := NULL) RETURN SETOF accessHistoryRow AS
  fHandle  UTL_FILE.FILE_TYPE;
  logLine  TEXT;
  caller   TEXT;
  result   accessHistoryRow;
  isSuper  BOOLEAN;
  userName TEXT;
  paramVal TEXT;
BEGIN

  userName := targetRole;

  SELECT pg_catalog.SESSION_USER INTO caller;

  IF (targetRole IS NULL) THEN
    userName := caller;
  ELSIF (targetRole != caller) THEN

    SELECT usesuper INTO isSuper FROM pg_catalog.pg_user WHERE usename = caller;

	IF (NOT isSuper) THEN
	  RAISE 'Must be a superuser to query access history for other users';
	END IF;
  END IF;

  -- Validate log_line_prefix

  SELECT setting INTO paramVal FROM pg_catalog.pg_settings WHERE name = 'log_line_prefix';

  IF (substring(paramVal, 1, 7) != '%t, %u:') THEN
      RAISE 'log_line_prefix is misconfigured - must be set to "%%t, %%u:"';
  END IF;

  -- Validate log_connections

  SELECT setting INTO paramVal FROM pg_catalog.pg_settings WHERE name = 'log_connections';

  IF( paramVal != 'on') THEN
      RAISE 'log_connections is misconfigured - must be set to "on"';
  END IF;

  -- Open the log file

  fHandle := UTL_FILE.fopen(directory, logFileName, 'r');

  -- Read all entries in the log file, searching for 
  -- connect entries for the specified role

  LOOP
     UTL_FILE.GET_LINE(fHandle, logLine);

     IF (logLine ~ '^.*, .*:LOG:[ ]*connection authorized:') THEN
       IF (logLine ~ ( 'user=' || userName)) THEN

	     result.roleName = userName;
		 result.connectTime = trim(trailing ',' FROM substring(logLine FROM '^.*,'));
		 result.success = TRUE;

		 RETURN NEXT result;
       END IF;
	 ELSIF (logLine ~ '^.*, .*:FATAL:[ ]*password authentication failed for user.*') THEN
       IF (logLine ~ ( 'user "' || userName || '"')) THEN

	     result.roleName = userName;
		 result.connectTime = trim(trailing ',' FROM substring(logLine FROM '^.*,'));
		 result.success = FALSE;

		 RETURN NEXT result;
       END IF;
     END IF;
  END LOOP;

  EXCEPTION 
    WHEN NO_DATA_FOUND THEN
	  UTL_FILE.FCLOSE(fHandle);
END;
