-- Adjust this setting to control where the objects get created.
SET search_path = sys;
-----------------------------------------------------------------------------
-- DBMS_CRYPTO
--
-- The DBMS_CRYPTO package provides the interface to encrypt and decrypt data.
-- It provides support for various encryption and hashing algorithms.
-- This module is built on the pgcrypto contrib module
-- The pgcrypto module provides the same kind of functionality with
-- different syntax and though it does not have all the cipher algorithms
-- supported by dbms_crypto package but it does offer widely used
-- functionality.

CREATE OR REPLACE PACKAGE DBMS_CRYPTO AS

  --hash functions
  HASH_MD4			CONSTANT INTEGER := 1;
  HASH_MD5 			CONSTANT INTEGER := 2;
  HASH_SH1 			CONSTANT INTEGER := 3;

  --mac functions
  HMAC_MD5			CONSTANT INTEGER := 1;
  HMAC_SH1 			CONSTANT INTEGER := 2;

  --block cipher algorithms
  ENCRYPT_DES		CONSTANT INTEGER := 1;
  ENCRYPT_3DES		CONSTANT INTEGER := 3;
  ENCRYPT_AES		CONSTANT INTEGER := 4;
  ENCRYPT_AES128	CONSTANT INTEGER := 6;

  --block cipher modifiers
  CHAIN_CBC			CONSTANT INTEGER := 256;
  CHAIN_ECB			CONSTANT INTEGER := 768;

  --block cipher padding modifiers
  PAD_PKCS5 	 	CONSTANT INTEGER := 4096;
  PAD_NONE  		CONSTANT INTEGER := 8192;

  --block cipher suits
  DES_CBC_PKCS5 	CONSTANT INTEGER := ENCRYPT_DES + CHAIN_CBC + PAD_PKCS5;
  DES3_CBC_PKCS5 	CONSTANT INTEGER := ENCRYPT_3DES + CHAIN_CBC + PAD_PKCS5;
  AES_CBC_PKCS5 	CONSTANT INTEGER := ENCRYPT_AES + CHAIN_CBC + PAD_PKCS5;

  --declare exceptions
  CipherSuiteInvalid	EXCEPTION; -- cipher suite invalid
  CipherSuiteNull		EXCEPTION; -- null cipher suite
--   DoubleEncryption		EXCEPTION; -- double encryption
--   KeyBadSize			EXCEPTION; -- key bad size
  KeyNull				EXCEPTION; -- key null

  PRAGMA EXCEPTION_INIT(CipherSuiteInvalid,	-28827);
  PRAGMA EXCEPTION_INIT(CipherSuiteNull,	-28829);
--   PRAGMA EXCEPTION_INIT(DoubleEncryption,	-28233);
--   PRAGMA EXCEPTION_INIT(KeyBadSize,			-28234);
  PRAGMA EXCEPTION_INIT(KeyNull,			-28239);

  ---------------------------------------------------------------------------
  -- These function and procedures encrypt data using specified cipher
  -- algorithm with user supplied key and an optional initialization vector.
  -- The function returns encrypted data as RAW. While procedures are used to
  -- process LOB data and return the encrypted data in given buffer.
  --   src: data that needs to be encrypted
  --   typ: block cipher suite, its a combination of cipher algorithm, modifiers
  --	    and padding modifiers
  --   key: key to be used for encryption
  --   iv: optional, initialization vector
  FUNCTION ENCRYPT(src IN RAW,
				   typ IN INTEGER,
				   key IN RAW,
				   iv  IN RAW DEFAULT NULL)
		   RETURN RAW;

  PROCEDURE ENCRYPT(dst INOUT BLOB,
					src IN BLOB,
					typ IN INTEGER,
					key IN RAW,
					iv  IN RAW DEFAULT NULL);

  PROCEDURE ENCRYPT(dst INOUT BLOB,
					src IN CLOB,
					typ IN INTEGER,
					key IN RAW,
					iv  IN RAW DEFAULT NULL);

  ---------------------------------------------------------------------------
  -- These functions and procedures decrypt data using the specified cipher
  -- algorithm with user supplied key and an optional initialization vector.
  -- The function returns decrypted data as RAW, while procedures are used to
  -- process LOB data and return the decrypted data in given buffer.
  --   src: data to be decrypted
  --   typ: block cipher suite, its a combination of cipher algorithm, modifiers
  --	    and padding modifiers
  --   key: key to be used for decryption
  --   iv:  optional, initialization vector
  FUNCTION DECRYPT(src IN RAW,
				   typ IN INTEGER,
				   key IN RAW,
				   iv  IN RAW DEFAULT NULL)
		   RETURN RAW;

  PROCEDURE DECRYPT(dst INOUT BLOB,
					src IN BLOB,
					typ IN INTEGER,
					key IN RAW,
					iv  IN RAW DEFAULT NULL);

  PROCEDURE DECRYPT(dst INOUT CLOB,
					src IN BLOB,
					typ IN INTEGER,
					key IN RAW,
					iv  IN RAW DEFAULT NULL);

  ---------------------------------------------------------------------------
  -- These functions return the hash value of the given data using the 
  -- specified hash algorithm.
  --   src: data to be hashed
  --   typ: hash algorithm to use.
  FUNCTION HASH(src IN RAW, typ IN INTEGER) RETURN RAW;

  -- BLOB as RAW are the same in PPAS
  -- FUNCTION HASH (src IN BLOB, typ IN INTEGER) RETURN RAW;
  FUNCTION HASH(src IN CLOB, typ IN INTEGER) RETURN RAW;

  ---------------------------------------------------------------------------
  -- These functions return the hashed MAC value of the given data using
  -- the specified hash algorithm and user specified key.
  --   src: data for which hashed MAC is to be calculated
  --   typ: MAC algorithm to be used.
  --   key: key to be used for calculating hashed MAC.
  FUNCTION MAC(src IN RAW, typ IN INTEGER, key IN RAW) RETURN RAW;

  -- BLOB as RAW are the same in PPAS
  -- FUNCTION MAC (src IN BLOB, typ IN INTEGER, key IN RAW) RETURN RAW;
  FUNCTION MAC(src IN CLOB, typ IN INTEGER, key IN RAW) RETURN RAW;

  ---------------------------------------------------------------------------
  -- returns a number of cryptographically random bytes.
  --   number_bytes: number of random bytes to be returned
  FUNCTION RANDOMBYTES(number_bytes IN INTEGER) RETURN RAW;

  ---------------------------------------------------------------------------
  -- returns a random number in the range of [0..2^128-1]
  FUNCTION RANDOMNUMBER RETURN NUMBER;

  ---------------------------------------------------------------------------
  -- returns a random integer.
  FUNCTION RANDOMINTEGER RETURN INTEGER;
END;
