CREATE OR REPLACE PACKAGE BODY SYS.DBMS_PROFILER AS

	-- Private variables

	runComment							TEXT;
	runComment1							TEXT;
	runID								INTEGER;
	startTime							TIMESTAMP;

	microSecondsPerSecond	  CONSTANT  INTEGER := 1000000;
	nanoSecondsPerMicroSecond CONSTANT  INTEGER := 1000;
	majorVersion              CONSTANT  INTEGER := 1;
	minorVersion			  CONSTANT  INTEGER := 0;

    ----------------------------------------------------------------------
    PROCEDURE _install IS
	  DECLARE
	    createTableCommand TEXT;
	  BEGIN
		CREATE OR REPLACE FUNCTION sys.pg_profiler_start(runID INTEGER) RETURNS INTEGER LANGUAGE C AS '$libdir/dbms_profiler';
		CREATE OR REPLACE FUNCTION sys.pg_profiler_stop(tableName TEXT) RETURNS INTEGER LANGUAGE C AS '$libdir/dbms_profiler';
		CREATE OR REPLACE FUNCTION sys.pg_profiler_flush(tableName TEXT) RETURNS INTEGER LANGUAGE C AS '$libdir/dbms_profiler';
		CREATE OR REPLACE FUNCTION sys.pg_profiler_create_table(tableName TEXT) RETURNS TEXT LANGUAGE C AS '$libdir/dbms_profiler';
		CREATE OR REPLACE FUNCTION sys.pg_profiler_pause() RETURNS INTEGER LANGUAGE C AS '$libdir/dbms_profiler';
		CREATE OR REPLACE FUNCTION sys.pg_profiler_resume() RETURNS INTEGER LANGUAGE C AS '$libdir/dbms_profiler';
		CREATE OR REPLACE FUNCTION sys.pg_profiler_major() RETURNS INTEGER LANGUAGE C AS '$libdir/dbms_profiler';
		CREATE OR REPLACE FUNCTION sys.pg_profiler_minor() RETURNS INTEGER LANGUAGE C AS '$libdir/dbms_profiler';

	    createTableCommand := sys.pg_profiler_create_table(sys.dbms_profiler.table_name);

	    EXECUTE IMMEDIATE createTableCommand;

		CREATE SEQUENCE sys.plsql_profiler_runid;

		CREATE TABLE sys.plsql_profiler_runs
		(
			runid				INTEGER		PRIMARY KEY,
			related_run	  		INTEGER,
			run_owner			TEXT,
			run_date			TIMESTAMP,
			run_comment			TEXT,
			run_total_time		BIGINT,
			run_system_info		TEXT,
			run_comment1		TEXT,
			spare1				TEXT
		);

		CREATE TABLE sys.plsql_profiler_units
        (
			runid				INTEGER,
			unit_number			OID,
			unit_type			TEXT,
			unit_owner			TEXT,
			unit_name			TEXT,
			unit_timestamp		TIMESTAMP,
			total_time			BIGINT,
			spare1				BIGINT,
			spare2				BIGINT
        );

        CREATE VIEW sys.plsql_profiler_data AS
          SELECT 
            runid, 
            func_oid            AS unit_number,
            line_number         AS line#,
            exec_count          AS total_occur,
            time_total          AS total_time,
            time_shortest       AS min_time,
            time_longest        AS max_time,
            NULL::NUMBER        AS spare1,
            NULL::NUMBER        AS spare2,
            NULL::NUMBER        AS spare3,
            NULL::NUMBER        AS spare4
          FROM 
            sys.plsql_profiler_rawdata;
      END;

    ----------------------------------------------------------------------
	PROCEDURE _reset IS
	  BEGIN
	    TRUNCATE plsql_profiler_rawdata, plsql_profiler_runs, plsql_profiler_units;
		PERFORM setval('sys.plsql_profiler_runid', 1, false);
      END;
	  
    ----------------------------------------------------------------------
	-- Function start_profiler()
	--
	--	Increments the runID sequence and starts the profiler at the next
	--  SPL or PL/pgSQL invocation. 
	--
	--  Sets run_number to the run ID for this profiling session.
	--
	--  Returns one of the result codes defined
	--  for this package (error_param, error_io, or success).  
    ----------------------------------------------------------------------
    FUNCTION start_profiler(run_comment IN TEXT := sysdate, run_comment1 IN TEXT := '', run_number OUT INTEGER) RETURN INTEGER IS
      BEGIN
	    sys.dbms_profiler.runID       := pg_catalog.nextval('sys.plsql_profiler_runid');
		sys.dbms_profiler.runComment  := run_comment;
		sys.dbms_profiler.runComment1 := run_comment1;
		sys.dbms_profiler.startTime   := pg_catalog.now();

		run_number := sys.dbms_profiler.runID;

		RETURN sys.pg_profiler_start(sys.dbms_profiler.runID);
	  END;

    ----------------------------------------------------------------------
	-- Procedure start_profiler()
	--
	--	Increments the runID sequence and starts the profiler at the next
	--  SPL or PL/pgSQL invocation. 
	--
	--  Sets run_number to the run ID for this profiling session.
	--
	--	Raises an exception if something goes wrong
    ----------------------------------------------------------------------
    PROCEDURE start_profiler(run_comment IN TEXT := sysdate, run_comment1 IN TEXT := '', run_number OUT INTEGER) IS
	    result INTEGER;
      BEGIN
		result := sys.dbms_profiler.start_profiler(run_comment, run_comment1, run_number);

	    IF (result = sys.dbms_profiler.error_param) THEN 
	      RAISE EXCEPTION 'A subprogram was called with an incorrect parameter';
	    ELSIF (result = sys.dbms_profiler.error_io) THEN
	      RAISE EXCEPTION 'Data flush operation failed. Check whether the profiler tables have been created, are accessible, and that there is adequate space';
	    END IF;
	  END;

    ----------------------------------------------------------------------
	-- Function start_profiler()
	--
	--	Increments the runID sequence and starts the profiler at the next
	--  SPL or PL/pgSQL invocation. 
	--
	--  Returns one of the result codes defined for this package 
	--  (error_param, error_io, or success).  
    ----------------------------------------------------------------------
    FUNCTION start_profiler(run_comment IN TEXT := sysdate, run_comment1 IN TEXT := '') RETURN INTEGER IS
	    run_number INTEGER;
      BEGIN

	    -- Note: we discard the run_number

	    RETURN sys.dbms_profiler.start_profiler(run_comment, run_comment1, run_number);
	  END;

    ----------------------------------------------------------------------
	-- Procedure start_profiler()
	--
	--	Increments the runID sequence and starts the profiler at the next
	--  SPL or PL/pgSQL invocation. 
	--
	--	Raises an exception if something goes wrong
    ----------------------------------------------------------------------
    PROCEDURE start_profiler(run_comment IN TEXT := sysdate, run_comment1 IN TEXT := '') IS
		result     INTEGER;
      BEGIN
		result := sys.dbms_profiler.start_profiler(run_comment, run_comment1);

	    IF (result = sys.dbms_profiler.error_param) THEN 
	      RAISE EXCEPTION 'A subprogram was called with an incorrect parameter';
	    ELSIF (result = sys.dbms_profiler.error_io) THEN
	      RAISE EXCEPTION 'Data flush operation failed. Check whether the profiler tables have been created, are accessible, and that there is adequate space';
	    END IF;
	  END;

    ----------------------------------------------------------------------
	-- Function start_profiler()
	--
	--	Terminates the current profiling session (or does nothing if 
	--  we are not profiling).  Populates the plsql_profiler_runs,
	--  plsql_profiler_units, and plsql_profiler_rawdata tables.
	--
	--  Returns one of the result codes defined
	--  for this package (error_param, error_io, or success).  
    ----------------------------------------------------------------------
	FUNCTION stop_profiler RETURN INTEGER IS
	    result    INTEGER;
	  BEGIN
	    IF (sys.dbms_profiler.runID IS NULL) THEN
		    RETURN sys.dbms_profiler.success;
        ELSE
		    DELETE sys.plsql_profiler_runs     WHERE runid = sys.dbms_profiler.runID;
			DELETE sys.plsql_profiler_units    WHERE runid = sys.dbms_profiler.runID;
			DELETE sys.plsql_profiler_rawdata  WHERE runid = sys.dbms_profiler.runID;

	    	result := sys.pg_profiler_stop(sys.dbms_profiler.table_name);

	    	sys.dbms_profiler._write_runs(runid);
	    	sys.dbms_profiler._write_units(runid);

			RETURN result;
		END IF;
      END; 	  

    ----------------------------------------------------------------------
	-- Function start_profiler()
	--
	--	Terminates the current profiling session (or does nothing if 
	--  we are not profiling).  Populates the plsql_profiler_runs,
	--  plsql_profiler_units, and plsql_profiler_rawdata tables.
	--
	--	Raises an exception if something goes wrong
    ----------------------------------------------------------------------
	PROCEDURE stop_profiler IS
	    result    INTEGER;
      BEGIN
	    result := sys.dbms_profiler.stop_profiler;

		IF (result = sys.dbms_profiler.error_param) THEN 
          RAISE EXCEPTION 'A subprogram was called with an incorrect parameter';
	    ELSIF (result = sys.dbms_profiler.error_io) THEN
	      RAISE EXCEPTION 'Data flush operation failed. Check whether the profiler tables have been created, are accessible, and that there is adequate space';
	    END IF;
	END;

    ----------------------------------------------------------------------
	-- Function pause_profiler()
	--
	--  Pauses the profiler (if active).  This function does not flush the
	--  collected stats and does not reset any stats.
	--
	--  Returns one of the result codes defined for this package 
	--  (error_param, error_io, or success).  
    ----------------------------------------------------------------------
	FUNCTION pause_profiler RETURN INTEGER IS
	  BEGIN
	    RETURN sys.pg_profiler_pause;
      END; 	  

    ----------------------------------------------------------------------
	-- Procedure pause_profiler()
	--
	--  Pauses the profiler (if active).  This function does not flush the
	--  collected stats and does not reset any stats.
	--
	--	Raises an exception if something goes wrong
    ----------------------------------------------------------------------
	PROCEDURE pause_profiler IS
	  result INTEGER;
    BEGIN
	  result := sys.dbms_profiler.pause_profiler;

	  -- NOTE: Oracle does not complain if you pause the profiler even
	  --       when the profiler is not running so the following code 
	  --       is disabled.

	  -- IF (result = sys.dbms_profiler.error_param) THEN 
      --   RAISE EXCEPTION 'A subprogram was called with an incorrect parameter';
	  -- ELSIF (result = sys.dbms_profiler.error_io) THEN
	  --   RAISE EXCEPTION 'Data flush operation failed. Check whether the profiler tables have been created, are accessible, and that there is adequate space';
	  -- END IF;

	END;

    ----------------------------------------------------------------------
	-- Function resume_profiler()
	--
	--  Resumes profiling after a call to pause_profiler; if profiling is 
	--  not active (that is, if the user has not previously invoked 
	--  start_profiler), this function has no effect.
	--
	--  Returns one of the result codes defined for this package 
	--  (error_param, error_io, or success).  
    ----------------------------------------------------------------------
	FUNCTION resume_profiler RETURN INTEGER IS
	  BEGIN
	    RETURN sys.pg_profiler_resume;
      END; 	  

    ----------------------------------------------------------------------
	-- Function resume_profiler()
	--
	--  Resumes profiling after a call to pause_profiler; if profiling is 
	--  not active (that is, if the user has not previously invoked 
	--  start_profiler), this function has no effect.
	--
	--	Raises an exception if something goes wrong
    ----------------------------------------------------------------------
	PROCEDURE resume_profiler IS
	  result INTEGER;
    BEGIN
	  result := sys.dbms_profiler.resume_profiler;

	  -- NOTE: Oracle does not complain if you resume the profiler even
	  --       when the profiler is not running so the following code 
	  --       is disabled.

	  -- IF (result = sys.dbms_profiler.error_param) THEN 
      --   RAISE EXCEPTION 'A subprogram was called with an incorrect parameter';
	  -- ELSIF (result = sys.dbms_profiler.error_io) THEN
	  --   RAISE EXCEPTION 'Data flush operation failed. Check whether the profiler tables have been created, are accessible, and that there is adequate space';
	  -- END IF;

	END;

    ----------------------------------------------------------------------
	-- Function flush_data()
	--
	--  Populates the plsql_profiler_runs, plsql_profiler_units, and 
	--  plsql_profiler_rawdata tables with the stats collected during 
	--	the current session.  Does *not* stop or pause the current 
	--  session.  If no profiling session is active, this function does
	--  nothing.
	--
	--  Returns one of the result codes defined for this package 
	--  (error_param, error_io, or success).  
    ----------------------------------------------------------------------
	FUNCTION flush_data RETURN INTEGER IS
        result INTEGER;
	  BEGIN
	    IF (sys.dbms_profiler.runID IS NULL) THEN
		   RETURN sys.dbms_profiler.success;
        ELSE
	       DELETE sys.plsql_profiler_runs     WHERE runid = sys.dbms_profiler.runID;
		   DELETE sys.plsql_profiler_units    WHERE runid = sys.dbms_profiler.runID;
		   DELETE sys.plsql_profiler_rawdata  WHERE runid = sys.dbms_profiler.runID;

	       result := sys.pg_profiler_flush(sys.dbms_profiler.table_name);

	       sys.dbms_profiler._write_runs(runid);
	       sys.dbms_profiler._write_units(runid);

		   RETURN result;
        END IF;
      END; 	  

    ----------------------------------------------------------------------
	-- Procedure flush_data()
	--
	--  Populates the plsql_profiler_runs, plsql_profiler_units, and 
	--  plsql_profiler_rawdata tables with the stats collected during 
	--	the current session.  Does *not* stop or pause the current 
	--  session.  If no profiling session is active, this function does
	--  nothing.
	--
	--	Raises an exception if something goes wrong
    ----------------------------------------------------------------------
	PROCEDURE flush_data IS
	  result INTEGER;
	BEGIN
	  result := sys.dbms_profiler.flush_data;

	  IF (result = sys.dbms_profiler.error_param) THEN 
        RAISE EXCEPTION 'A subprogram was called with an incorrect parameter';
	  ELSIF (result = sys.dbms_profiler.error_io) THEN
	    RAISE EXCEPTION 'Data flush operation failed. Check whether the profiler tables have been created, are accessible, and that there is adequate space';
	  END IF;
	END;

    ----------------------------------------------------------------------
	-- Procedure get_version()
	--
	--	Returns the major and minor version of this package.	
    ----------------------------------------------------------------------
	PROCEDURE get_version(major OUT INTEGER, minor OUT INTEGER) IS
	  BEGIN
	    major := majorVersion;
	    minor := minorVersion;
	  END;

    ----------------------------------------------------------------------
	-- Function internal_version_check()
	--
	--	Determines whether this package is compatible with the C code in
	--  the profiler plugins. 
	--
	--  Returns sys.dbms_profiler.success or sys.dbms_profiler.error_version.
    ----------------------------------------------------------------------
	FUNCTION internal_version_check RETURN INTEGER IS
	  pluginMajor INTEGER;
	  pluginMinor INTEGER;
    BEGIN
	  IF (sys.pg_profiler_major() != sys.dbms_profiler.majorVersion) THEN
	    RETURN sys.dbms_profiler.error_version;
      END IF;

	  IF (sys.pg_profiler_minor() != sys.dbms_profiler.minorVersion) THEN
	    RETURN sys.dbms_profiler.error_version;
      END IF;

	  RETURN sys.dbms_profiler.success;
    END;

    ----------------------------------------------------------------------
	-- Private functions/procedures follow
    ----------------------------------------------------------------------

    ----------------------------------------------------------------------
	-- Procedure _write_units()
	--
	--	Aggregates the run times/execution counts for each function/procedure
	--  executed within the given run.  The results are written to the 
	--	sys.plsql_profiler_units table.
    ----------------------------------------------------------------------
    PROCEDURE _write_units(p_run_id INTEGER) IS
	  u_type TEXT;
	  u_owner TEXT;
	  u_name TEXT;
    BEGIN
	  FOR unit IN (SELECT func_oid, sum(time_total) * nanoSecondsPerMicroSecond total FROM sys.plsql_profiler_rawdata WHERE runid = p_run_id GROUP BY func_oid)
      LOOP
        BEGIN
		  SELECT 
		    DECODE(protype, '0', 'FUNCTION', '1', 'PROCEDURE', '2', 'TRIGGER'), 
		    pg_catalog.pg_get_userbyid(proowner), 
	        proname || '(' || pg_catalog.pg_get_function_identity_arguments(oid) || ')' 
          INTO 
		    u_type, u_owner, u_name 
		  FROM 
		    pg_catalog.pg_proc 
          WHERE 
		    oid = unit.func_oid;
        EXCEPTION WHEN NO_DATA_FOUND THEN
          u_type  := '<unknown>';
		  u_owner := '<unknown>';
		  u_name  := '<deleted>';
		END;
		  
        INSERT INTO sys.plsql_profiler_units (runid, unit_number, unit_type, unit_owner, unit_name, unit_timestamp, total_time, spare1, spare2)
		  VALUES (p_run_id, unit.func_oid, u_type, u_owner, u_name, NULL, unit.total, NULL, NULL);
      END LOOP;
    END;

    ----------------------------------------------------------------------
	-- Procedure _write_runs()
	--
	--	Computes the total run time for the given run and writes the result
	--  into the sys.plsql_profiler_runs table.
    ----------------------------------------------------------------------
	PROCEDURE _write_runs(p_run_id INTEGER) IS
		totalTime DOUBLE PRECISION;
	BEGIN
	  SELECT pg_catalog.SUM(time_total) INTO totalTime FROM sys.plsql_profiler_rawdata WHERE runid = p_run_id;

	  INSERT INTO sys.plsql_profiler_runs(runid, run_owner, run_date, run_comment, run_comment1, run_total_time)
		VALUES(p_run_id, CURRENT_USER, sys.dbms_profiler.startTime, sys.dbms_profiler.runComment, sys.dbms_profiler.runComment1, totalTime * microSecondsPerSecond);
    END;

    ----------------------------------------------------------------------
	-- Package initializer
    ----------------------------------------------------------------------
	BEGIN
	  LOAD '$libdir/dbms_profiler';
    END;

END;

EXEC SYS.DBMS_PROFILER._INSTALL;
