
/*
 * NOTE: If you change anything in the SQL below, do remember to make the same
 * changes in index_advisor.c:save_advice():create_temp_log_table
 */

/* The table where Index Advisor would store its recommendations */
create table index_advisor_log(
				/* Table's Oid */
				reloid		oid,
				/* Table's name; may be a temporary table */
				relname		name,
				/* Array of column numbers of the table */
				attrs		integer[],
				/* What was the cost improvement after using this index */
				benefit		real,
				/* Estimated size of the index, if created */
				index_size	integer,
				/* Process ID of the backend that generated the recommendation */
				backend_pid	integer,
				/* When was this recommendation generated */
				timestamp	timestamptz default now());

/* Index to speed up lookup by table's Oid */
create index IA_reloid on index_advisor_log( reloid );

/* Index to speed up lookup by generating backend's pid */
create index IA_backend_pid on index_advisor_log( backend_pid );

/* Read table index_advisor_log and display the constructed SQL (CREATE INDEX) statements */
create or replace function show_index_recommendations(p_backend_pid index_advisor_log.backend_pid%type) returns setof text as $$
declare
	pid       p_backend_pid%type;
	q_advice  text;
	r_advice  record;
	q_column  text;
	r_column  record;
	ret       text;
	rowcount    int;

	NAMEDATALEN	int := (select setting::integer
						from pg_settings
						where name = 'max_identifier_length');

	collist_w_C		text;	/* Column name list with commas */
	collist_w_U		text;	/* Column name list with underscores */
	colidlist_w_U	text;	/* Column id list with underscores */
begin
	if p_backend_pid is null then
		pid = pg_backend_pid();
	else
		pid = p_backend_pid;
	end if;

	q_advice :=	'SELECT	c.relname,
						(select nspname from pg_namespace where oid = c.relnamespace) as relnspname,	
						c.oid as reloid,
						a.attrs AS colids,
						MAX( a.index_size ) AS size_in_KB,
						SUM( a.benefit ) AS benefit,
						SUM( a.benefit )/MAX( a.index_size ) AS gain
				FROM    index_advisor_log a,
						pg_class c
				WHERE   a.backend_pid = ' || pid || '
				AND     a.reloid = c.oid
				GROUP BY    c.relname, relnspname, c.oid, a.attrs
				ORDER BY    gain
					DESC';

	for r_advice in execute q_advice loop

		collist_w_C		:= '';
		collist_w_U     := '';
		colidlist_w_U	:= '';

		for i in array_lower( r_advice.colids, 1 )
					.. array_upper( r_advice.colids, 1 )
		loop
			q_column :=	'SELECT a.attname as name,
								a.attnum as id
						FROM    pg_class c,
								pg_attribute a
						WHERE   c.oid = ' || r_advice.reloid || '
						AND     a.attrelid = c.oid
						AND     a.attnum = ' || r_advice.colids[i] || '
						';

			execute q_column into r_column;
			get diagnostics rowcount = ROW_COUNT;

			if rowcount > 1 then
				raise EXCEPTION 'an internal query failed';
			end if;

			if i <> 1 then
				collist_w_C		:= collist_w_C		|| ', ';
				collist_w_U		:= collist_w_U		|| '_';
				colidlist_w_U	:= colidlist_w_U	|| '_';
			end if;

			/* List of column on which we want to create index */
			collist_w_C		:= collist_w_C		|| quote_ident(r_column.name);
			/* List of columns to generate the unique index name */
			collist_w_U		:= collist_w_U		|| r_column.name;
			colidlist_w_U	:= colidlist_w_U	|| r_column.id;

		end loop;

		ret := 'create index ';

		if (length('idx_' || r_advice.relname || '_' || collist_w_U)
				<= NAMEDATALEN)
		then
			ret := ret || quote_ident('idx_' || r_advice.relname	|| '_' || collist_w_U);
		else
			ret := ret || 'idx_' || r_advice.reloid		|| '_' || colidlist_w_U;
		end if;

		ret := ret || ' on ' || quote_ident(r_advice.relnspname) || '.' || quote_ident(r_advice.relname) || '(' || collist_w_C || ');';

		ret := ret ||
				'/* size: ' || r_advice.size_in_KB || ' KB, '
				|| 'benefit: ' || r_advice.benefit || ', '
				|| 'gain: ' || r_advice.gain || E' */';

		return next ret;
	end loop;

  return;
end;
$$ language plpgsql;

create or replace function show_index_recommendations() returns setof text as $$
	select * from show_index_recommendations(null);
$$ language sql;

create or replace view index_recommendations as
select	backend_pid ,
		show_index_recommendations( v.backend_pid )
from	(select	distinct backend_pid
		from	index_advisor_log as adv
		/*
		 * Do not consider tables that no longer exist.
		 *
		 * TODO: Consider emitting recommendations for TEMP tables.
		 */
		where	exists (select	oid
								from	pg_class
								where	oid = adv.reloid)
		) as v;

