-- This function copies data from catalog tables to history tables

-- 01/02/2008 P. Steinheuser - Added load of new tables
-- 01/10/2008 P. Steinheuser - Added edbreport and supporting functions
-- 03/27/2008 P. Steinheuser - Changed some functions to also work with PG+ AS
------------------------------------------------------------------------------

SET search_path = sys;
--------------------------------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION edbsnap()
  RETURNS text AS
$BODY$    
 
declare
v_seq   bigint;
v_db    name;
msg     text;
v_version text;
v_stats_cnt	numeric :=0;

begin
select nextval('snapshot_num_seq') into v_seq;
select current_database() into v_db;
select version() into v_version;
/*
select count(*) into v_stats_cnt from pg_settings
	where name in ('stats_block_level','stats_row_level')
	and setting = 'on';

if v_stats_cnt < 2 then
	msg := 'Stats_block_level and stats_row_level parameters should be set = on.';
	return msg;
	exit;
end if;
*/
	
insert into edb$snap
values (v_seq,
        v_db,
     now(),
     null      ,
     null,
     null,
     'N')
;

if v_version like 'EnterpriseDB%' then
	insert into edb$system_waits
		select v_seq,v_db,
		wait_name  ,
		wait_count  ,
		avg_wait  ,
		max_wait  ,
		total_wait
		from system_waits;
 
	insert into edb$session_waits
		select v_seq, v_db,
		backend_id   ,
		wait_name   ,
		wait_count     ,
		avg_wait_time    ,
		max_wait_time    ,
		total_wait_time  
		, usename, query
			from session_waits LEFT OUTER JOIN pg_stat_activity 
			on backend_id = pg_stat_activity.pid;

	insert into edb$session_wait_history
		select v_seq,
		v_db,
		backend_id   ,
		seq          ,
		wait_name    ,
		elapsed      ,
		p1          ,
		p2           ,
		p3           
		from session_wait_history;
end if;

------------------------------ NEW ------------------------------
insert into edb$stat_all_tables
select v_seq,v_db,
relid              ,                     
 schemaname         ,                 
 relname            ,                
 seq_scan            ,                  
 seq_tup_read        ,                  
 idx_scan            ,                  
 idx_tup_fetch       ,                  
 n_tup_ins           ,                  
 n_tup_upd          ,                 
 n_tup_del         -- ,                 
 -- last_vacuum         ,
 -- last_autovacuum     ,
 -- last_analyze        ,
 -- last_autoanalyze   
from pg_stat_all_tables;

insert into edb$stat_all_indexes
select v_seq,v_db,
relid           ,   
 indexrelid       ,   
 schemaname       ,
 relname          ,
 indexrelname     ,
 idx_scan         ,
 idx_tup_read     ,
 idx_tup_fetch   
from pg_stat_all_indexes;

insert into edb$statio_all_tables
select v_seq,v_db,
relid             ,  
 schemaname        ,
 relname            ,
 heap_blks_read     ,
 heap_blks_hit     , 
 heap_blks_icache_hit,
 idx_blks_read      ,
 idx_blks_hit       ,
 idx_blks_icache_hit,
 toast_blks_read    ,
 toast_blks_hit     ,
 toast_blks_icache_hit,
 tidx_blks_read     ,
 tidx_blks_hit      ,
 tidx_blks_icache_hit
from pg_statio_all_tables;

insert into edb$statio_all_indexes
select v_seq,v_db,
relid           ,  
 indexrelid        ,  
 schemaname      , 
 relname          ,
 indexrelname     ,
 idx_blks_read    ,
 idx_blks_hit     ,
 idx_blks_icache_hit
from pg_statio_all_indexes;

insert into edb$stat_database
select v_seq,v_db,
datid           ,    
 datname          ,
 numbackends      ,
 xact_commit      ,
 xact_rollback   , 
 blks_read         ,
 blks_hit          ,
 blks_icache_hit
from pg_stat_database;
------------------------------ END OF NEW -----------------------
 
msg := 'Statement processed.';
return msg;
/* 
exception
when others then
msg := 'Unsupported version.';
return msg;
*/ 
end;

  $BODY$
  LANGUAGE 'plpgsql' VOLATILE
  ;

------------------------------------------------------------------
CREATE OR REPLACE FUNCTION get_snaps()
RETURNS setof text AS $$    

declare
textstr text;
rec RECORD;
begin

for rec in
(select edb_id, snap_tm 
from edb$snap
order by edb_id  )  loop
textstr := rec.edb_id||' '||rec.snap_tm;
return next textstr;
end loop;


return;
end;

  $$ LANGUAGE 'plpgsql' ;

------------------------------------------------------------------

CREATE OR REPLACE FUNCTION purgesnap     (
    INT4, 
    INT4
    )
RETURNS text AS $$ 
declare
bid     alias for $1;
eid     alias for $2;
msg text;

rowcnt  int;

begin

delete from edb$system_waits where edb_id between bid and eid;
delete from edb$session_waits where edb_id between bid and eid; 
delete from edb$session_wait_history where edb_id between bid and eid; 
delete from edb$snap where edb_id between bid and eid; 
-- added 01/07/2008 - P. Steinheuser
delete from edb$stat_database where edb_id between bid and eid; 
delete from edb$stat_all_tables where edb_id between bid and eid; 
delete from edb$statio_all_tables where edb_id between bid and eid; 
delete from edb$stat_all_indexes where edb_id between bid and eid; 
delete from edb$statio_all_indexes where edb_id between bid and eid; 

get diagnostics rowcnt = ROW_COUNT;

if rowcnt = 0  then
  msg := 'Snapshots not found.';
else
    msg := 'Snapshots in range '||bid||' to '||eid||' deleted.';
end if;

-- msg := 'Rows deleted = '||rowcnt;
  return msg;

exception
when others then
msg := 'Function failed.';
return msg;
end;

   $$ LANGUAGE 'plpgsql' ;
-----------------------------------------------

CREATE OR REPLACE FUNCTION truncsnap()
RETURNS text AS $$ declare
msg text;

begin

truncate table edb$system_waits;
truncate table edb$session_waits;
truncate table edb$session_wait_history;
truncate table edb$snap;
-- added 01/07/2008 - P. Steinheuser
truncate table edb$stat_database;
truncate table edb$stat_all_tables;
truncate table edb$statio_all_tables;
truncate table edb$stat_all_indexes;
truncate table edb$statio_all_indexes;

msg := 'Snapshots truncated.';
return msg;

exception
when others then
msg := 'Function failed.';
return msg;
end;

   $$ LANGUAGE 'plpgsql' ;
---------------------------------------------------------------------------
-----------------------------------------------------------------------------------
-- This function produces a report of system waits
-- Input arguments are 1. begin snap id 2. end snap id and 3. top n rows to return
--
-- To run: select * from sys_rpt(14,15,10) as System Waits;
-----------------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION sys_rpt         (
    INT4, 
    INT4, 
    INT4
    )
RETURNS setof text AS $$      

declare
bid     alias for $1;
eid     alias for $2;
topn    alias for $3;

textstr text;
rec RECORD;
begin

textstr := rpad('WAIT NAME',40,' ')||' '||rpad('COUNT',10,' ')||' '||rpad('WAIT TIME',15,' ')||
' '||'% WAIT';
return next textstr;
textstr := '---------------------------------------------------------------------------';
return next textstr;

for rec in (
select e.wait_name,
        abs(e.wait_count - b.wait_count)  as wait_count,   -- sum of all session wait counts
        abs(e.totalwait - b.totalwait)  as totalwait,      -- sum of all session wait times
        round(100* (sumwait/twaits),2) as pctwait     -- divides total waits per event/total of all waits 
                 from  edb$system_waits as b
                    ,  edb$system_waits  as e
    , (select b.wait_name ,sum(abs(e.totalwait - b.totalwait)) as sumwait 
		from edb$system_waits b,
		edb$system_waits e
	where b.edb_id = bid
	and e.edb_id = eid
	and b.wait_name = e.wait_name 
        group by b.wait_name
        ) as sw,           			-- this gets total waits per event for snap period
        (select sum(abs(e.totalwait - b.totalwait)) as twaits 
     		from edb$system_waits b,
			edb$system_waits e
		where b.edb_id = bid
		and e.edb_id = eid
		and b.wait_name = e.wait_name) as tw  -- this gets sum of all waits per snap period
where b.edb_id              = bid
and e.edb_id              = eid
and b.dbname              = e.dbname
and b.wait_name           = e.wait_name
and e.wait_name          = sw.wait_name
and twaits               > 0              -- avoids divide by zero error
    order by pctwait desc, totalwait desc 
    limit topn)  LOOP                               

textstr := rpad(rec.wait_name,40,' ')||' '||rpad(rec.wait_count::text,10,' ')||' '||rpad(rec.totalwait::text,15,' ')||
' '||rec.pctwait;
return next textstr;
end loop;

return;
end;
    $$ LANGUAGE 'plpgsql' ;
-----------------------------------------------------------------------------------
-- This function produces a report of session waits
-- Input arguments are 1. begin snap id 2. end snap id and 3. top n rows to return
--
-----------------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION sess_rpt         (
    INT4, 
    INT4, 
    INT4
    )
RETURNS setof text AS $$       

declare
bid     alias for $1;
eid     alias for $2;
topn    alias for $3;

textstr text;
rec RECORD;
begin

textstr := 'ID   '||' '||rpad('USER',10,' ')||' '||rpad('WAIT NAME',30,' ')||' '||rpad('COUNT',5,' ')||
' '||rpad('TIME(ms)',12,' ')||' '||rpad('% WAIT SESS',10)||' '||'% WAIT ALL';

return next textstr;
textstr := '---------------------------------------------------------------------------------------------------';
return next textstr;

for rec in (select fv.backend_id,
        fv.usename,
        fv.wait_name,
        fv.waitcnt,
        fv.totwait,
        -- sw.sumwait, tw.twaits
        round(100* (fv.totwait/sw.sumwait),2) as pctwaitsess,
        round(100 *(fv.totwait/tw.twaits),2) as pctwaitall 
 from (
 select e.backend_id, 
          e.usename,
          e.wait_name,
          abs(e.wait_count - b.wait_count)  as waitcnt,   		 
          abs(e.total_wait_time - b.total_wait_time)  as totwait      
             from  edb$session_waits  as b
                    ,  edb$session_waits  as e
                  where b.edb_id              	= bid
                  and e.edb_id              	= eid
                  and b.dbname              	= e.dbname
                  and b.wait_name           	= e.wait_name
                  and b.backend_id 		= e.backend_id
                  and (b.total_wait_time > 0 or e.total_wait_time > 0)
              order by backend_id, totwait desc ) as fv , 
 	(select e2.backend_id,sum(abs(e2.total_wait_time-b2.total_wait_time)) as sumwait 
		from edb$session_waits b2,
                    edb$session_waits e2
		where b2.edb_id = bid 
                    and e2.edb_id = eid
                      and b2.backend_id = e2.backend_id
                      and b2.wait_name = e2.wait_name
            	group by e2.backend_id having sum(abs(e2.total_wait_time-b2.total_wait_time)) > 0 
                order by e2.backend_id 	) as sw  ,          
    (select sum(abs(e3.total_wait_time - b3.total_wait_time)) as twaits  
    		from edb$session_waits b3, edb$session_waits e3
 		where b3.edb_id = bid and e3.edb_id = eid
                and b3.backend_id = e3.backend_id
		and b3.wait_name = e3.wait_name) as tw
where fv.backend_id = sw.backend_id
        limit topn)  LOOP   
                                 
textstr := rec.backend_id||' '||rpad(rec.usename,10,' ')||' '||rpad(rec.wait_name,30,' ')||' '||
rpad(rec.waitcnt::text,5,' ')||' '||rpad(rec.totwait::text,12,' ')||' '||rpad(rec.pctwaitsess::text,10,' ')||' '||rec.pctwaitall;

return next textstr;
end loop;

return;
end;
 $$ LANGUAGE 'plpgsql' ;

-----------------------------------------------------------------------------------
-- This function produces a report of session waits for a specific backend
-- Input arguments are 1. begin snap id 2. end snap id and 3. backend_id to return
--
-----------------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION sessid_rpt         (
    INT4, 
    INT4, 
    INT4
    )
RETURNS setof text AS $$       

declare
bid     alias for $1;
eid     alias for $2;
sessid    alias for $3;

textstr text;
rec RECORD;
begin

textstr := 'ID   '||' '||rpad('USER',10,' ')||' '||rpad('WAIT NAME',30,' ')||' '||rpad('COUNT',5,' ')||
' '||rpad('TIME(ms)',12,' ')||' '||rpad('% WAIT SESS',10)||' '||'% WAIT ALL';

return next textstr;
textstr := '---------------------------------------------------------------------------------------------------';
return next textstr;

for rec in (select fv.backend_id,
        fv.usename,
        fv.wait_name,
        fv.waitcnt,
        fv.totwait,
        round(100* (fv.totwait/sw.sumwait),2) as pctwaitsess,
        round(100 *(fv.totwait/tw.twaits),2) as pctwaitall 
 from (
 select e.backend_id, 
          e.usename,
          e.wait_name,
          abs(e.wait_count - b.wait_count)  as waitcnt,   		 
          abs(e.total_wait_time - b.total_wait_time)  as totwait      
            from  edb$session_waits  as b
                    ,  edb$session_waits  as e
                  where b.edb_id              	= bid
                  and e.edb_id              	= eid
                  and b.dbname              	= e.dbname
                  and b.wait_name           	= e.wait_name
                  and b.backend_id 		= e.backend_id
		and b.backend_id	= sessid
                  and (b.total_wait_time > 0 or e.total_wait_time > 0)
              order by backend_id, totwait desc ) as fv , 
 	(select e2.backend_id,sum(abs(e2.total_wait_time-b2.total_wait_time)) as sumwait 
		from edb$session_waits b2,
                    edb$session_waits e2
		where b2.edb_id = bid 
                    and e2.edb_id = eid
		and e2.wait_name = b2.wait_name
                      and b2.backend_id = e2.backend_id
		and b2.backend_id = sessid
            	group by e2.backend_id having sum(abs(e2.total_wait_time-b2.total_wait_time)) > 0 
                order by e2.backend_id 	) as sw  ,          
    (select sum(abs(e3.total_wait_time - b3.total_wait_time)) as twaits  
    		from edb$session_waits b3, edb$session_waits e3
 		where b3.edb_id = bid and e3.edb_id = eid
		and b3.backend_id = e3.backend_id
		and b3.wait_name = e3.wait_name) as tw
where fv.backend_id = sw.backend_id
        -- limit topn
)  LOOP   
                                 
textstr := rec.backend_id||' '||rpad(rec.usename,10,' ')||' '||rpad(rec.wait_name,30,' ')||' '||
rpad(rec.waitcnt::text,5,' ')||' '||rpad(rec.totwait::text,12,' ')||' '||rpad(rec.pctwaitsess::text,10,' ')||' '||rec.pctwaitall;

return next textstr;
end loop;

return;
end;
 $$ LANGUAGE 'plpgsql' ;

-----------------------------------------------------------------------------------
-- This function produces a report of session history waits for a specific backend
-- Input arguments are 1. begin snap id 2.backend_id to return
-- The Sum of Blks column (p3) is number of blocks read. Currently, this will always 
-- be 1. Multi-block reads will be available in later releases.
-----------------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION sesshist_rpt         (
    INT4, 
    INT4
    )
RETURNS setof text AS $$         

declare
snapid     	alias for $1;
sessid  	alias for $2;

textstr text;
rec RECORD;
begin

textstr := 'ID   '||' '||rpad('USER',10,' ')||rpad('SEQ',5,' ')||' '||rpad('WAIT NAME',24,' ')||' '||
rpad('ELAPSED(ms)',12,' ')||' '||rpad('File',6,' ')||' '||rpad('Name',20,' ')||' '||rpad('# of Blk',10,' ')||' '||rpad('Sum of Blks',12,' ');

return next textstr;
textstr := '---------------------------------------------------------------------------------------------------';
return next textstr;

for rec in (
   select b.backend_id, 
          w.usename,
	  b.seq,
          b.wait_name,
          b.elapsed,   		 
          b.p1,     
          c.relname as relname,   
	  b.p2 as p2,     
	  sum(b.p3) as p3          
            from  edb$session_wait_history  as b,
                  edb$session_waits as w,
                  (select relfilenode,relname from pg_class
                  union
                  select 0,'N/A') as c  -- added this to handle file=0 for query plan
                  where b.edb_id              	= snapid
                  and b.backend_id              = sessid
		  and b.backend_id              = w.backend_id
		  and b.dbname                  = w.dbname
		  and b.wait_name               = w.wait_name
		  and b.edb_id                  = w.edb_id
                  and b.p1                      = c.relfilenode
             group by b.backend_id, 
          w.usename,
	  b.seq,
          b.wait_name,
          b.elapsed,
           b.p1 , c.relname, b.p2      
              order by backend_id, seq   
         )  LOOP   
                                 
textstr := rec.backend_id||' '||rpad(rec.usename,10,' ')||' '||rpad(rec.seq::text,5,' ')||' '||rpad(rec.wait_name,24,' ')||
' '||rpad(rec.elapsed::text,12,' ')||' '||rpad(rec.p1::text,6,' ')||' '||rpad(rec.relname,20,' ')||' '||rpad(rec.p2::text,10,' ')||' '||rpad(rec.p3::text,12,' ');

return next textstr;
end loop;

return;
end;
   $$ LANGUAGE 'plpgsql' ;

-----------------------------------------------------------------------------------

-- Function: stat_db_rpt(integer, integer)

-- DROP FUNCTION stat_db_rpt(integer, integer);

CREATE OR REPLACE FUNCTION stat_db_rpt(p_bid integer, p_eid integer)
  RETURNS SETOF text AS
$BODY$declare
v_bid       int4;
v_eid       int4;
v_db        text;

v_hitrate	numeric;

textstr text;
rec RECORD;

begin

select current_database() into v_db;

textstr := '  DATA from pg_stat_database';
return next textstr;
textstr := null;
return next textstr;

textstr := rpad('DATABASE',10,' ')||' '||rpad('NUMBACKENDS',12,' ')||' '||rpad('XACT COMMIT',12,' ')||
' '||rpad('XACT ROLLBACK',15,' ')||' '||rpad('BLKS READ',10,' ')||' '||rpad('BLKS HIT',10,' ')||
' '||rpad('BLKS ICACHE HIT',20,' ')||' '||rpad('HIT RATIO',10,' ')||' '||rpad('ICACHE HIT RATIO',20,' ');
return next textstr;
textstr := rpad('-', 126, '-'); -- (126 = 10*4 + 12*2 + 15 + 20*2 + 6 spaces)
return next textstr;
 
v_bid := p_bid;
v_eid := p_eid;

-----------------------------------------------------------------------------
-- NOTE: blks_fetch = blks_read + blks_hit + blks_icache_hit
-- blks_read are actual reads from the disk if the blk is not found in the 
-- shared buffers or the InfiniteCache
-- hence, hitrate is calculated as blks_hit / blks_fetch, 
-- similarly icachehitrate is calculated as blks_icache_hit / blks_fetch
-----------------------------------------------------------------------------
for rec in (
select e.datname,
	(e.numbackends - b.numbackends)  as numbackends,   	 
    (e.xact_commit - b.xact_commit)  as xact_commit,       
	(e.xact_rollback - b.xact_rollback) as xact_rollback,
	(e.blks_read - b.blks_read) as blks_read,
	(e.blks_hit - b.blks_hit) as blks_hit,
	(e.blks_icache_hit - b.blks_icache_hit) as blks_icache_hit,
    round(100 * ( (e.blks_hit - b.blks_hit)/( (e.blks_read - b.blks_read) + (e.blks_hit - b.blks_hit) + (e.blks_icache_hit - b.blks_icache_hit) )::numeric),2) as hitrate,
    round(100 * ( (e.blks_icache_hit - b.blks_icache_hit)/( (e.blks_read - b.blks_read) + (e.blks_hit - b.blks_hit) + (e.blks_icache_hit - b.blks_icache_hit) )::numeric),2) as icachehitrate
from edb$stat_database as e
    ,edb$stat_database as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.datname = v_db 
and e.datname = v_db
and ((e.blks_read - b.blks_read) + (e.blks_hit - b.blks_hit) + (e.blks_icache_hit - b.blks_icache_hit)) <> 0)
	 LOOP

textstr := rpad(rec.datname,10,' ')||' '||rpad(rec.numbackends::text,12,' ')||' '||rpad(rec.xact_commit::text,12,' ')||
' '||rpad(rec.xact_rollback::text,15,' ')||' '||rpad(rec.blks_read::text,10,' ')||' '||rpad(rec.blks_hit::text,10,' ')||
' '||rpad(rec.blks_icache_hit::text,20,' ')||' '||rpad(rec.hitrate::text,10,' ')||' '||rpad(rec.icachehitrate::text,20,' ');
return next textstr;
end loop;

return;
end;
         $BODY$
  LANGUAGE 'plpgsql' VOLATILE
 ;
-------------------------------------------
-- Function: stat_indexes_rpt(integer, integer, integer, text)

-- DROP FUNCTION stat_indexes_rpt(integer, integer, integer, text);

CREATE OR REPLACE FUNCTION stat_indexes_rpt(p_bid integer, p_eid integer, p_topn integer, p_stat text)
  RETURNS SETOF text AS
$BODY$declare
v_bid       int4;
v_eid       int4;
v_topn      int4;
v_stat      text;

textstr text;
rec RECORD;

refcur  refcursor;

begin
textstr := '  DATA from pg_stat_all_indexes';
return next textstr;
textstr := null;
return next textstr;

textstr := rpad('SCHEMA',20,' ')||' '||rpad('RELATION',25,' ')||' '||rpad('INDEX',35,' ')||' '||rpad('IDX SCAN',10,' ')||
' '||rpad('IDX TUP READ',12,' ')||' '||rpad('IDX TUP FETCH',15,' ');
return next textstr;
textstr := '-------------------------------------------------------------------------------------------------------------------------';
return next textstr;
 
v_bid := p_bid;
v_eid := p_eid;
v_topn := p_topn;
v_stat := upper(p_stat);

if v_stat not in ('ALL','USER','SYS') then
	raise exception 'Invalid stat type.';
end if;

if v_stat = 'ALL' then
 open refcur for
select e.schemaname,
	e.relname,
	e.indexrelname,
    e.idx_scan - b.idx_scan  as idx_scan,   	 
    e.idx_tup_read - b.idx_tup_read  as idx_tup_read,       
	e.idx_tup_fetch - b.idx_tup_fetch as idx_tup_fetch
from edb$stat_all_indexes as e
    ,edb$stat_all_indexes as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.indexrelname	  = e.indexrelname
    order by idx_scan desc, indexrelname   
    limit p_topn;
elsif v_stat = 'USER' then
  open refcur for
select e.schemaname,
	e.relname,
	e.indexrelname,
    e.idx_scan - b.idx_scan  as idx_scan,   	 
    e.idx_tup_read - b.idx_tup_read  as idx_tup_read,       
	e.idx_tup_fetch - b.idx_tup_fetch as idx_tup_fetch
from edb$stat_all_indexes as e
    ,edb$stat_all_indexes as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.indexrelname	  = e.indexrelname
and e.schemaname in (select distinct schemaname from pg_stat_user_indexes)
    order by idx_scan desc, indexrelname   
    limit p_topn;
elsif v_stat = 'SYS' then
  open refcur for
select e.schemaname,
	e.relname,
	e.indexrelname,
    e.idx_scan - b.idx_scan  as idx_scan,   	 
    e.idx_tup_read - b.idx_tup_read  as idx_tup_read,       
	e.idx_tup_fetch - b.idx_tup_fetch as idx_tup_fetch
from edb$stat_all_indexes as e
    ,edb$stat_all_indexes as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.indexrelname	  = e.indexrelname
and e.schemaname in (select distinct schemaname from pg_stat_sys_indexes)
    order by idx_scan desc, indexrelname   
    limit p_topn;
end if;

loop
fetch refcur into rec;
exit when not found;

textstr := rpad(rec.schemaname,20,' ')||' '||rpad(rec.relname,25,' ')||' '||rpad(rec.indexrelname,35,' ')||
' '||rpad(rec.idx_scan::text,10,' ')||' '||rpad(rec.idx_tup_read::text,12,' ')||' '||rpad(rec.idx_tup_fetch::text,15,' ');
return next textstr;
end loop;

close refcur;
return;
end;
          $BODY$
  LANGUAGE 'plpgsql' VOLATILE;

-------------------------------------------
-- Function: stat_tables_rpt(integer, integer, integer, text)

-- DROP FUNCTION stat_tables_rpt(integer, integer, integer, text);

CREATE OR REPLACE FUNCTION stat_tables_rpt(p_bid integer, p_eid integer, p_topn integer, p_stat text)
  RETURNS SETOF text AS
$BODY$declare
v_bid       int4;
v_eid       int4;
v_topn      int4;
v_stat      text;

textstr text;
rec RECORD;

refcur  refcursor;

begin
textstr := '  DATA from pg_stat_all_tables ordered by seq scan';
return next textstr;
textstr := null;
return next textstr;

textstr := rpad('SCHEMA',20,' ')||' '||rpad('RELATION',30,' ')||' '||rpad('SEQ SCAN',10,' ')||
' '||rpad('REL TUP READ',12,' ')||' '||rpad('IDX SCAN',10,' ')||' '||rpad('IDX TUP READ',12,' ')||
' '||rpad('INS',6,' ')||' '||rpad('UPD',6,' ')||' '||rpad('DEL',6,' ');
return next textstr;
textstr := '-----------------------------------------------------------------------------------------------------------------------';
return next textstr;
 
v_bid := p_bid;
v_eid := p_eid;
v_topn := p_topn;
v_stat := upper(p_stat);

if v_stat not in ('ALL','USER','SYS') then
	raise exception 'Invalid stat type.';
end if;

if v_stat = 'ALL' then
 open refcur for
select e.schemaname,
	 e.relname,
     e.seq_scan - b.seq_scan   as seq_scan,   	 
     e.seq_tup_read - b.seq_tup_read  as seq_tup_read,       
	 e.idx_scan - b.idx_scan as idx_scan,
	 e.idx_tup_fetch - b.idx_tup_fetch as idx_tup_fetch,
	 e.n_tup_ins - b.n_tup_ins as ins,
	 e.n_tup_upd - b.n_tup_upd as upd,
	 e.n_tup_del - b.n_tup_del as del 
from edb$stat_all_tables as e
    ,edb$stat_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		      = e.relname
and b.idx_scan is not null
and e.idx_scan is not null
and b.idx_tup_fetch is not null
and e.idx_tup_fetch is not null
    order by seq_scan desc, relname   
    limit p_topn;
elsif v_stat = 'USER' then
  open refcur for
select e.schemaname,
	e.relname,
    e.seq_scan - b.seq_scan   as seq_scan,   	 
    e.seq_tup_read - b.seq_tup_read   as seq_tup_read,       
	e.idx_scan - b.idx_scan  as idx_scan,
	e.idx_tup_fetch - b.idx_tup_fetch  as idx_tup_fetch,
	e.n_tup_ins - b.n_tup_ins  as ins,
	e.n_tup_upd - b.n_tup_upd  as upd,
	e.n_tup_del - b.n_tup_del  as del 
from edb$stat_all_tables as e
    ,edb$stat_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.idx_scan is not null
and e.idx_scan is not null
and b.idx_tup_fetch is not null
and e.idx_tup_fetch is not null
and e.schemaname in (select distinct schemaname from pg_stat_user_tables)
    order by seq_scan desc, relname   
    limit p_topn;
elsif v_stat = 'SYS' then
  open refcur for
select e.schemaname,
	e.relname,
    e.seq_scan - b.seq_scan  as seq_scan,   	 
    e.seq_tup_read - b.seq_tup_read  as seq_tup_read,       
	e.idx_scan - b.idx_scan as idx_scan,
	e.idx_tup_fetch - b.idx_tup_fetch as idx_tup_fetch,
	e.n_tup_ins - b.n_tup_ins as ins,
	e.n_tup_upd - b.n_tup_upd as upd,
	e.n_tup_del - b.n_tup_del as del 
from edb$stat_all_tables as e
    ,edb$stat_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.idx_scan is not null
and e.idx_scan is not null
and b.idx_tup_fetch is not null
and e.idx_tup_fetch is not null
and e.schemaname in (select distinct schemaname from pg_stat_sys_tables)
    order by seq_scan desc, relname   
    limit p_topn;
end if;

loop
fetch refcur into rec;
exit when not found;

textstr := rpad(rec.schemaname,20,' ')||' '||rpad(rec.relname,30,' ')||' '||rpad(rec.seq_scan::text,10,' ')||
' '||rpad(rec.seq_tup_read::text,12,' ')||' '||rpad(rec.idx_scan::text,10,' ')||' '||rpad(rec.idx_tup_fetch::text,12,' ')||
' '||rpad(rec.ins::text,6,' ')||' '||rpad(rec.upd::text,6,' ')||' '||rpad(rec.del::text,6,' ');
return next textstr;
end loop;

close refcur;

textstr := null;
return next textstr;
textstr := '  DATA from pg_stat_all_tables ordered by rel tup read';
return next textstr;
textstr := null;
return next textstr;

textstr := rpad('SCHEMA',20,' ')||' '||rpad('RELATION',30,' ')||' '||rpad('SEQ SCAN',10,' ')||
' '||rpad('REL TUP READ',12,' ')||' '||rpad('IDX SCAN',10,' ')||' '||rpad('IDX TUP READ',12,' ')||
' '||rpad('INS',6,' ')||' '||rpad('UPD',6,' ')||' '||rpad('DEL',6,' ');
return next textstr;
textstr := '-----------------------------------------------------------------------------------------------------------------------';
return next textstr;

if v_stat = 'ALL' then
 open refcur for
select e.schemaname,
	 e.relname,
     e.seq_scan - b.seq_scan   as seq_scan,   	 
     e.seq_tup_read - b.seq_tup_read  as seq_tup_read,       
	 e.idx_scan - b.idx_scan as idx_scan,
	 e.idx_tup_fetch - b.idx_tup_fetch as idx_tup_fetch,
	 e.n_tup_ins - b.n_tup_ins as ins,
	 e.n_tup_upd - b.n_tup_upd as upd,
	 e.n_tup_del - b.n_tup_del as del 
from edb$stat_all_tables as e
    ,edb$stat_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		      = e.relname
and b.idx_scan is not null
and e.idx_scan is not null
and b.idx_tup_fetch is not null
and e.idx_tup_fetch is not null
    order by seq_tup_read desc, relname   
    limit p_topn;
elsif v_stat = 'USER' then
  open refcur for
select e.schemaname,
	e.relname,
    e.seq_scan - b.seq_scan   as seq_scan,   	 
    e.seq_tup_read - b.seq_tup_read   as seq_tup_read,       
	e.idx_scan - b.idx_scan  as idx_scan,
	e.idx_tup_fetch - b.idx_tup_fetch  as idx_tup_fetch,
	e.n_tup_ins - b.n_tup_ins  as ins,
	e.n_tup_upd - b.n_tup_upd  as upd,
	e.n_tup_del - b.n_tup_del  as del 
from edb$stat_all_tables as e
    ,edb$stat_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.idx_scan is not null
and e.idx_scan is not null
and b.idx_tup_fetch is not null
and e.idx_tup_fetch is not null
and e.schemaname in (select distinct schemaname from pg_stat_user_tables)
    order by seq_tup_read desc, relname   
    limit p_topn;
elsif v_stat = 'SYS' then
  open refcur for
select e.schemaname,
	e.relname,
    e.seq_scan - b.seq_scan  as seq_scan,   	 
    e.seq_tup_read - b.seq_tup_read  as seq_tup_read,       
	e.idx_scan - b.idx_scan as idx_scan,
	e.idx_tup_fetch - b.idx_tup_fetch as idx_tup_fetch,
	e.n_tup_ins - b.n_tup_ins as ins,
	e.n_tup_upd - b.n_tup_upd as upd,
	e.n_tup_del - b.n_tup_del as del 
from edb$stat_all_tables as e
    ,edb$stat_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.idx_scan is not null
and e.idx_scan is not null
and b.idx_tup_fetch is not null
and e.idx_tup_fetch is not null
and e.schemaname in (select distinct schemaname from pg_stat_sys_tables)
    order by seq_tup_read desc, relname   
    limit p_topn;
end if;

loop
fetch refcur into rec;
exit when not found;

textstr := rpad(rec.schemaname,20,' ')||' '||rpad(rec.relname,30,' ')||' '||rpad(rec.seq_scan::text,10,' ')||
' '||rpad(rec.seq_tup_read::text,12,' ')||' '||rpad(rec.idx_scan::text,10,' ')||' '||rpad(rec.idx_tup_fetch::text,12,' ')||
' '||rpad(rec.ins::text,6,' ')||' '||rpad(rec.upd::text,6,' ')||' '||rpad(rec.del::text,6,' ');
return next textstr;
end loop;

close refcur;

return;
end;
         $BODY$
  LANGUAGE 'plpgsql' VOLATILE;

----------------------------------------------------------------------
-- Function: statio_indexes_rpt(integer, integer, integer, text)

-- DROP FUNCTION statio_indexes_rpt(integer, integer, integer, text);

CREATE OR REPLACE FUNCTION statio_indexes_rpt(p_bid integer, p_eid integer, p_topn integer, p_stat text)
  RETURNS SETOF text AS
$BODY$declare
v_bid       int4;
v_eid       int4;
v_topn      int4;
v_stat      text;

textstr text;
rec RECORD;

refcur  refcursor;

begin
textstr := '  DATA from pg_statio_all_indexes';
return next textstr;
textstr := null;
return next textstr;

textstr := rpad('SCHEMA',20,' ')||' '||rpad('RELATION',25,' ')||' '||rpad('INDEX',35,' ')||' '||rpad('IDX BLKS READ',15,' ')||
' '||rpad('IDX BLKS HIT',15,' ')||' '||rpad('IDX BLKS ICACHE HIT',20,' ');
return next textstr;
textstr := rpad('-',135,'-'); -- (135 = 20*2 + 25 + 35 + 15*2 + 5 spaces)
return next textstr;
 
v_bid := p_bid;
v_eid := p_eid;
v_topn := p_topn;
v_stat := upper(p_stat);

if v_stat not in ('ALL','USER','SYS') then
	raise exception 'Invalid stat type.';
end if;

if v_stat = 'ALL' then
 open refcur for
select e.schemaname,
	e.relname,
	e.indexrelname,
    e.idx_blks_read - b.idx_blks_read  as idx_blks_read,       
	e.idx_blks_hit - b.idx_blks_hit as idx_blks_hit,
	e.idx_blks_icache_hit - b.idx_blks_icache_hit as idx_blks_icache_hit
from edb$statio_all_indexes as e
    ,edb$statio_all_indexes as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.indexrelname	  = e.indexrelname
    order by idx_blks_hit desc, indexrelname   
    limit p_topn;
elsif v_stat = 'USER' then
  open refcur for
select e.schemaname,
	e.relname,
	e.indexrelname,
    e.idx_blks_read - b.idx_blks_read  as idx_blks_read,       
	e.idx_blks_hit - b.idx_blks_hit as idx_blks_hit,
	e.idx_blks_icache_hit - b.idx_blks_icache_hit as idx_blks_icache_hit
from edb$statio_all_indexes as e
    ,edb$statio_all_indexes as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.indexrelname	  = e.indexrelname
and e.schemaname in (select distinct schemaname from pg_statio_user_indexes)
    order by idx_blks_hit desc, indexrelname   
    limit p_topn;
elsif v_stat = 'SYS' then
  open refcur for
select e.schemaname,
	e.relname,
	e.indexrelname,
    e.idx_blks_read - b.idx_blks_read  as idx_blks_read,       
	e.idx_blks_hit - b.idx_blks_hit as idx_blks_hit,
	e.idx_blks_icache_hit - b.idx_blks_icache_hit as idx_blks_icache_hit
from edb$statio_all_indexes as e
    ,edb$statio_all_indexes as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and b.indexrelname	  = e.indexrelname
and e.schemaname in (select distinct schemaname from pg_statio_sys_indexes)
    order by idx_blks_hit desc, indexrelname   
    limit p_topn;
end if;

loop
fetch refcur into rec;
exit when not found;

textstr := rpad(rec.schemaname,20,' ')||' '||rpad(rec.relname,25,' ')||' '||rpad(rec.indexrelname,35,' ')||
' '||rpad(rec.idx_blks_read::text,15,' ')||' '||rpad(rec.idx_blks_hit::text,15,' ')||' '||rpad(rec.idx_blks_icache_hit::text,20,' ');
return next textstr;
end loop;

close refcur;
return;
end;
         $BODY$
  LANGUAGE 'plpgsql' VOLATILE;

-----------------------------------------------------------
-- Function: statio_tables_rpt(integer, integer, integer, text)

-- DROP FUNCTION statio_tables_rpt(integer, integer, integer, text);

CREATE OR REPLACE FUNCTION statio_tables_rpt(p_bid integer, p_eid integer, p_topn integer, p_stat text)
  RETURNS SETOF text AS
$BODY$declare
v_bid       int4;
v_eid       int4;
v_topn      int4;
v_stat      text;

textstr text;
rec RECORD;

refcur  refcursor;

begin
textstr := '  DATA from pg_statio_all_tables';
return next textstr;
textstr := null;
return next textstr;

textstr := rpad('SCHEMA',20,' ')||' '||rpad('RELATION',20,' ')||
' '||rpad('HEAP',8,' ') ||' '||rpad('HEAP',8,' ') ||' '||rpad('HEAP',8,' ')||
' '||rpad('IDX',8,' ')  ||' '||rpad('IDX',8,' ')  ||' '||rpad('IDX',8,' ')||
' '||rpad('TOAST',8,' ')||' '||rpad('TOAST',8,' ')||' '||rpad('TOAST',8,' ')||
' '||rpad('TIDX',8,' ') ||' '||rpad('TIDX',8,' ') ||' '||rpad('TIDX',8,' ');
return next textstr;
textstr := rpad('      ',20,' ')||' '||rpad('        ',20,' ')||
' '||rpad('READ',8,' ')||' '||rpad('HIT',8,' ')||' '||rpad('ICACHE',8,' ')||
' '||rpad('READ',8,' ')||' '||rpad('HIT',8,' ')||' '||rpad('ICACHE',8,' ')||
' '||rpad('READ',8,' ')||' '||rpad('HIT',8,' ')||' '||rpad('ICACHE',8,' ')||
' '||rpad('READ',8,' ')||' '||rpad('HIT',8,' ')||' '||rpad('ICACHE',8,' ');
return next textstr;
textstr := rpad('      ',20,' ')||' '||rpad('        ',20,' ')||
' '||rpad('    ',8,' ')||' '||rpad('   ',8,' ')||' '||rpad('HIT',8,' ')||
' '||rpad('    ',8,' ')||' '||rpad('   ',8,' ')||' '||rpad('HIT',8,' ')||
' '||rpad('    ',8,' ')||' '||rpad('   ',8,' ')||' '||rpad('HIT',8,' ')||
' '||rpad('    ',8,' ')||' '||rpad('   ',8,' ')||' '||rpad('HIT',8,' ');
return next textstr;
textstr := rpad('-', 149, '-'); -- (149 = 20*2 + 8*12 + 13 spaces)
return next textstr;
 
v_bid := p_bid;
v_eid := p_eid;
v_topn := p_topn;
v_stat := upper(p_stat);

if v_stat not in ('ALL','USER','SYS') then
	raise exception 'Invalid stat type.';
end if;

if v_stat = 'ALL' then
 open refcur for
 select e.schemaname,
	e.relname,
    e.heap_blks_read - b.heap_blks_read  as heap_blks_read,   	 
    e.heap_blks_hit - b.heap_blks_hit  as heap_blks_hit,       
    e.heap_blks_icache_hit - b.heap_blks_icache_hit  as heap_blks_icache_hit,
	coalesce(e.idx_blks_read,0) - coalesce(b.idx_blks_read,0) as idx_blks_read,
	coalesce(e.idx_blks_hit,0) - coalesce(b.idx_blks_hit,0) as idx_blks_hit,
        coalesce(e.idx_blks_icache_hit,0) - coalesce(b.idx_blks_icache_hit,0) as idx_blks_icache_hit,
	coalesce(e.toast_blks_read,0) - coalesce(b.toast_blks_read,0) as toast_blks_read,
	coalesce(e.toast_blks_hit,0) - coalesce(b.toast_blks_hit,0) as toast_blks_hit,
        coalesce(e.toast_blks_icache_hit,0) - coalesce(b.toast_blks_icache_hit,0) as toast_blks_icache_hit,
	coalesce(e.tidx_blks_read,0) - coalesce(b.tidx_blks_read,0) as tidx_blks_read,
 	coalesce(e.tidx_blks_hit,0) - coalesce(b.tidx_blks_hit,0) as tidx_blks_hit,
        coalesce(e.tidx_blks_icache_hit,0) - coalesce(b.tidx_blks_icache_hit,0) as tidx_blks_icache_hit
from edb$statio_all_tables as e
    ,edb$statio_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
    order by heap_blks_hit desc, relname   
    limit p_topn;
elsif v_stat = 'USER' then
  open refcur for
select e.schemaname,
	e.relname,
    e.heap_blks_read - b.heap_blks_read  as heap_blks_read,   	 
    e.heap_blks_hit - b.heap_blks_hit  as heap_blks_hit,       
    e.heap_blks_icache_hit - b.heap_blks_icache_hit  as heap_blks_icache_hit,
	coalesce(e.idx_blks_read,0) - coalesce(b.idx_blks_read,0) as idx_blks_read,
	coalesce(e.idx_blks_hit,0) - coalesce(b.idx_blks_hit,0) as idx_blks_hit,
	coalesce(e.idx_blks_icache_hit,0) - coalesce(b.idx_blks_icache_hit,0) as idx_blks_icache_hit,
	coalesce(e.toast_blks_read,0) - coalesce(b.toast_blks_read,0) as toast_blks_read,
	coalesce(e.toast_blks_hit,0) - coalesce(b.toast_blks_hit,0) as toast_blks_hit,
	coalesce(e.toast_blks_icache_hit,0) - coalesce(b.toast_blks_icache_hit,0) as toast_blks_icache_hit,
	coalesce(e.tidx_blks_read,0) - coalesce(b.tidx_blks_read,0) as tidx_blks_read,
 	coalesce(e.tidx_blks_hit,0) - coalesce(b.tidx_blks_hit,0) as tidx_blks_hit,
 	coalesce(e.tidx_blks_icache_hit,0) - coalesce(b.tidx_blks_icache_hit,0) as tidx_blks_icache_hit
from edb$statio_all_tables as e
    ,edb$statio_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and e.schemaname in (select distinct schemaname from pg_statio_user_tables)
    order by heap_blks_read desc, relname   
    limit p_topn;
elsif v_stat = 'SYS' then
  open refcur for
select e.schemaname,
	e.relname,
    e.heap_blks_read - b.heap_blks_read  as heap_blks_read,   	 
    e.heap_blks_hit - b.heap_blks_hit  as heap_blks_hit,       
    e.heap_blks_icache_hit - b.heap_blks_icache_hit  as heap_blks_icache_hit,
	coalesce(e.idx_blks_read,0) - coalesce(b.idx_blks_read,0) as idx_blks_read,
	coalesce(e.idx_blks_hit,0) - coalesce(b.idx_blks_hit,0) as idx_blks_hit,
	coalesce(e.idx_blks_icache_hit,0) - coalesce(b.idx_blks_icache_hit,0) as idx_blks_icache_hit,
	coalesce(e.toast_blks_read,0) - coalesce(b.toast_blks_read,0) as toast_blks_read,
	coalesce(e.toast_blks_hit,0) - coalesce(b.toast_blks_hit,0) as toast_blks_hit,
	coalesce(e.toast_blks_icache_hit,0) - coalesce(b.toast_blks_icache_hit,0) as toast_blks_icache_hit,
	coalesce(e.tidx_blks_read,0) - coalesce(b.tidx_blks_read,0) as tidx_blks_read,
 	coalesce(e.tidx_blks_hit,0) - coalesce(b.tidx_blks_hit,0) as tidx_blks_hit,
 	coalesce(e.tidx_blks_icache_hit,0) - coalesce(b.tidx_blks_icache_hit,0) as tidx_blks_icache_hit
from edb$statio_all_tables as e
    ,edb$statio_all_tables as b
where b.edb_id            = p_bid
and e.edb_id              = p_eid
and b.dbname              = e.dbname
and b.schemaname          = e.schemaname
and b.relname		  = e.relname
and e.schemaname in (select distinct schemaname from pg_statio_sys_tables)
    order by heap_blks_read desc, relname   
    limit p_topn;
end if;

loop
fetch refcur into rec;
exit when not found;

textstr := rpad(rec.schemaname,20,' ')||' '||rpad(rec.relname,20,' ')||
' '||rpad(rec.heap_blks_read::text,8,' ') ||' '||rpad(rec.heap_blks_hit::text,8,' ') ||' '||rpad(rec.heap_blks_icache_hit::text,8,' ')||
' '||rpad(rec.idx_blks_read::text,8,' ')  ||' '||rpad(rec.idx_blks_hit::text,8,' ')  ||' '||rpad(rec.idx_blks_icache_hit::text,8,' ')||
' '||rpad(rec.toast_blks_read::text,8,' ')||' '||rpad(rec.toast_blks_hit::text,8,' ')||' '||rpad(rec.toast_blks_icache_hit::text,8,' ')||
' '||rpad(rec.tidx_blks_read::text,8,' ') ||' '||rpad(rec.tidx_blks_hit::text,8,' ') ||' '||rpad(rec.tidx_blks_icache_hit::text,8,' ');
return next textstr;
end loop;

close refcur;
return;
end;
          $BODY$
  LANGUAGE 'plpgsql' VOLATILE;
-------------------------------------------------------------------
-- New functions added Jan, 2008
-- P. Steinheuser, EnterpriseDB

-- Function: edbreport(bigint, bigint)

-- DROP FUNCTION edbreport(bigint, bigint);

CREATE OR REPLACE FUNCTION edbreport(beginsnap bigint, endsnap bigint)
  RETURNS SETOF text AS
$BODY$ declare
-- v_topn controls how many rows to report by section
-- v_stattype determines where user,sys or all data is used
v_topn		int := 10;
v_stattype	text := 'all';

v_start 	bigint;
v_end   	bigint;
v_db		text;
v_dbsize	text;
v_tspsize	text;
v_tabsize	numeric;
v_version	text;
v_textstr   	text;
v_currdt	text;

snaprec     	record;
spcrec		record;
nsprec		record;
tabrec		record;
parmrec		record;
toptabrec	record;
topidxrec	record;
rec		record;
dmlrec		record;
bufrec		record;

v_nspcnt	numeric := 0;
have_pg_buffercache_ext bool;

begin

v_start := beginsnap;
v_end   := endsnap;
-- get date/time
-- get period from snapids

select current_database() into v_db;
select version() into v_version;
select current_date into v_currdt;

for snaprec in select sn1.edb_id, sn1.snap_tm as start_tm, sn2.edb_id,
    sn2.snap_tm as end_tm
    from edb$snap as sn1, edb$snap as sn2
    where sn1.edb_id = v_start
    and sn2.edb_id = v_end
    order by sn1.edb_id LOOP

v_textstr := '   EnterpriseDB Report for database '||v_db||'        '||v_currdt;
return next v_textstr;
v_textstr := 'Version: '||v_version;
return next v_textstr;
v_textstr := null;
return next v_textstr;

v_textstr := '     Begin snapshot: '||v_start::text||' at '||snaprec.start_tm;
return next v_textstr;
v_textstr := '     End snapshot:   '||v_end::text||' at '||snaprec.end_tm;
return next v_textstr;

end loop;

-- Get database size
v_textstr := null;
return next v_textstr;

select pg_size_pretty(pg_database_size(v_db)) into v_dbsize;
v_textstr := 'Size of database '||v_db||' is '||v_dbsize;
return next v_textstr;

-- Get tablespace info
for spcrec in select spcname,usename from pg_catalog.pg_tablespace as tsp, pg_user as usr
	where tsp.spcowner = usr.usesysid LOOP

select pg_size_pretty(pg_tablespace_size(spcrec.spcname)) into v_tspsize;

v_textstr := '     Tablespace: '||spcrec.spcname||' Size: '||v_tspsize||' Owner: '||spcrec.usename;
return next v_textstr;

end loop;
v_textstr := null;
return next v_textstr;

-- Get schema info
if v_version like 'EnterpriseDB%' then
	for nsprec in select nspname,usename from pg_catalog.pg_namespace as nsp, pg_user as usr
	where usr.usesysid = nsp.nspowner
	and nsp.nspname not in ('sys','dbo','information_schema','pg_toast','pg_catalog')
	and nsp.nspname not like E'pg\\_temp\\_%'
	and nspheadsrc is null LOOP

	v_nspcnt := 0;

	for tabrec in select schemaname,tablename from pg_tables where schemaname = nsprec.nspname LOOP
	select pg_total_relation_size(tabrec.schemaname||'.'||tabrec.tablename) into v_tabsize;

	v_nspcnt := v_nspcnt + v_tabsize;
	end loop;

	v_textstr := 'Schema: '||rpad(nsprec.nspname,30,' ')||' Size: '||rpad(pg_size_pretty(v_nspcnt::bigint),15,' ')||
	' Owner: '||rpad(nsprec.usename,20,' ');
	return next v_textstr;

	end loop;
else
	for nsprec in select nspname,usename from pg_catalog.pg_namespace as nsp, pg_user as usr
	where usr.usesysid = nsp.nspowner
	and nsp.nspname not in ('sys','dbo','information_schema','pg_toast','pg_catalog')
	and nsp.nspname not like E'pg\\_temp\\_%'
	 LOOP

	v_nspcnt := 0;

	for tabrec in select schemaname,tablename from pg_tables where schemaname = nsprec.nspname LOOP
	select pg_total_relation_size(tabrec.schemaname||'.'||tabrec.tablename) into v_tabsize;

	v_nspcnt := v_nspcnt + v_tabsize;
	end loop;

	v_textstr := 'Schema: '||rpad(nsprec.nspname,30,' ')||' Size: '||rpad(pg_size_pretty(v_nspcnt::bigint),15,' ')||
	' Owner: '||rpad(nsprec.usename,20,' ');
	return next v_textstr;

	end loop;
end if;

v_textstr := null;
return next v_textstr;

-- Get Top 10 largest tables
v_textstr := '               Top 10 Relations by pages';
return next v_textstr;
v_textstr := null;
return next v_textstr;

v_textstr := rpad('TABLE',45,' ')||rpad('RELPAGES',10,' ');
return next v_textstr;
v_textstr := '----------------------------------------------------------------------------------';
return next v_textstr;

for toptabrec in SELECT relname, relpages FROM pg_class 
where relkind = 'r' 
ORDER BY relpages DESC
limit 10 loop

v_textstr := rpad(toptabrec.relname,45,' ')||' '||rpad(toptabrec.relpages::text,10,' ');
return next v_textstr;
end loop;

v_textstr := null;
return next v_textstr;

-- Get Top 10 largest indexes
v_textstr := '               Top 10 Indexes by pages';
return next v_textstr;
v_textstr := null;
return next v_textstr;

v_textstr := rpad('INDEX',45,' ')||rpad('RELPAGES',10,' ');
return next v_textstr;
v_textstr := '----------------------------------------------------------------------------------';
return next v_textstr;

for topidxrec in SELECT relname, relpages FROM pg_class 
where relkind = 'i' 
ORDER BY relpages DESC
limit 10 loop

v_textstr := rpad(topidxrec.relname,45,' ')||' '||rpad(topidxrec.relpages::text,10,' ');
return next v_textstr;

end loop;
v_textstr := null;
return next v_textstr;

-- Top 10 relations by update,delete,insert activity
v_textstr := '               Top 10 Relations by DML';
return next v_textstr;
v_textstr := null;
return next v_textstr;

v_textstr := rpad('SCHEMA',15,' ')||' '||rpad('RELATION',45,' ')||rpad('UPDATES',10,' ')||
rpad('DELETES',10,' ')||rpad('INSERTS',10,' ');
return next v_textstr;
v_textstr := '--------------------------------------------------------------------------------------------------';
return next v_textstr;

for dmlrec in
select schemaname as SCHEMA,
		relname as RELATION,
		n_tup_upd as UPDATES,
		n_tup_del as DELETES,
		n_tup_ins as INSERTS
from pg_stat_user_tables
order by n_tup_upd desc,n_tup_del desc, n_tup_ins desc,schemaname,relname
limit 10 LOOP

v_textstr := rpad(dmlrec.schema,15,' ')||' '||rpad(dmlrec.relation,45,' ')||rpad(dmlrec.updates::text,10,' ')||
rpad(dmlrec.deletes::text,10,' ')||rpad(dmlrec.inserts::text,10,' ');
return next v_textstr;

end loop;

v_textstr := null;
return next v_textstr;

-- Call stat functions
for rec in (select * from  stat_db_rpt(v_start::int4,v_end::int4) z )
	loop
		v_textstr := rec.z;
  		return next v_textstr;
	end loop;
v_textstr := null;
return next v_textstr;

-- If pg_buffercache extension is installed, fetch buffer usage information
select count(*) > 0 from pg_extension where extname = 'pg_buffercache' into have_pg_buffercache_ext;

if have_pg_buffercache_ext then
	-- Top 10 Relations in buffer cache
	v_textstr := '  DATA from pg_buffercache';
	return next v_textstr;
	v_textstr := null;
	return next v_textstr;

	v_textstr := rpad('RELATION',35,' ')||' '||rpad('BUFFERS',10,' ');
	return next v_textstr;
	v_textstr := '-----------------------------------------------------------------------------';
	return next v_textstr;
	for bufrec in
		SELECT c.relname, count(*) AS buffers
		FROM pg_class c
		INNER JOIN pg_buffercache b ON b.relfilenode = c.relfilenode
		INNER JOIN pg_database d ON (b.reldatabase = d.oid AND d.datname = current_database())
		GROUP BY c.relname
		ORDER BY 2 DESC LIMIT 10
	loop
		v_textstr := rpad(bufrec.relname,35,' ')||' '||rpad(bufrec.buffers::text,10,' ');
		return next v_textstr;

	end loop;
	v_textstr := null;
	return next v_textstr;
else
	v_textstr := '  DATA from pg_buffercache not included because pg_buffercache is not installed';
	return next v_textstr;
	v_textstr := null;
	return next v_textstr;
end if;

for rec in (select * from  stat_tables_rpt(v_start::int4,v_end::int4,v_topn,v_stattype) z )
	loop
  		v_textstr := rec.z;
  		return next v_textstr;
	end loop;
v_textstr := null;
return next v_textstr;

for rec in (select * from  statio_tables_rpt(v_start::int4,v_end::int4,v_topn,v_stattype) z )
	loop
  		v_textstr := rec.z;
  		return next v_textstr;
	end loop;
v_textstr := null;
return next v_textstr;
 
for rec in (select * from  stat_indexes_rpt(v_start::int4,v_end::int4,v_topn,v_stattype) z )
	loop
  		v_textstr := rec.z;
  		return next v_textstr;
	end loop;
v_textstr := null;
return next v_textstr;

for rec in (select * from  statio_indexes_rpt(v_start::int4,v_end::int4,v_topn,v_stattype) z )
	loop
  		v_textstr := rec.z;
  		return next v_textstr;
	end loop;
v_textstr := null;
return next v_textstr;

-- Incluce DRITA system data, if Enterprisedb
if v_version like 'EnterpriseDB%' then
    v_textstr := '   System Wait Information';
    return next v_textstr;
	v_textstr := null;
	return next v_textstr;

	for rec in (select * from  sys_rpt(v_start::int4,v_end::int4,v_topn) z )
	loop
  		v_textstr := rec.z;
  		return next v_textstr;
	end loop;
	v_textstr := null;
	return next v_textstr;
end if;

-- Get database parameters
v_textstr := '   Database Parameters from postgresql.conf ';
return next v_textstr;
v_textstr := null;
return next v_textstr;
v_textstr := 'PARAMETER                         SETTING                                  CONTEXT     MINVAL       MAXVAL       ';
return next v_textstr;
v_textstr := '-------------------------------------------------------------------------------------------------------------------------';
return next v_textstr;

for parmrec in select name as parm,
	-- coalesce(unit,'') as unit,
    setting as setting,
-- category,
	coalesce(context,'') as context,
--	coalesce(vartype,'') as vartype,	
	coalesce(min_val,'') as minval,
	coalesce(max_val,'') as maxval
 from pg_settings 
 order by name LOOP

v_textstr := rpad(parmrec.parm,35,' ') -- ||rpad(parmrec.unit,5,' ')
||rpad(parmrec.setting,40,' ')||' '||rpad(parmrec.context,12,' ')||
rpad(parmrec.minval::text,13,' ')||rpad(parmrec.maxval::text,12,' ');
return next v_textstr;

end loop;
end;
               $BODY$
  LANGUAGE 'plpgsql' VOLATILE;


------------ End of changes ---------------------------------------
