-- ###############################################################################
-- #
-- # Postgres Enterprise Manager - SQL Profiler Plugin
-- #
-- # Copyright 2011 - 2013, EnterpriseDB Corporation
-- #
-- # This code is licenced under the EnterpriseDB Limited Use Licence for use
-- # by customers with an active PostgreSQL, Postgres Plus Advanced Server or
-- # or Postgres Enterprise Manager subscription.
-- #
-- # Redistribution is strictly prohibited.
-- #
-- ###############################################################################

--SQL for Query Profiler

-- Version Info
CREATE OR REPLACE FUNCTION public.sp_profiler_version() RETURNS float8 AS
'$libdir/sql-profiler', 'sp_version'
LANGUAGE C;

COMMENT ON FUNCTION public.sp_profiler_version()  IS '
- This represents the current version of the SQL Profiler.
- This is used by PEM profiler UI.
';

-- Query Profiler function
CREATE OR REPLACE FUNCTION public.sp_activate(comments character varying(100), users oidvector, databases oidvector, max_log_limit int4, log_min_duration int4, time_limit interval = NULL) RETURNS text AS
'$libdir/sql-profiler', 'sp_activate'
LANGUAGE C;

COMMENT ON FUNCTION public.sp_activate(character varying(100), oidvector, oidvector, int4, int4, interval)  IS '
Description:
- Start a new trace for the given user(s) and database(s).
Returns
- trace_id on success, which can be used to stop this trace.

Parameters:
  comments         - Any text to describe the purpose for the trace.
  users            - Vector of user OID for whom queries needs to be logged.
                   - Empty oidvector suggests logging the queries for all the user.
  databases        - Vector of datbase OID for which sql needs to be logged.
                   - Empty oidvector suggests logging the queries for all the database.
  max_log_limit    - Total size of trace should not go beyond this limit.
                   - Unit is in MB.
                   - -1 suggests no limit.
  log_min_duration - Logs only statements running at least this number of milliseconds.
  time_limit       - Run the trace for the given number of time interval.
                   - NULL suggests no limit.
                     Interval must be a positive value.
';

CREATE OR REPLACE FUNCTION public.sp_deactivate(trace_id text) RETURNS bool AS
'$libdir/sql-profiler', 'sp_deactivate'
LANGUAGE C;

COMMENT ON FUNCTION public.sp_deactivate(text)  IS '
Description:
- Stops the trace for the given trace_id
Returns
- true on success, otherwise the error.

Parameters:
  trace_id - the return value of sp_activate function.
';

CREATE OR REPLACE FUNCTION public.sp_active_traces(
	OUT trace_id         text,
	OUT owner            oid,
	OUT start_time       timestamptz,
	OUT max_log_size     int4,
	OUT curr_log_size    int4,
	OUT log_min_duration int4,
	OUT running_queries  int4,
	OUT users            oidvector,
	OUT databases        oidvector) RETURNS SETOF RECORD AS
'$libdir/sql-profiler', 'sp_active_traces'
LANGUAGE C;

COMMENT ON FUNCTION public.sp_active_traces()  IS '
Description:
For superusers:
- List down all the active trace(s).
For non-superusers:
- List down the active trace(s) initiated by them.

Returns:
- ID of the trace (trace_id)
- Owner of the trace
- Starting time
- Maximum trace size in MB
- Currently logged trace size in MB
- Limit for the logging the queries which run at least this number of milliseconds. 
- Owner of the trace
- List of User OID(s)
- List of database OID(s)
';

CREATE OR REPLACE FUNCTION public.sp_traces_list(
	OUT trace_id    text,
	OUT comments    text,
	OUT start_time  timestamptz,
	OUT owner       oid,
	OUT users       oidvector,
	OUT databases   oidvector,
	OUT max_size    int4,
	OUT end_time    timestamptz,
	OUT finish_time timestamptz,
	OUT status      int4,
	OUT version     float8) RETURNS SETOF RECORD AS
'$libdir/sql-profiler', 'sp_traces_list'
LANGUAGE C;

COMMENT ON FUNCTION public.sp_traces_list()  IS '
Description:
For superusers:
- List down all the trace(s)
For non-superusers:
- List down the trace(s) initiated by them.

Returns:
- ID of the trace (trace_id)
- Description that set by user at the time of activation
- Start time
- Owner of the trace
- List of User OID(s)
- List of Database OID(s)
- Limit of the maximum trace size in MB
- End time denotes the time at which sp_deactivate() was called or scheduled to stop.
- Finish time denotes the time at which the trace actually stopped
- Status. 0: Unknown status (Server should have restared)
          1. Active
          2. Stopped
          3. Max size exceeded
          4. Time limit exceeded
';

CREATE OR REPLACE FUNCTION public.sp_load_trace (trace_id text, refresh bool) RETURNS bool AS
'$libdir/sql-profiler', 'sp_load_trace'
LANGUAGE C;

COMMENT ON FUNCTION public.sp_load_trace(text, bool)  IS '
Description:
- Load all the trace files of particular trace_id in the database in temporary table "_sp_tmp_tbl_sql_profiler"

Returns:
- Returns true on successful loading the trace.

Parameters:
- Trace_id for the particular trace
- For refresh:
  True - All the temp table data will be removed from the temp table for that particular trace-id,
         and then load the data from the scratch.
  False - The trace files will be read from the position, where it was successfully loaded last time.
';

CREATE OR REPLACE FUNCTION public.sp_cleanup (trace_id text, remove_active BOOl) RETURNS bool AS
'$libdir/sql-profiler', 'sp_cleanup'
LANGUAGE C;

COMMENT ON FUNCTION public.sp_cleanup(text, bool) IS '
Description:
- Remove the trace permanently.

Returns:
- Returns true on successful removal of the trace.

Parameters:
- Trace_id for the particular trace, which needs to be removed.
- Remove active flag:
  True  - suggests you are not accedently removing the active trace.
  False - results in an error if the given trace is still active.
';

