
CREATE SCHEMA sqlprotect;
GRANT USAGE ON SCHEMA sqlprotect TO public;
SET SEARCH_PATH=sqlprotect;

CREATE TABLE edb_sql_protect
(
  dbid oid NOT NULL,
  roleid oid NOT NULL,
  protect_relations boolean,
  allow_utility_cmds boolean,
  allow_tautology boolean,
  allow_empty_dml boolean,
  CONSTRAINT edb_sql_protect_pkey PRIMARY KEY (roleid)
);

GRANT select ON edb_sql_protect TO public;

CREATE TABLE edb_sql_protect_rel
(
  dbid oid NOT NULL,
  roleid oid NOT NULL,
  relid oid NOT NULL,
  CONSTRAINT edb_sql_protect_rel_pkey PRIMARY KEY (roleid, relid)
);

GRANT select ON edb_sql_protect_rel TO public;

/* ------------------------------------------------------------------------- */
/* Function to add a new user to protect. The default is maximum protection  */
/* ------------------------------------------------------------------------- */
CREATE OR REPLACE FUNCTION protect_role (text)
RETURNS void
AS '$libdir/sqlprotect'
LANGUAGE C;

CREATE OR REPLACE FUNCTION unprotect_role (text)
RETURNS void
AS '$libdir/sqlprotect', 'unprotect_role_by_name'
LANGUAGE C;

CREATE OR REPLACE FUNCTION unprotect_role (Oid)
RETURNS void
AS '$libdir/sqlprotect', 'unprotect_role_by_oid'
LANGUAGE C;

CREATE OR REPLACE FUNCTION unprotect_rel(
	username text,
	schema text,
	relname text
)
RETURNS void
AS '$libdir/sqlprotect', 'unprotect_rel_by_name'
LANGUAGE C;

CREATE OR REPLACE FUNCTION unprotect_rel(
	username text,
	relname text
)
RETURNS void
AS '$libdir/sqlprotect', 'unprotect_rel_by_name'
LANGUAGE C;

CREATE OR REPLACE FUNCTION unprotect_rel(
	userid Oid,
	relid Oid
)
RETURNS void
AS '$libdir/sqlprotect', 'unprotect_rel_by_oid'
LANGUAGE C;

CREATE FUNCTION edb_sql_protect_stats(
    OUT roleid oid,
    OUT superusers int8,
    OUT relations int8,
    OUT commands int8,
    OUT tautology int8,
    OUT dml int8
)
RETURNS SETOF record
AS '$libdir/sqlprotect'
LANGUAGE C;

CREATE FUNCTION edb_sql_protect_queries(
    OUT roleid oid,
    OUT ip_address inet,
    OUT port int4,
    OUT machine_name text,
    OUT date_time timestamp,
    OUT query text
)
RETURNS SETOF record
AS '$libdir/sqlprotect'
LANGUAGE C;

CREATE FUNCTION drop_stats(username text)
RETURNS void
AS '$libdir/sqlprotect','drop_stats_by_name'
LANGUAGE C;

CREATE FUNCTION drop_stats(userid Oid)
RETURNS void
AS '$libdir/sqlprotect','drop_stats_by_oid'
LANGUAGE C;

CREATE FUNCTION drop_queries(username text)
RETURNS int4
AS '$libdir/sqlprotect','drop_queries_by_name'
LANGUAGE C;

CREATE FUNCTION drop_queries(userid Oid)
RETURNS int4
AS '$libdir/sqlprotect','drop_queries_by_oid'
LANGUAGE C;

-- Register a view on the function for ease of use.
CREATE VIEW edb_sql_protect_stats AS
  SELECT 
    pg_catalog.pg_get_userbyid(roleid) username,
    superusers,
    relations,
    commands,
    tautology,
    dml 
    FROM edb_sql_protect_stats();

GRANT SELECT ON edb_sql_protect_stats TO PUBLIC;

CREATE VIEW edb_sql_protect_queries AS
  SELECT 
    pg_catalog.pg_get_userbyid(roleid) username,
    ip_address,
    port,
    machine_name,
    date_time,
    query
    FROM edb_sql_protect_queries() order by date_time;

GRANT SELECT ON edb_sql_protect_queries TO PUBLIC;

-- Register a view to list protected users for ease of use.
CREATE OR REPLACE VIEW list_protected_users AS
  SELECT
      d.datname as dbname,
      pg_catalog.pg_get_userbyid(roleid) as username,
      p.protect_relations,
      p.allow_utility_cmds,
      p.allow_tautology,
      p.allow_empty_dml
  FROM  pg_database d, sqlprotect.edb_sql_protect p
  WHERE p.dbid = d.oid;


GRANT SELECT ON list_protected_users TO PUBLIC;

-- Register a view to list protected relations and their info.
CREATE OR REPLACE VIEW list_protected_rels AS
  SELECT
    d.datname AS "Database",
    u.usename AS "Protected User",
    n.nspname AS "Schema",
    c.relname AS "Name",
    CASE c.relkind
      WHEN 'i' THEN 'Index'
      WHEN 'r' THEN 'Table'
      WHEN 'v' THEN 'View'
      WHEN 'S' THEN 'Sequence'
      WHEN 'c' THEN 'Composite'
      WHEN 'p' THEN 'Composite package type'
      WHEN 's' THEN 'Special relation'
      WHEN 't' THEN 'TOAST table'
    END AS "Type",
    pg_catalog.pg_get_userbyid(c.relowner) AS "Owner"
  FROM pg_namespace n, pg_class c, pg_user u,
       sqlprotect.edb_sql_protect_rel r, pg_database d
  WHERE
    d.oid = r.dbid AND
    c.oid = r.relid AND
    u.usesysid = r.roleid AND
    n.oid = c.relnamespace;

GRANT SELECT ON list_protected_rels TO PUBLIC;

CREATE OR REPLACE FUNCTION export_sqlprotect (text)
		RETURNS void
		AS '$libdir/sqlprotect'
		LANGUAGE C;

CREATE OR REPLACE FUNCTION import_sqlprotect (text)
		RETURNS void
		AS '$libdir/sqlprotect'
		LANGUAGE C;

SET SEARCH_PATH TO DEFAULT;
