SET search_path = sys;

CREATE OR REPLACE PACKAGE utl_smtp IS

  ------------------------------------------------------------------------------
  -- UTL_SMTP package type and variable definitions
  ------------------------------------------------------------------------------

  TYPE connection IS RECORD
  (
    host            VARCHAR2(255),  -- Name or ip address of the remote host running SMTP server
                                    -- [src : RFC 2181 section 11 states that 
                                    --  "A full domain name is limited to 255 octets (including the separators)"]
    port            INTEGER,        -- 25 usually, unless some one is using some other special port number
    tx_timeout      INTEGER,        -- The read and write functions wait this many seconds before giving up
    private_sd      INTEGER,
    private_newline VARCHAR2(2),    -- This is a private member, user should not use it directly.
    private_state   INTEGER         -- This is a private member for maintaining SMTP client state
                                    -- user should not use it directly.
  );

  ------------------------------------------------------------------------------
  -- RFC 821 section 4.5.3 states that 
  -- "The maximum total length of a reply line including the reply code and the <CR><LF> is 512 characters."
  -- We have a 3 digit code followed by a space and the a 508 character message
  ------------------------------------------------------------------------------

  TYPE reply IS RECORD
  (
    code    INTEGER,                -- 3 digit reply code received from the SMTP server
    text    VARCHAR2(508)           -- Message text received from the SMTP server
  );

  TYPE replies IS TABLE OF reply INDEX BY BINARY_INTEGER;

  FUNCTION  OPEN_CONNECTION(host IN VARCHAR2(255), port IN INTEGER DEFAULT 25, tx_timeout IN INTEGER DEFAULT NULL) RETURN connection;
  FUNCTION  COMMAND(c IN OUT connection, cmd IN VARCHAR2, arg IN VARCHAR2 DEFAULT NULL) RETURN reply;
  FUNCTION  COMMAND_REPLIES(c IN OUT connection, cmd IN VARCHAR2, arg IN VARCHAR2 DEFAULT NULL) RETURN replies;

  FUNCTION  HELP(c IN OUT connection, command IN VARCHAR2 DEFAULT NULL) RETURN replies;
  FUNCTION  NOOP(c IN OUT connection) RETURN reply;
  FUNCTION  QUIT(c IN OUT connection) RETURN reply;
  FUNCTION  VRFY(c IN OUT connection, recipient IN VARCHAR2) RETURN reply;

--   --
--   -- TODO: Uncomment the following functions once db server allows function having an out parameter.
--   --
--   FUNCTION  OPEN_CONNECTION(host IN VARCHAR2(255), port IN INTEGER, c OUT connection, tx_timeout IN INTEGER DEFAULT NULL) RETURN reply; 
--
--   FUNCTION  EHLO(c IN OUT connection, domain IN VARCHAR2) RETURN replies;
--   FUNCTION  HELO(c IN OUT connection, domain IN VARCHAR2) RETURN reply;
--   FUNCTION  MAIL(c IN OUT connection, sender IN VARCHAR2, parameters IN VARCHAR2 DEFAULT NULL) RETURN reply;
--   FUNCTION  RCPT(c IN OUT connection, recipient IN VARCHAR2, parameters IN VARCHAR2 DEFAULT NULL) RETURN reply;
--
--   FUNCTION  RSET(c IN OUT connection) RETURN reply;
--
--   FUNCTION  OPEN_DATA(c IN OUT connection) RETURN reply;
--   FUNCTION  CLOSE_DATA(c IN OUT connection) RETURN reply;
--   FUNCTION  DATA(c IN OUT connection, body IN VARCHAR2) RETURN reply;
--   FUNCTION  QUIT(c IN OUT connection) RETURN reply;

  PROCEDURE COMMAND(c IN OUT connection, cmd IN VARCHAR2, arg IN VARCHAR2 DEFAULT NULL);

  PROCEDURE DATA(c IN OUT connection, body IN VARCHAR2);
  PROCEDURE EHLO(c IN OUT connection, domain IN VARCHAR2);
  PROCEDURE HELO(c IN OUT connection, domain IN VARCHAR2);
  PROCEDURE MAIL(c IN OUT connection, sender IN VARCHAR2, parameters IN VARCHAR2 DEFAULT NULL);

  PROCEDURE NOOP(c IN OUT connection);
  PROCEDURE RCPT(c IN OUT connection, recipient IN VARCHAR2, parameters IN VARCHAR2 DEFAULT NULL);
  PROCEDURE RSET(c IN OUT connection);
  PROCEDURE QUIT(c IN OUT connection);

  PROCEDURE OPEN_DATA(c IN OUT connection);
  PROCEDURE WRITE_DATA(c IN OUT connection, data IN VARCHAR2);
  PROCEDURE CLOSE_DATA(c IN OUT connection);

END;

